/******************************************************************************
	collectionlistview.cpp - A custom list view for displaying ROMs.
	
	Copyright (c) 2006 by Brian S. Stephan
	bssteph@incorporeal.org
	
	Copyright (c) 2006 by Michaël Larouche <michael.larouche@kdemail.net>
	
	*************************************************************************
	*                                                                       *
	* This program is free software; you can redistribute it and/or modify  *
	* it under the terms of the GNU General Public License as published by  *
	* the Free Software Foundation; either version 2 of the License, or     *
	* (at your option) any later version.                                   *
	*                                                                       *
	*************************************************************************
	
 *****************************************************************************/

#include <qtimer.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>

#include "collectionlistview.h"
#include <romlistviewitem.h>

class CollectionListView::Private
{
public:
	Private()
	{}
};

CollectionListView::CollectionListView(QWidget* parent, const char *name) :
	KListView(parent, name), d(new Private)
{
	setAllColumnsShowFocus(true);
	setSelectionModeExt( KListView::FileManager );
	addColumn(i18n("Name"));
	addColumn(i18n("Developer"));
	addColumn(i18n("Country"));
	addColumn(i18n("System"));

	// capture our click events and process the rom information
	connect(this, SIGNAL(clicked(QListViewItem*)), this, SLOT(slotClicked(QListViewItem*)));
	connect(this, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotDoubleClicked(QListViewItem*)));

	QTimer::singleShot(0, this, SLOT(slotRestoreLayout()));
}

CollectionListView::~CollectionListView()
{
	kdDebug() << k_funcinfo << "Saving listview layout." << endl;
	// Save listview's layout settings.
	KConfig *layoutConfig= KGlobal::config();
	saveLayout(layoutConfig, QString::fromUtf8("CollectionListView"));

	delete d;
}

void CollectionListView::clear()
{
	KListView::clear();

	slotRestoreLayout();
}

void CollectionListView::contextMenuEvent(QContextMenuEvent* event)
{
	event->accept();
	if( selectedItems().count() )
	{
		int count = selectedItems().count();

		KPopupMenu *popup = new KPopupMenu(this, "rom_context_menu");

		popup->insertTitle(i18n("ROM Management"));
		
		popup->insertItem(SmallIcon("edittrash"), i18n("Trash file", "Trash %n files", count), this, SLOT(slotRomManipulationSelected(int)), 0, TrashRomMenu);
		popup->insertItem( SmallIcon("editdelete"), i18n("Delete file", "Delete %n files", count), this, SLOT(slotRomManipulationSelected(int)), 0, DeleteRomMenu);
		popup->insertSeparator();
		popup->insertItem(SmallIcon("edit"), i18n("&View/Edit Information", "Edit information for %n files", count), this, SLOT(slotRomManipulationSelected(int)), 0, ViewRomMenu);

		// draw the popup
		popup->popup(event->globalPos());
	}
}

void CollectionListView::slotRomManipulationSelected(int id)
{
	// rom was selected for view/edit
	QPtrList<QListViewItem> roms = selectedItems();
		
	KURL::List romList;
		
	// get the rom names that are selected in the gui
	QPtrListIterator<QListViewItem> itr(roms);
	QListViewItem* rom;
	while ((rom = itr.current()) != 0) 
	{
		// get the class we want
		RomListViewItem *romItem = dynamic_cast<RomListViewItem*>(rom);
		romList.append( KURL::fromPathOrURL(romItem->romUrl()) );

		++itr;
	}
	
	switch(id)
	{
		case ViewRomMenu:
			emit editRoms(romList);
			break;
		case DeleteRomMenu:
			emit deleteRoms(romList);
			break;
		case TrashRomMenu:
			emit trashRoms(romList);
			break;
		default:
			kdDebug() << k_funcinfo << "Unhandled popup menu item: " << id << endl;
			break;
	}
}

void CollectionListView::slotClicked(QListViewItem *item)
{
	RomListViewItem *romItem = dynamic_cast<RomListViewItem*>(item);
	
	emit romClicked(romItem->romUrl());
}

void CollectionListView::slotDoubleClicked(QListViewItem *item)
{
	RomListViewItem *romItem = dynamic_cast<RomListViewItem*>(item);
	
	emit romDoubleClicked(romItem->romUrl());
}

void CollectionListView::slotRestoreLayout()
{
	// Restore ListView layout settings.
	kdDebug() << k_funcinfo << "Reading listview layout." << endl;
	KConfig *layoutConfig= KGlobal::config();
	restoreLayout(layoutConfig, QString::fromUtf8("CollectionListView"));
}

#include "collectionlistview.moc"
