/******************************************************************************
	collectionlistview.h - A custom list view for displaying ROMs.
	
	Copyright (c) 2006 by Brian S. Stephan
	bssteph@incorporeal.org
	
	Copyright (c) 2006 by Michaël Larouche <michael.larouche@kdemail.net>
	
	*************************************************************************
	*                                                                       *
	* This program is free software; you can redistribute it and/or modify  *
	* it under the terms of the GNU General Public License as published by  *
	* the Free Software Foundation; either version 2 of the License, or     *
	* (at your option) any later version.                                   *
	*                                                                       *
	*************************************************************************
	
 *****************************************************************************/

#ifndef COLLECTIONLISTVIEW_H
#define COLLECTIONLISTVIEW_H

#include <klistview.h>
#include <kpopupmenu.h>
#include <kurl.h>

/**
 * @brief List view for the ROMs with context menu
 *
 * @author Brian S. Stephan <bssteph@incorporeal.org>
 * @author Michaël Larouche <michael.larouche@kdemail.net>
*/
class CollectionListView : public KListView
{
	Q_OBJECT
public:
	/**
	 * Define the id for the popup menu (IMO, Qt4 is better with this ;))
	 */
	enum MenuItem { ViewRomMenu = 1, DeleteRomMenu, TrashRomMenu };
	/**
	 * Create a new list for ROM display.
	 * @param parent this widget's parent
	 */
	CollectionListView(QWidget *parent = 0, const char *name = 0);
	
	virtual ~CollectionListView();

signals:
	/**
	 * ROMs were selected for editing.
	 * @param romList list of selected ROMs
	 */
	void editRoms(const KURL::List &romList);
	/**
	 * ROMs were selected for deletion.
	 * @param romList list of selected ROMs
	 */
	void deleteRoms(const KURL::List &romList);
	/**
	 * ROMs were selected from trashing.
	 * @param romList list of selected ROMs
	 */
	void trashRoms(const KURL::List &romList);
	
	/**
	 * A rom was clicked.
	 * @param romUrl URL of the ROM clicked
	 */
	void romClicked(const QString &romUrl);
	
	/**
	 * A rom was double clicked.
	 * @param romUrl URL of the ROM double clicked
	 */
	void romDoubleClicked(const QString &romUrl);

public slots:
	/**
	 * Restore column layout after clearing the list view.
	 */
	virtual void clear();

private:
	/**
	 * Overriding from KListView, this handles context menu events.
	 * @param event details of the event
	 */
	void contextMenuEvent(QContextMenuEvent* event);

private slots:
	/**
	 * Manage ROM manipulation popup menu items.
	 * @param id Menu called identified by its ID.
	 */
	void slotRomManipulationSelected(int id);

	/**
	 * Translate the item click to a rom URL
	 * @param item the entry clicked
	 */
	void slotClicked(QListViewItem *item);
	
	/**
	 * Translate the item double click to a rom URL
	 * @param item the entry double clicked
	 */
	void slotDoubleClicked(QListViewItem *item);

	/**
	 * Restore list view layout at load.
	 * Called from a singleshot Timer.
	 */
	void slotRestoreLayout();

private:
	class Private;
	Private *d;
};

#endif
