/******************************************************************************
	metadataedit.cpp - Provide a means to view/edit ROM metadata info.
	
	Copyright (c) 2006 by Brian S. Stephan
	bssteph@incorporeal.org
	
	Copyright (c) 2006 by Michaël Larouche <michael.larouche@kdemail.net>
	
	Based on code in kdemultimedia/juk/tageditor.[h/cpp]:
	(c) 2002 - 2004 by Scott Wheeler
	
	*************************************************************************
	*                                                                       *
	* This program is free software; you can redistribute it and/or modify  *
	* it under the terms of the GNU General Public License as published by  *
	* the Free Software Foundation; either version 2 of the License, or     *
	* (at your option) any later version.                                   *
	*                                                                       *
	*************************************************************************
	
 *****************************************************************************/

#include <qlayout.h>
#include <qlabel.h>
#include <qstringlist.h>

#include <kiconloader.h>
#include <klineedit.h>
#include <knuminput.h>
#include <kdebug.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kurlrequester.h>
#include <kcombobox.h>
#include <kcompletion.h>
#include <kimageio.h>

#include "metadataedit.h"
#include "rommetainformation.h"
#include "collectionaccess.h"

class MetadataEdit::Private
{
public:
	Private() : changed(false), nameLineEdit(0), systemLineEdit(0), developerComboBox(0),
		hashLineEdit(0), yearSpinBox(0), countryComboBox(0), genreComboBox(0), pathLineEdit(NULL)
	{}
	
	QString currentRomUrl;
	KURL::List romList;
	bool changed;
	
	// GUI elements
	KLineEdit* nameLineEdit;
	KLineEdit* systemLineEdit;
	KComboBox* developerComboBox;
	KLineEdit* hashLineEdit;
	KIntSpinBox* yearSpinBox;
	KComboBox* countryComboBox;
	KComboBox* genreComboBox;
	KLineEdit* pathLineEdit;
	KURLRequester* titleScreenURLRequester;
	KURLRequester* screenShotURLRequester;
	
	// StringList cache
	QStringList developerList;
	QStringList genreList;
	QStringList countryList;

	// Current rom information viewed/edited.
	Kamefu::RomMetaInformation currentRomInformation;
};

MetadataEdit::MetadataEdit(const QString &romUrl, QWidget *parent, const char *name) :
	QWidget(parent, name), d(new Private)
{
	loadSingleRom(romUrl);
	init();
}

MetadataEdit::MetadataEdit(const KURL::List &romList, QWidget *parent, const char *name) :
	QWidget(parent, name), d(new Private)
{
	loadRoms(romList);
	init();
}

MetadataEdit::~MetadataEdit()
{
	delete d;
}

void MetadataEdit::init()
{
	setupLayout();
	setupConnections();
	showMetaInformation();
}

void MetadataEdit::save()
{
	if( d->changed ) 
	{
		saveMetaInformation();
	}
}

void MetadataEdit::loadSingleRom(const QString &romUrl)
{
	if( romUrl.isEmpty() )
	{
		kdDebug() << k_funcinfo << "Told to load an empty rom URL!" << endl;
		return;
	}
	
	if( d->changed )
	{
		// save the existing changes before trampling
		saveMetaInformation();
	}
	
	d->currentRomUrl = romUrl;
	d->currentRomInformation = Kamefu::CollectionAccess::self()->getRomMetadataFromRomUrl(d->currentRomUrl);
	if( !d->currentRomInformation.isValid() )
	{
		kdDebug() << k_funcinfo << "Could not retrieve ROM!" << endl;
		return;
	}

	d->changed = false;
}

void MetadataEdit::loadRoms(const KURL::List &romList)
{
	if( romList.isEmpty() )
	{
		kdDebug() << k_funcinfo << "Tried to load a empty list of ROM url !" << endl;
		return;
	}

	d->romList = romList;
	
	QValueList<Kamefu::RomMetaInformation> metaInfoList = Kamefu::CollectionAccess::self()->getRomList(romList);
	
	QString editor, country, genre, year;
	bool commonEditor = true, commonCountry = true, commonGenre = true, commonYear = true;
	
	Kamefu::RomMetaInformation temp;
	
	QValueList<Kamefu::RomMetaInformation>::ConstIterator it, itEnd = metaInfoList.constEnd();

	temp = *(metaInfoList.constBegin());
	editor = temp.editor();
	country = temp.country();
	genre = temp.genre();
	year = temp.year();

	for( it = metaInfoList.constBegin(); it != itEnd; ++it )
	{
		temp = *it;
		if(temp.editor() != editor)
			commonEditor = false;
		if(temp.country() != country)
			commonCountry = false;
		if(temp.genre() != genre)
			commonGenre = false;
		if(temp.year() != year )
			commonYear = false;
	}

	if( commonEditor )
		d->currentRomInformation.setEditor( editor );
	if( commonCountry )
		d->currentRomInformation.setCountry( country );
	if( commonGenre )
		d->currentRomInformation.setGenre( genre );
	if( commonYear )
		d->currentRomInformation.setYear( year );

	d->changed = false;
}

void MetadataEdit::slotFieldsChanged()
{
	bool hasChanged = false;
	
	// Check if the fields has really changed.	
	if( d->currentRomInformation.romName() != d->nameLineEdit->text() ||
		d->currentRomInformation.editor() != d->developerComboBox->currentText() ||
		d->currentRomInformation.year() != QString::number(d->yearSpinBox->value()) ||
		d->currentRomInformation.country() != d->countryComboBox->currentText() ||
		d->currentRomInformation.genre() != d->genreComboBox->currentText() ||
		d->currentRomInformation.titlescreenPath() != d->titleScreenURLRequester->url() ||
		d->currentRomInformation.screenshotPath() != d->screenShotURLRequester->url()
	)
		hasChanged = true;

	d->changed = hasChanged;

	emit changed(hasChanged);
}

void MetadataEdit::saveMetaInformation()
{
	if( d->changed )
	{
		kdDebug() << k_funcinfo << "save to the database" << endl;
		
		d->currentRomInformation.setEditor(d->developerComboBox->currentText());
		d->currentRomInformation.setYear(d->yearSpinBox->text());
		d->currentRomInformation.setCountry(d->countryComboBox->currentText());
		d->currentRomInformation.setGenre(d->genreComboBox->currentText());
		
		// do transaction
		if( d->romList.isEmpty() )
		{
			d->currentRomInformation.setRomName(d->nameLineEdit->text());
			d->currentRomInformation.setTitlescreenPath(d->titleScreenURLRequester->url());
			d->currentRomInformation.setScreenshotPath(d->screenShotURLRequester->url());

			Kamefu::CollectionAccess::self()->modifyExistingRom(d->currentRomInformation);	
			// let others know
			emit metadataChanged(d->currentRomUrl);
		}
		else
		{
			Kamefu::CollectionAccess::self()->modifyRoms( d->currentRomInformation, d->romList );
			emit metadataChanged(d->romList);
		}
	}
}

void MetadataEdit::showMetaInformation()
{	
	d->nameLineEdit->setText( d->currentRomInformation.romName() );
	d->systemLineEdit->setText( d->currentRomInformation.system() );
	d->yearSpinBox->setValue( d->currentRomInformation.year().toInt() );

	if( !d->currentRomInformation.editor().isEmpty() )
		d->developerComboBox->setCurrentItem( d->developerList.findIndex(d->currentRomInformation.editor()) );
	else
		d->developerComboBox->setCurrentItem( QString(), true );
		
	if( !d->currentRomInformation.country().isEmpty() )
		d->countryComboBox->setCurrentItem( d->countryList.findIndex(d->currentRomInformation.country()) );
	else
		d->countryComboBox->setCurrentItem( QString(), true );

	if( !d->currentRomInformation.genre().isEmpty() )
		d->genreComboBox->setCurrentItem( d->genreList.findIndex(d->currentRomInformation.genre()) );
	else
		d->genreComboBox->setCurrentItem( QString(), true );
		
	if( d->romList.isEmpty() )
	{
		d->hashLineEdit->setText( d->currentRomInformation.md5Hash() );
		d->pathLineEdit->setText( d->currentRomInformation.url() );
		d->titleScreenURLRequester->setURL( d->currentRomInformation.titlescreenPath() );
		d->screenShotURLRequester->setURL( d->currentRomInformation.screenshotPath() );
	}
}

void MetadataEdit::setupLayout()
{
	// TODO: QLabel for screenshot / title screen display
	QVBoxLayout *layout = new QVBoxLayout(this, 10, 10);
	
	// two columns for the input fields
	QHBoxLayout *columnsLayout = new QHBoxLayout(layout, 10);
	QVBoxLayout *leftColumnLayout = new QVBoxLayout(columnsLayout, 2);
	QVBoxLayout *rightColumnLayout = new QVBoxLayout(columnsLayout, 2);
	
	// elements in the left column
	d->nameLineEdit = new KLineEdit(this, "nameLineEdit");
	addItem(i18n("&Name:"), d->nameLineEdit, leftColumnLayout, "package_games");
	
	d->developerComboBox = new KComboBox(true, this, "developerComboBox");
	// Retrieve the developer/editor list from the database and sort it.
	d->developerList = Kamefu::CollectionAccess::self()->retrieveTableEntries("editor");
	d->developerList.sort();
	// Insert the items into the combo box and the completion object.
	d->developerComboBox->insertStringList(d->developerList);
	d->developerComboBox->setCompletionMode( KGlobalSettings::CompletionAuto  );
	d->developerComboBox->completionObject()->insertItems( d->developerList );
	connect(d->developerComboBox,SIGNAL(returnPressed(const QString&)),d->developerComboBox->completionObject(),SLOT(addItem(const QString&)));
	addItem(i18n("&Developer:"), d->developerComboBox, leftColumnLayout, "personal");
	
	d->countryComboBox = new KComboBox(true, this, "countryComboBox");
	// Retrieve the country list from the database and sort it.
	d->countryList = Kamefu::CollectionAccess::self()->retrieveTableEntries("country");
	d->countryList.sort();
	// Insert the items into the combo box and the completion object.
	d->countryComboBox->insertStringList(d->countryList);
	d->countryComboBox->setCompletionMode( KGlobalSettings::CompletionAuto  );
	d->countryComboBox->completionObject()->insertItems( d->countryList );
	connect(d->countryComboBox,SIGNAL(returnPressed(const QString&)),d->countryComboBox->completionObject(),SLOT(addItem(const QString&)));
	addItem(i18n("&Country:"), d->countryComboBox, leftColumnLayout, "locale");
	
	// elements in the right column
	d->systemLineEdit = new KLineEdit(this, "systemLineEdit");
	d->systemLineEdit->setReadOnly(true);
	addItem(i18n("&System:"), d->systemLineEdit, rightColumnLayout, "kcmprocessor");
	
	d->yearSpinBox = new KIntSpinBox(this, "yearSpinBox");
	d->yearSpinBox->setMaxValue(9999);
	addItem(i18n("&Year:"), d->yearSpinBox, rightColumnLayout, "date");
	
	d->genreComboBox = new KComboBox(true, this, "genreComboBox");
	// Retrieve the genre list from the database and sort it.
	d->genreList = Kamefu::CollectionAccess::self()->retrieveTableEntries("genre");
	d->genreList.sort();
	// Insert the items into the combo box and the completion object.
	d->genreComboBox->insertStringList(d->genreList);
	d->genreComboBox->setCompletionMode( KGlobalSettings::CompletionAuto  );
	d->genreComboBox->completionObject()->insertItems( d->genreList );
	connect(d->genreComboBox,SIGNAL(returnPressed(const QString&)),d->genreComboBox->completionObject(),SLOT(addItem(const QString&)));
	addItem(i18n("&Genre:"), d->genreComboBox, rightColumnLayout);
	
	// We are in single ROM edit mode.
	if( d->romList.isEmpty() )
	{
		QVBoxLayout *fieldsLayout = new QVBoxLayout(layout, 2);
		
		// a couple things go outside the columns and span both
		QStringList availableImagesTypes = KImageIO::mimeTypes( KImageIO::Reading ); // Get the available image mime type.
	
		d->titleScreenURLRequester = new KURLRequester(this, "titleScreenURLRequester");
		d->titleScreenURLRequester->setFilter( availableImagesTypes.join(" ") );
		addItem(i18n("&Title Screen Image:"), d->titleScreenURLRequester, fieldsLayout);
	
		d->screenShotURLRequester = new KURLRequester(this, "screenShotURLRequester");
		d->screenShotURLRequester->setFilter( availableImagesTypes.join(" ") );
		addItem(i18n("Screenshot Image:"), d->screenShotURLRequester, fieldsLayout);
	
		d->hashLineEdit = new KLineEdit(this, "hashLineEdit");
		d->hashLineEdit->setReadOnly(true);
		addItem(i18n("&Hash:"), d->hashLineEdit, fieldsLayout);
		
		d->pathLineEdit = new KLineEdit(this, "pathLineEdit");
		d->pathLineEdit->setReadOnly(true);
		addItem(i18n("&Path:"), d->pathLineEdit, fieldsLayout);
	}
	else
	{
		d->nameLineEdit->setReadOnly(true);
	}
	layout->addItem(new QSpacerItem(0, 20, QSizePolicy::Expanding, QSizePolicy::Expanding));
}

void MetadataEdit::setupConnections()
{
	// watch all of the text entry fields for changes
	connect(d->nameLineEdit, SIGNAL(textChanged(const QString&)), this, SLOT(slotFieldsChanged()));
	connect(d->developerComboBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotFieldsChanged()));
	connect(d->countryComboBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotFieldsChanged()));
	connect(d->yearSpinBox, SIGNAL(valueChanged(int)), this, SLOT(slotFieldsChanged()));
	connect(d->genreComboBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotFieldsChanged()));
	if( d->romList.isEmpty() )
	{
		connect(d->titleScreenURLRequester, SIGNAL(textChanged(const QString&)), this, SLOT(slotFieldsChanged()));
		connect(d->screenShotURLRequester, SIGNAL(textChanged(const QString&)), this, SLOT(slotFieldsChanged()));
	}
}

void MetadataEdit::addItem(const QString &text, QWidget *item, QBoxLayout *layout, const QString &iconName)
{
	if( !item || !layout )
		return;

	QLabel *label = new QLabel(item, text, this);
	QLabel *iconLabel = new QLabel(item, 0, this);
	
	if( !iconName.isNull() )
	{
		iconLabel->setPixmap(SmallIcon(iconName));
		label->setMinimumHeight(iconLabel->height());
	}	

	if( layout->direction() == QBoxLayout::LeftToRight ) 
	{
		layout->addWidget(iconLabel);
		layout->addWidget(label);
		layout->addWidget(item);
	}
	else 
	{
		QHBoxLayout *l = new QHBoxLayout(layout);
		
		l->addWidget(iconLabel);
		l->addWidget(label);
		l->setStretchFactor(label, 1);
	
		l->insertStretch(-1, 1);
	
		layout->addWidget(item);
	}
}

#include "metadataedit.moc"
