/*
    kfile_nes.cpp - NES(Nintendo Entertainment System) Rom image KFile Plugin

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "kfile_nes.h"

#include <kgenericfactory.h>
#include <kmdcodec.h>

#include <qfile.h>
#include <qdatastream.h>

#include "nes_switch.h"

typedef KGenericFactory<KNesFilePlugin> knesfileFactory;

K_EXPORT_COMPONENT_FACTORY(kfile_nes, knesfileFactory( "kfile_nes" ))

KNesFilePlugin::KNesFilePlugin(QObject *parent, const char *name,
                       const QStringList &args)
    : KFilePlugin(parent, name, args)
{
	//add the mimetype here:
	KFileMimeTypeInfo *info = addMimeTypeInfo( "application/x-rom-nes" );
	KFileMimeTypeInfo::ItemInfo* item;

	// General Group
	KFileMimeTypeInfo::GroupInfo *groupGeneral = 0L;
	groupGeneral = addGroupInfo(info, "romGeneralInfo", i18n("General"));
	addItemInfo(groupGeneral, "md5Hash", i18n("MD5 Hash"), QVariant::String);
	
	// NES-specific group
	KFileMimeTypeInfo::GroupInfo* groupNES = 0L;
	groupNES = addGroupInfo(info, "nesInfo", i18n("NES Info"));

	item =  addItemInfo(groupNES, "PRG", i18n("PRG Size"), QVariant::Int);
	setUnit(item, KFileMimeTypeInfo::Bytes);
	item = addItemInfo(groupNES, "CHR", i18n("CHR Size"), QVariant::Int);
	setUnit(item, KFileMimeTypeInfo::Bytes);
	addItemInfo(groupNES, "Mapper", i18n("Mapper"), QVariant::Int);
	addItemInfo(groupNES, "Switch", i18n("Switch"), QVariant::String);
}

bool KNesFilePlugin::readInfo( KFileMetaInfo& info, uint /*what*/)
{
	KFileMetaInfoGroup groupGeneral = appendGroup(info, "romGeneralInfo");
	KFileMetaInfoGroup groupNES = appendGroup(info, "nesInfo");

	Q_INT32 header;
	Q_UINT8 prg, chr, mapper, mapper1, mapper2, type;
	QString md5Hash;
	int prgSize, chrSize;

	QFile nes_file(info.path());

	if(nes_file.open(IO_ReadOnly))
	{
		QDataStream stream( &nes_file);

		stream >> header;
		// Check for "NES"+0x1A in header for validity
		if(header == 0x4e45531a)
		{
			nes_file.at(4);
			stream >> prg;
			stream >> chr;
			stream >> mapper1;
			stream >> mapper2;
			type = mapper1;
			// Read first nibble of byte 6 (Mapper+ Switches) and byte 7 (extended iNES format)
			mapper1 = mapper1 >> 4;
			mapper2 = mapper2 >> 4;
			// Make the final mapper, combinason of first nibble of byte 7 and 6
			mapper = (mapper2 << 4) | mapper1;
			// Get the second nibble of byte 6 (Mapper + Switches)
			type = type & 0x0f;

			nes_file.reset();
			// Generate the MD5 Hash
			KMD5 context(0L);
			context.update(nes_file);
			md5Hash = context.hexDigest();
			
			nes_file.close();
		}
		else
		{
			nes_file.close();
			return false;
		}
	}
	else
	{
		return false;
	}

	prgSize = int(prg * (16 * 1024)); // The number read is the number of 16kb of PRG page.
	chrSize = int(chr * (8 * 1024)); // The number read is the number of 8kb of CHR page.

	appendItem(groupGeneral, "md5Hash", md5Hash);

	appendItem(groupNES, "PRG", prgSize);
	appendItem(groupNES, "CHR", chrSize);
	appendItem(groupNES, "Mapper", mapper);
	// qs_nes_switch is defined in nes_switch.h
	appendItem(groupNES, "Switch",  qs_nes_switch[type]);

    return true;
}

#include "kfile_nes.moc"

