/***************************************************************************
                          carchive.h  -  description
                             -------------------
    begin                : Sat Dec 2 2000
    copyright            : (C) 2000 by Eric Coquelle
    email                : coquelle@caramail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CARCHIVE_H
#define CARCHIVE_H

#include <sys/types.h>

#include <qstring.h>
#include <qstringlist.h>
#include <qlistview.h>
#include <qdatetime.h>
#include <qobject.h>
#include <qcombobox.h>
#include <qprogressbar.h>

#include <karchive.h>
#include <kglobal.h>
#include <klocale.h>
#include <qregexp.h>

#include "processus.h"
#include "clistviewitem.h"
#include "ccheckfiles.h"
#include "resource.h"


/**This is the base class for all kind of archives
  *@author Eric Coquelle
  */

class CArchive : public QObject {
Q_OBJECT

protected:
  struct archiveElement
  {
  	QString name;
	int size;
	QDateTime date;
  };

public: 
  CArchive();
  virtual ~CArchive();
  
  /*Static functions */
  /** set the main widget so as to this class can draw the archive content */
  static void setWidgetListView( QListView *l );
  /** Set the size of the icons in the listview to @param size */
  static void setIconSize( int size );
  /** if @param icons is false, don't display the icons, to speed up archive reading */
  static void setDisplayIcons( bool icons );
  /** set the main widget so as to this class can display the subfolders */
  static void setWidgetComboDirectories( QComboBox *c );
  /** if @param overwr is true, one will overwrite files when extracting files */
  static void setOverwriteFilesWhenExtracting( bool overwr );
  /** If @param read is true, read current archive
   with a stream, ie by a quick way, but this freezes karchiveur's interface */
  static void setReadArchiveWithStream( bool read );
  /** A link to the progressbar */
  static void setWidgetProgressBar( QProgressBar *progress );
  /** If true, when adding a directory, also add the subdirs */
  static void setRecursiveMode (bool rec);
  /** the the format of displayed date to d */
  static void setKindOfDate( int d );
  /** If avaible for the compressor, set the compress
      rate to @param rate. @param rate must be in [1,9] */
  static void setCompressRate( int rate );

  /*virtual methods implemented in child classes*/
  virtual void displayArchiveContent();
  virtual void extractArchive(QString & extractpath, int extractall, QString & filetoextract);
  virtual void removeFilesFromArchive(QStringList filesToDelete);
  virtual void addFilesToArchive( QStringList filestoadd, bool removeoriginalfiles, int action, QString relativepath=NULL);
  virtual void createArchive(QString nameofarchive, QStringList filestoadd, QString relativepath=NULL);
  virtual void repairCurrentArchive(){};
  virtual void testCurrentArchiveIntegrity(){emit archiveReadEnded();};


  /** set the name of the working archive to @param archName */
  void setArchiveName( QString archName );
  QString getArchiveName();
  /** set the name of the password to @param archPassword */
  void setPassword( QString archPassword );
  /** Kill the extract process */
  void stopExtracting();
  /** Kill the read process */
  void stopReading();
  /**get the StdErr of compressors*/
  QStringList getErrors();
  /** Returns true if selected entry is a subdirectory of currently displayed ArchiveDirectory*/
  bool isDirectory(QString entryname);
  /**Display archive content, using KIO methods */
  void displayArchiveContent(QString , QString );
  /** if true, KIO supported archives will be displayed dir. by dir. */
  void setDisplayArchiveByDirectories(bool);
  /** Return a list of all files (with their path, as stored in the archive, their size and owner) of all archives' files */
  QValueList<CArchive::archiveElement> getAllFilesWithInfos(bool fetchonlyselected=false);
  QValueList<CArchive::archiveElement> getAllFilesWithInfosInCurrentSubdir(QString, QString);
  QValueList<CArchive::archiveElement> getAllFilesWithInfos(const KArchiveDirectory*, QString);
  /** Return a list of all files (with their path, as stored in the archive,) of all archives' files;
    * or only selected ones if @param  fetchonlyselected is true (default is false)*/
  QStringList getAllFiles(bool fetchonlyselected=false);
  /** Return a list of all files (with their path, as stored in the archive,) of all archives' files
    contained in the @param foldername directory. Add @param folderpath to each entry of the returned
    filelist*/
  //QStrList getAllFiles(QString foldername, QString folderpath);
  /**Returns false as all archives cannot be listed by KIO*/
  virtual bool supportDisplayByDirs(){return false;};
  /**Returns true if archive type supports passwords*/
  virtual bool supportPassword(){return false;};
  /**Returns true if current archive can be repaired*/
  virtual bool canRepairArchive(){return false;};
  /** Returns the password */
  QCString getPassword();
  /**Returns the name of the repaired archive*/
  QString getRepairedArchiveName(){return repairedArchiveName;};
  /**Returns the numbers of files in current archive*/
  int countFiles();
  /**Get some infos on current archive (size, compress rate...)*/
  void fetchArchiveInfos(int& nbfiles, int& archiveSize, int& totalSizeOfFiles, float& sigma, float& compressrate);
  /**Set a regular expression (used for extraction)*/
  void setRegExp(QRegExp);
  /** Returns the comments stored in the archive*/
  QStringList getArchiveComments();
  /** Returns true if current archive is an autoconf-based source software*/
  bool isSourceSoftware();

protected:
  /** Set the files which can be extracted - return false if none of the files can be extracted*/
  QStringList checkFilesOnly(QString& extractpath, int& extractall);
  bool checkFiles(QString, int);
  /** Find which icon to insert in the listview, by
    examinating the extension of files the archive contains */
  void setIcon(QString name, QString permissions, CListViewItem* elementListe);
  void setIcon(QString name, mode_t permissions, CListViewItem* elementListe);
  /** prepare the local variable, clear the listview
      before reading any archive */
  void initializeReadingArchive();
  /** Display archive content, using KIO methods */
  void displayArchiveContent(const KArchiveDirectory*, QString);
  /** Convert mode_t permissions in human readable ones */
  QString getIndividualPermissions(bool b, QString perm);
  QString getReadablePermissions(mode_t mode);
  /**Convert date to string*/
  QString getLocalizedDate(QDate date);
  /**Convert string to date*/
  QDate getDateFromLocalizedDate(QString localizeddate);

protected slots:
  void haveStdErr(KProcess *, char *buffer, int length);
  virtual void haveSdtErrExtract(KProcess*, char*, int );

public:
signals:
  void archiveReadEnded(void);
  void newarchiveentryreaded();
  void sourceSoftwareSpotted();
  
  public:
    /* A pointer to the listview (where archive content is displayed */
    static QListView *list;
    
protected:
  /* The cprocessus object, it will be used to list archive content...*/
  CProcessus processread;
  /* another one to extract files from archive */
  CProcessus processextract;
  /* another one to add files from archive */
  CProcessus processadd;
  /* The counter for the progressbar*/
  int counter;
  /* The name of the archive we are working on*/
  QString archiveName;
  /* If true, read the archive using the CProcessus class.
            Quicker, but freeze the interface */
  static bool readArchiveWithStream;
  /* if readArchiveWithStream is false, we store in a buffer the output
    * of the running command */
  char m_buffer[1024];
  /* The size of icons in the listview*/
  static int iconsize;
  /*Has user canceled current process ?*/
  bool stopreadprocess;
  /*option for displaying the date*/
  static int kindofdate;
  /*A pointer to the progressbar*/
  static QProgressBar *progressbar;
  /*If true, overwrite files when extracting*/
  static bool overwrite;
  /*If true, add to the archive subdirectories*/
  static bool recursivemode;
  /*Display icons for archive content ?*/
  static bool displayicons;
  /*The compress rate, if avaible: 1=min, 9=max*/
  static int compressrate;
  /*The error messages*/
  QStringList errors;
  /*The comments stored in the archive*/
  QStringList archivecomments;
  /* View mode: directory bu directory, or all files at once */
  static bool viewbydirectories;
  /*Combobox that displays a "tree" of folders when browsing archives dir by dir*/
  static QComboBox* combodirectories;
  /* Current directory*/
  QValueList<KArchiveDirectory*> knowndirectories;
  QValueList<archiveElement> listArchiveElements;
  /** Password for archive */
  static QCString archivePassword;
  /**A regexp to filter files one wishes to extract*/
  QRegExp regularexpression;
  /**Check is current archive is an autoconf-based source software*/
  int issourcesoftware;
  bool lookforsoucearchive;
  enum sourcesoftwareflags {HAS_CONFIGURE=2, HAS_MAKEFILE_AM=4};
  /**The name of a repaired archives*/
  QString repairedArchiveName;
};

#endif
