/***************************************************************************
                          karchiveur.cpp  -  description
                             -------------------
    begin                : mer jun 14 00:26:31 CEST 2000
    copyright            : (C) 2000 by Eric Coquelle
    email                : coquelle@caramail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// system includes
#include <errno.h>
#include <math.h>

// include files for QT
#include <qdir.h>
#include <qprinter.h>
#include <qpainter.h>

// include files for KDE
#include <kdeversion.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kconfig.h>
#include <kfileitem.h>
#include <kopenwith.h>
#include <kstdaction.h>
#include <kurldrag.h>
#include <kedittoolbar.h>
#include <kdebug.h>
#include <kinputdialog.h>
#include <kpopupmenu.h>
#include <karchive.h>
#include <ktar.h>
#include <kzip.h>
#include <kpassdlg.h>

// application specific includes
#include "cfileinfo.h"
#include "cfiledialog.h"
#include "karchiveur.h"
#include "karchiveurview.h"
#include "cgpg.h"
#include "cmail.h"
#include "cupdate.h"
#include "resource.h"

KarchiveurApp::KarchiveurApp()
{
  selectedfilesStatusbarMessage.nbselectedfiles=0;
  selectedfilesStatusbarMessage.sizeselectedfiles=0;
  listUndo.clear();
  listRedo.clear();

  archiveobj=NULL;
  archiveoperation=NULL;
  controlkeypressed=false;
    
	config=kapp->config();

	///////////////////////////////////////////////////////////////////
	// call inits to invoke all other construction parts
        initView();
	initStatusBar();
	initActions();

  //Set toolbar directory browser
        cbdirectories=new KComboBox(toolBar("mainToolBar"), "toolbar_combo");
        connect(cbdirectories, SIGNAL(activated(int)), this, SLOT(slotComboDirectoriesActivated(int)));

        enableMenus(false);
	operation=NOARCHIVE; //No file loaded, nothing to do !

	archivechoice = new CArchiveChoice(this, "Identify_and_Open_Archive_Class");
	archivechoice->hide();

	//Launch the archive browser
	navigateur = new CArchiveBrowser(this,"File_Browser_Dialog");
	connect( navigateur, SIGNAL(windowwillbeclosed()), this, SLOT(slotHideButineur()));
	connect(navigateur->QListBox_1, SIGNAL(selected(const QString&)), this, SLOT(slotOpenFileFromArchiveBrowser(const QString&)));

	// reads in all config-options
	readOptions();

	//Initialise for drops
	//The main drop zone is in KarchiveurView (via QListView* liste)
	setAcceptDrops(true);
    connect(view,SIGNAL(doDragAndDrop()), this, SLOT(slotDoDragAndDrop()));
	connect(view,SIGNAL(rightButtonPressed ( QListViewItem *, const QPoint &, int )), this, SLOT(slotShowContextualMenu ( QListViewItem *, const QPoint &, int )));
	connect(view,SIGNAL(clicked ( QListViewItem * )), this, SLOT(slotMouseClick ( QListViewItem * )));

	//Initialise the process object:
	process=new CProcessus();
	process->clearArguments();
	connect(process,SIGNAL(receivedStdout(KProcess*, char*, int)),this,SLOT(recueSdtOut(KProcess*,char*,int)));

	//Initialise the find in this archive tool
	find = new CFind(this,"Search_in_this_Archive_Class");
	connect( find, SIGNAL(findNext(QString)), this, SLOT(slotFindInThisArchive(QString)));
	connect( find, SIGNAL(findClose()), this, SLOT(slotCloseFind()));

	//Initialise the find archives on disk tool
	searchBox=new CSearch(0,"Search_for_Archives_Dialog");
	connect(searchBox->foundarchives, SIGNAL(selected(const QString&)), this, SLOT(slotOpenFileFromArchiveFinder(const QString&)));

   QString directory;
   QString error;
   directory.sprintf("karchivertmp.%d/", getpid());
   error=strerror(errno);
   tmpdir = locateLocal( "tmp", directory );
  
    if (tmpdir.isNull())
      kdWarning() << i18n("Failed to create temporary "
              "attachment directory %2: %1")
              .arg(error)
              .arg(directory) << endl;
  
  
    directory.sprintf("karchivertmp.%d/extract/", getpid());
    tmpdir = locateLocal( "tmp", directory );
  
    if (tmpdir.isNull())
      kdWarning() <<i18n("Failed to create temporary "
              "attachment directory %2: %1")
              .arg(error)
              .arg(directory) << endl;
  
    directory.sprintf("karchivertmp.%d/undo/", getpid());
    tmpdirundo = locateLocal( "tmp", directory );
  
    if (tmpdirundo.isNull())
      kdWarning() <<i18n("Failed to create temporary "
              "attachment directory %2: %1")
              .arg(error)
              .arg(directory) << endl;
  
  slotViewButineur();
  
  CArchive::setWidgetListView ( view );
  CArchive::setWidgetComboDirectories(cbdirectories);
  CArchive::setWidgetProgressBar(progress);
  CArchive::setOverwriteFilesWhenExtracting(overwrite);
  CArchive::setIconSize(16);
  CArchive::setReadArchiveWithStream(readarchivewithstream);
  animation=new CAnimation();
  animation->hide();
}

KarchiveurApp::~KarchiveurApp()
{
  QString directory;
  directory.sprintf("karchivertmp.%d/", getpid());
  tmpdir = locateLocal( "tmp", directory );

  process->clearArguments();
  *process << "rm" << "-Rf" << tmpdir;
  process->start(KProcess::Block);

  if ( archiveobj != NULL )
    delete archiveobj;
  
  if ( archiveoperation != NULL )
    delete archiveoperation;
  
  delete archivechoice;
  delete cbdirectories;
  delete navigateur;
  delete find;
  delete searchBox;
  delete process;
  delete tipofday;
}

void KarchiveurApp::initStatusBar()
{
	///////////////////////////////////////////////////////////////////
	// STATUSBAR
	statusBar()->insertItem(i18n(IDS_STATUS_DEFAULT), ID_STATUS_MSG);

	//Additionnal init to the statusbar
        progress = new QProgressBar(statusBar(),"Progress_Bar");
        progress->setFixedHeight ( statusBar()->sizeHint().height() -4 );
        statusBar()->addWidget(progress,0,true);

        led=new CLed(statusBar(),"statusbar_led");
        led->setLook(KLed::Sunken);
        led->setFixedHeight ( statusBar()->sizeHint().height()-4 );
        led->setFixedWidth ( statusBar()->sizeHint().height()-4 );
        statusBar()->addWidget(led,0,true);
}

void KarchiveurApp::initView()
{
	////////////////////////////////////////////////////////////////////
	// create the main widget here that is managed by KTMainWindow's view-region and
	// connect the widget to your document to display document contents.
	view = new KarchiveurView(this, "list_files_in_archive");
	setCentralWidget(view);
}

void KarchiveurApp::saveOptions()
{
	config->setGroup("General Options");
	config->writeEntry("Geometry", size());
	config->writeEntry("Show Toolbar", viewToolbar->isChecked());
	config->writeEntry("Show Statusbar",viewStatusbar->isChecked());
	config->writeEntry("Show Navigateur",viewArchiveBrowser->isChecked() );
	config->writeEntry("ToolBarPos", (int) toolBar("mainToolBar")->barPos());
	fileOpenRecent->saveEntries(config,"Recent Files");

	config->setGroup("LASTDIRS");
#if KDE_IS_VERSION(3,1,3)
	config->writePathEntry("Extraction",CExtraction::getLastExtractPath());
	config->writePathEntry("Ouverture",archivechoice->getAccessPath());
#else
	config->writeEntry("Extraction",CExtraction::getLastExtractPath());
	config->writeEntry("Ouverture",archivechoice->getAccessPath());
#endif
}


void KarchiveurApp::readOptions()
{
  config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
  viewToolbar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewStatusbar->setChecked(bViewStatusbar);
  slotViewStatusBar();

  bool bViewNavigateur = config->readBoolEntry("Show Navigateur", true);
	viewArchiveBrowser->setChecked(bViewNavigateur);
	slotViewButineur();

  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) config->readNumEntry("ToolBarPos", KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);

  // initialize the recent file list
  fileOpenRecent->loadEntries(config,"Recent Files");

  QSize size=config->readSizeEntry("Geometry", new QSize(640,480));
  if( !size.isEmpty() )
  {
          resize(size);
  }

  config->setGroup("OPTIONS");
  includewholepath = config->readBoolEntry("ToutChemin",false);
  recursive = config->readBoolEntry("Recursif",true);//when creating an archive, aslo compress subfolders
  overwrite = config->readBoolEntry("OverwriteWhenExtracting",false);
  cheminabsolu = config->readBoolEntry("CheminAbsolu",false);
  view->setAlternateBackground( config->readBoolEntry("AlternateBackground",true) );
  viewbydirectories=config->readBoolEntry("ReadByDirs",true);
  preferedArchiveType=config->readEntry("PreferedArchiveType",".tar.gz");
  defaultdropbehaviour=config->readBoolEntry("DefaultDropBehaviour",false);

	switch(config->readNumEntry("LectureParFlot",1))
	{
		case 0:
			readarchivewithstream=false;//Many problems with stream reading, not needed anymore
			CArchive::setDisplayIcons(true);
			break;
	 	case 1:
			readarchivewithstream=false;
			CArchive::setDisplayIcons(true);
			break;
		case 2:
			//Don't display icons for archive dontent display.
			readarchivewithstream = false;
			CArchive::setDisplayIcons(false);
			break;
	}

	if ( config->readNumEntry("SelectionMode",0) == 0 )
	{
		view->setSelectionMode(QListView::Multi);
	}
	else
	{
		view->setSelectionMode(QListView::Extended);
	}

	// For the compress rate, in the config file 1 = min compress
	compresslevel=config->readNumEntry("TauxCompression",9);
	CArchive::setCompressRate(compresslevel);

	config->setGroup("LASTDIRS");
	QString cheminExtraction=config->readPathEntry("Extraction",QDir::homeDirPath());
	QString cheminOuverture=config->readPathEntry("Ouverture",QDir::homeDirPath());

	int optionExtract=config->readNumEntry("OptionExtraction",1);//Default extraction: in home dir
	optionOpening=config->readNumEntry("OptionOuverture",1);//Default opening: in home dir
	archivechoice->setFullPath(cheminOuverture,optionOpening);
	archivechoice->setForInitPreviousPath(cheminOuverture);
	navigateur->setPath(archivechoice->getAccessPath());
	CExtraction::setDefaultOptions(cheminExtraction, optionExtract);
	
        config->setGroup("DATE");
	CArchive::setKindOfDate(config->readNumEntry("KindOfDate",0));

	config->setGroup("TipOfTheDay");
	displaytipofday=config->readBoolEntry("show_tod", true);

	CArchive::setRecursiveMode(recursive);
 
}


void KarchiveurApp::saveProperties(KConfig *)
{
}


void KarchiveurApp::readProperties(KConfig* _cfg)
{
	QString filename = _cfg->readPathEntry("filename");
	bool modified = _cfg->readBoolEntry("modified", false);

	if( modified )
	{
		bool canRecover;
		QString tempname = kapp->checkRecoverFile(filename, canRecover);

		if(canRecover)
		{
			CFileInfo info(filename);
			QFile::remove(tempname);
		}
	}
	else
	{
		if(!filename.isEmpty())
		{
		}
	}
}

bool KarchiveurApp::queryClose()
{
	return true;
}

bool KarchiveurApp::queryExit()
{
	saveOptions();
	return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void KarchiveurApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new application window..."));

  KarchiveurApp *new_window= new KarchiveurApp();
  new_window->show();

  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KarchiveurApp::slotFileNew()
{
  slotStatusMsg(i18n("Creating a new archive..."));
	setCaption("");

  view->clear();
  operation=NOARCHIVE;

	if(archiveobj!=NULL)
    {
		delete archiveobj;
        archiveobj=NULL;
    }

  slotAddFiles();
}

void KarchiveurApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening archive..."));
  led->setColor(QColor("red"));

  if(archivechoice->openArchive()==NULL)
  {
    slotStatusMsg(IDS_STATUS_DEFAULT);
    return;
  }

  displayArchiveContent();

  displayMessageArchiveSize();
}

void KarchiveurApp::slotFileOpenRecent(const KURL& url)
{
  QString archiveOpenName=openRemoteFile(url);
  if(!archiveOpenName.isEmpty())
    {
      archivechoice->setArchiveName(archiveOpenName);
      navigateur->setPath(archivechoice->getAccessPath());

      displayArchiveContent();
    }
}

void KarchiveurApp::slotFileOpenRecent(int id_)
{
  archivechoice->setArchiveName(recentFiles.at(recentFilesMenu->indexOf(id_)));
	navigateur->setPath(archivechoice->getAccessPath());
	displayArchiveContent();
}

/* This method will convert the archive format*/
void KarchiveurApp::slotFileSaveAs()
{
	led->setColor(QColor("red"));
	slotStatusMsg(i18n("Convert archive format..."));

  if( operation != NOARCHIVE )
  {
    QString newarchivename;
    CFileInfo  fileinfo;
    fileinfo.setFile(archivechoice->getArchiveName());
    
    CFileDialog newarchivenamedlg(CExtraction::getLastExtractPath(),NULL,0,"convert_dialog",false,new QLabel(i18n("Choose the destination format of the archive."),0));
    newarchivenamedlg.setSelection(fileinfo.baseName()+preferedArchiveType);
    newarchivenamedlg.setCaption(i18n("Convert an archive"));
    
    if(newarchivenamedlg.exec())
    {
      newarchivename=newarchivenamedlg.selectedFile();
      newarchivenamedlg.fixFileExtension(newarchivename);
    }
    
    if(newarchivename.isEmpty())
    {
      operation=STANDBY;
      led->setColor(QColor("orange"));
      return;
    }
    
    archiveoperation=new CArchiveOperationConversion(tmpdir, archiveobj, progress);
    connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
    archiveoperation->convertCurrentArchive(newarchivename);
  }
  else
  {
    led->setColor(QColor("orange"));
    slotStatusMsg(i18n("First open an archive!"));
    return;
  }

  slotStatusMsg(IDS_STATUS_DEFAULT);
}

/** Extract all files to a temp dir, and add them to the
  karchiver's sfx module */
void KarchiveurApp::slotMakeSfxArchive()
{
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("Make a Self-extract archive..."));

  archiveoperation=new CArchiveOperationSfx(archiveobj, progress, tmpdir);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  operation = MAKESFX;
  archiveoperation->makeSfxArchive();
}

void KarchiveurApp::slotFileClose()
{
	slotStatusMsg(i18n("Closing file..."));
	close();
	//slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KarchiveurApp::slotFilePrint()
{
	slotStatusMsg(i18n("Printing..."));

	QPrinter printer;

	if (printer.setup(this))
	{
		view->print(&printer);
	}

	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KarchiveurApp::slotFileQuit()
{
	slotStatusMsg(i18n("Exiting..."));
	saveOptions();
	close();
}

void KarchiveurApp::slotViewToolBar()
{
	slotStatusMsg(i18n("Toggle the toolbar..."));
	///////////////////////////////////////////////////////////////////
	// turn Toolbar on or off
	if(!viewToolbar->isChecked())
	{
		toolBar("mainToolBar")->hide();
	}
	else
	{
		toolBar("mainToolBar")->show();
    if(supportDisplayByDirs())
      cbdirectories->show();
	}

	slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

void KarchiveurApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  if(!viewStatusbar->isChecked())
  {
          statusBar()->hide();
  }
  else
  {
          statusBar()->show();
    if(viewbydirectories)
      cbdirectories->show();
    else
      cbdirectories->hide();
  }
  repaint();
  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
}

/** Temporary change the statusbar message */
void KarchiveurApp::slotStatusMsg(const QString &text)
{
    statusBar()->clear();
    if(led->color()==QColor("orange")) //displays a warning message
      statusBar()->message(text, 3500);
    else
      statusBar()->message(text);
}

/** displays a window and search recursively  *
 * for archives in a directory               */
void KarchiveurApp::slotFileSearch()
{
	searchBox->setPath(archivechoice->getAccessPath(), optionOpening);
	searchBox->exec();
}

/** Make all or some files to be extracted */
void KarchiveurApp::slotExtractArchive()
{
  led->setColor(KLed::red);
  slotStatusMsg(i18n("Extract files..."));
  enableMenus(false);

  archiveoperation=new CArchiveOperationExtract(archiveobj, progress);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  operation=EXTRACTING;
  archiveoperation->extractCurrentArchive();
}

/** add some files to the archive */
void KarchiveurApp::slotAddFiles()
{
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("Adding files.."));
  
  if(operation==NOARCHIVE)
  {
    archiveoperation=new CArchiveOperationCreate(archiveobj, progress);
    connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
    archiveoperation->createNewArchive();
  }
  else
  {
    archiveoperation=new CArchiveOperationAdd(archiveobj, progress);
    connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
    archiveoperation->addFilesToCurrentArchive();
  }
}

/** display a file using its associated program */
void KarchiveurApp::slotDisplayFile()
{
  slotStatusMsg(i18n("View this file..."));
  led->setColor(QColor("red"));
  archiveoperation=new CArchiveOperationViewFiles(archiveobj, progress, tmpdir);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  archiveoperation->displayFile( view->currentItem() );
}

/** display all selected files */
void KarchiveurApp::slotDisplaySelectedFiles()
{
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("View all selected files..."));

  archiveoperation=new CArchiveOperationViewFiles(archiveobj, progress, tmpdir);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  archiveoperation->displaySelectedFiles( );
}

/** remove from archive all selected files */
void KarchiveurApp::slotRemoveSelectedFiles(){
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("Deleting selected files..."));

	if(operation!=NOARCHIVE)
	{
		int i;
		int k;
		QListViewItem *li=view->firstChild();
		QString file;
		QStringList listselectedfiles;

		k=view->childCount();
		for(i=0;i<k;i++)
		{
			if(view->isSelected(li))
			{
				file=li->text(6);
                if(file.startsWith("/"))
                  file=file.remove(0,1);
                  file+=li->text(0);
				listselectedfiles.append(file);
				kdDebug()<<QString("KarchiveurApp::slotRemoveSelectedFiles:%1*").arg(file)<<endl;
			}
			if(i!=k)
				li=li->nextSibling();
		}
		if(KMessageBox::questionYesNoList(this, i18n("Do you really want to delete these files?"), listselectedfiles,	i18n("KArchiver"))
			==KMessageBox::No)
		{
			led->setColor(QColor("green"));
		  slotStatusMsg(IDS_STATUS_DEFAULT);
			return;
		}

		archiveobj->removeFilesFromArchive(listselectedfiles);
		operation=STANDBY;
	}

	displayArchiveContent();
	led->setColor(QColor("green"));
	displayMessageArchiveSize();
}

/** Remove form current archive the selected files,
with an undo option (i.e. extract to tmpdirundo, then
remove) */
void KarchiveurApp::slotRemoveWithUndoSelectedFiles(){
  slotStatusMsg(i18n("Deleting selected files..."));

  if(operation!=NOARCHIVE)
  {
    int i;
    int k;
    QListViewItem *li=view->firstChild();
    QString file;
    QString repTrash;

    if(li==0)
    {
      led->setColor(QColor("orange"));
      slotStatusMsg(i18n("First select a file"));
      return;
    }

    led->setColor(QColor("red"));

    k=view->childCount();
    listUndo.clear();
    listRedo.clear();

    for(i=0;i<k;i++)
    {
            if(view->isSelected(li))
            {
                    file=li->text(6);
    if(file.startsWith("/"))
      file=file.remove(0,1);
                    file+=li->text(0);
                    listUndo.append(file);
                    kdDebug()<<QString("KarchiveurApp::slotRemoveWithUndoSelectedFiles of %1").arg(file)<<endl;
                    //listRedo.append(file);
                    archiveobj->extractArchive(tmpdirundo, EXTRACTONE_AND_BLOCK, file);
            }
            if(i!=k)
                    li=li->nextSibling();
    }
  }

  slotRemoveSelectedFiles();
  editUndo->setEnabled(true);
  editRedo->setEnabled(false);
  led->setColor(QColor("green"));
  slotStatusMsg(IDS_STATUS_DEFAULT);
}

/** Undo the previous operation, i.e. the deletion
od files from archive */
void KarchiveurApp::slotUndo(){
  if(listUndo.isEmpty())
  {
          led->setColor(QColor("orange"));
          slotStatusMsg(i18n("Nothing to undo!"));
          return;
  }

  addToArchive( listUndo, true, ADD_AND_REPLACE_FILES, tmpdirundo);
  
  listRedo=listUndo;
  listUndo.clear();
  editUndo->setEnabled(false);
  editRedo->setEnabled(true);
}

/** Redo the last operation (i.e. remove once
again the selected file) */
void KarchiveurApp::slotRedo(){
  if((listRedo.isEmpty())||(!listUndo.isEmpty()))
  {
          led->setColor(QColor("orange"));
          slotStatusMsg(i18n("Nothing to redo!"));
          return;
  }

  slotRemoveWithUndoThoseFiles(listRedo);
  listUndo=listRedo;
  listRedo.clear();
  editUndo->setEnabled(true);
  editRedo->setEnabled(false);
}

/** Remove from current archives the files in
files, with undo possibility */
void KarchiveurApp::slotRemoveWithUndoThoseFiles( QStringList files ){
  slotStatusMsg(i18n("Deleting selected files..."));

	if(operation!=NOARCHIVE)
	{
		QString repTrash;

		led->setColor(QColor("red"));

		listUndo.clear();
		listRedo.clear();

		for(QStringList::Iterator file=files.begin(); file!=files.end(); ++file)
		{
				listUndo.append(/*tmpdirundo+"/"+*/ *file );
				//listRedo.append(file);
				archiveobj->extractArchive(tmpdirundo, EXTRACTONE_AND_BLOCK, *file);
		}
	}

	slotRemoveThoseFiles(files);

	led->setColor(QColor("green"));
  slotStatusMsg(IDS_STATUS_DEFAULT);
}

/** Remove the files in files from current archive */
void KarchiveurApp::slotRemoveThoseFiles( QStringList files ){

  if(operation==NOARCHIVE)
    return;
  
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("Deleting selected files..."));
  
  archiveobj->removeFilesFromArchive(files);
  operation=STANDBY;

  displayArchiveContent();
  led->setColor(QColor("green"));
  displayMessageArchiveSize();
}

/** display some infos about the current archive */
void KarchiveurApp::slotDisplayArchiveInfos(){
  int nbfiles;
  int archiveSize;
  int totalSizeOfFiles;
  float sigma;
  float compressrate;
  float meanFileSize;
  if(operation==NOARCHIVE)
    return;
  archiveobj->fetchArchiveInfos(nbfiles, archiveSize, totalSizeOfFiles, sigma, compressrate);
  meanFileSize=totalSizeOfFiles/(float)nbfiles;
  
  CInfos dlginfos;
  dlginfos.initialise(archivechoice->getArchiveName(), archiveobj->getArchiveComments(), nbfiles,archiveSize , totalSizeOfFiles, (int)meanFileSize, (int) compressrate, (int)sigma);
  dlginfos.exec();
}

/** allow user to set the options */
void KarchiveurApp::slotSetOptions(){
  CPreferences prefs;
  
  if(prefs.exec())
  {
    recursive=prefs.getRecursif();
    CArchive::setRecursiveMode(recursive);
    cheminabsolu=prefs.getCheminAbsolu();
    overwrite=prefs.getOverwriteWhenExtracting();
    CArchive::setOverwriteFilesWhenExtracting(overwrite);
    includewholepath=prefs.getWholePath();
    compresslevel=prefs.getCompressLevel();
    optionOpening=prefs.getKindofOpeningDirectory();
    CArchive::setIconSize(prefs.getSizeofIconListe());
    archivechoice->setOptionOpening(optionOpening);
    int defaultextractdir=prefs.getKindofExtractionDirectory();
    CExtraction::setDefaultOptions( CExtraction::getLastExtractPath(),  defaultextractdir );
    view->setFont(prefs.getFont());
    CArchive::setKindOfDate(prefs.getKindOfDate());
    view->setAlternateBackground(prefs.getAlternateBackground());
    viewbydirectories=prefs.getReadByDirs();
    preferedArchiveType=prefs.getPreferedArchiveType();
    defaultdropbehaviour=prefs.getDefaultDropBehaviour();

		switch(prefs.getLitParFlot())
		{
			case 0:
				readarchivewithstream=true;
				CArchive::setDisplayIcons(true);
				break;
			case 1:
				readarchivewithstream=false;
				CArchive::setDisplayIcons(true);
				break;
			case 2:
				readarchivewithstream=false;
				CArchive::setDisplayIcons(false);
				break;
		}

		CArchive::setCompressRate(prefs.getCompressLevel());
		if(prefs.getSelectionMode()==0)
			view->setSelectionMode(QListView::Multi);
		else
			view->setSelectionMode(QListView::Extended);
	}
}

/** Split/Unsplit the archive in/from small parts */
void KarchiveurApp::slotDiskCutUncut(){
  CWizardCutt wiz_cutt;
  led->setColor(QColor("red"));
      
  if(archiveobj==NULL)
    wiz_cutt.setNoneArchiveOpened();

  if(wiz_cutt.exec())
  {
    if(wiz_cutt.getRecordDirectory()!=NULL)
    {
      enableMenus(false);
      archiveoperation = new CArchiveOperationSplit( archiveobj,  progress );
      connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
      archiveoperation->doCuttUnCutt(wiz_cutt.getRecordDirectory(), wiz_cutt.getCuttSize());
    }
    else
      led->setColor(QColor("orange"));
  }
  else
    led->setColor(QColor("orange"));
}

/* the archive browser has been close => update the configuration menu */
void KarchiveurApp::slotHideButineur() {
  viewArchiveBrowser->setChecked(false);
  slotViewButineur();
}

/** display or not the archive browser */
void KarchiveurApp::slotViewButineur(){
 if(viewArchiveBrowser->isChecked())
	{
		navigateur->show();
	}
	else
		navigateur->hide();
}

/** this method will launch the right compressor to list the file contents
It will then launch the right method to display the content in the listview */
void KarchiveurApp::displayArchiveContent()
{
  QString errormessage;

  QString archivename=archivechoice->getArchiveName();
  led->setColor(KLed::red);
  slotStatusMsg(i18n("Running compressor..."));
  
  if(archiveoperation!=NULL)
    delete archiveoperation;

  archiveoperation = new CArchiveOperationDisplay( archivename, readarchivewithstream, viewbydirectories, archiveobj,  progress );
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));

  errormessage=archiveoperation->canDisplayArchive();

  if(!errormessage.isEmpty()) //cannot display archive
  {
    kdDebug()<<QString("KarchiveurApp::displayArchiveContent - Error:%1").arg(errormessage)<<endl;
    led->setColor(QColor("orange"));
    slotStatusMsg(errormessage);
    if((operation==STANDBY)&&(archiveobj!=NULL))
      archivechoice->setName(archiveobj->getArchiveName());//restore name of previous valide archive,which has been stored in CArchiveChoice
    else
    {
      archiveobj=NULL;
      operation=NOARCHIVE;
    }
    return;
  }

  setCaption(archivename);
  enableMenus(false);
  fileOpenRecent->addURL(KURL(archivechoice->getArchiveName()));
  navigateur->setPath(archivechoice->getCurrentArchiveDirectory());
  view->clear();
  for(int i=0;i<7;i++)
    view->setColumnWidth(i,50);
  operation=READING;
  cbdirectories->clear();
  selectedfilesStatusbarMessage.nbselectedfiles=0;
  selectedfilesStatusbarMessage.sizeselectedfiles=0;

  animation->startAnimation();
  archiveoperation->displayArchiveContent();
}

/** return the size in bytes of the current archive */
int KarchiveurApp::archiveSize(const QString & archive){
  QFileInfo file(archive);

  return file.size();
}

/** Launch the wizard */
void KarchiveurApp::slotWizard(){
  if(archiveobj!=NULL)
  {
    led->setColor(KLed::red);
    slotStatusMsg(i18n("Running wizard"));
    enableMenus(false);
  
    archiveoperation = new CArchiveOperationWizard(archiveobj, progress, tmpdir);
    connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
    archiveoperation->launchWizard(view->currentItem());
  }
}

/** we recive some informations through the standard output of the process */
void KarchiveurApp::recueSdtOut(KProcess *, char *buffer, int length){
  kdDebug()<<QString("KarchiveurApp::recueSdtOut")<<endl;
	if(operation==SCRIPT)
	{
		int i=0;
		if(archivechoice->getKindOfCompressor()==ZIP)
		{
			counter++;
			if(counter%2==0)
				progress->setProgress(progress->progress()+ 1 );
		}
		//A little animation for waiting gzip and bzip2 files to uncompress
		else if((archivechoice->getKindOfCompressor()==GZIP)||(archivechoice->getKindOfCompressor()==BZIP2))
		{
			counter++;
			fwrite(buffer,sizeof(char),length,fichierIO);
			QString s=i18n("Running compressor...");
  		switch(counter/4%8)
  		{
  			case 0:
  				s+="-";
  				break;
  			case 1:
  				s+="/";
  				break;
  			case 2:
  				s+="|";
  				break;
  			case 3:
  				s+="\\";
  				break;
  			case 4:
  				s+="-";
  				break;
  			case 5:
  				s+="/";
  				break;
  			case 6:
  				s+="|";
  				break;
  			case 7:
  				s+="\\";
  				break;
   		}
  		slotStatusMsg(s);
 		}
 		//For all other formats, use the progress bar
 		else
 		{
			QString inter;
			inter=buffer;
			inter=inter.left(length);

 			while((i=inter.find('\n',i+1))!=-1){
 				progress->setProgress(progress->progress()+1);
 			}
 		}
	}
}

/** Add some files to the archive
  * @param listfilestoadd : list of files to add
  * @param removefiles : remove or not those files from disk
  * @param actionfichiers : 0=mode append and replace files, 1=mode update files
  * @param relativePath : include only filenames, without their path */
void KarchiveurApp::addToArchive( QStringList listfilestoadd, bool removefiles, int actionfichiers, QString relativePath){
  if(archiveoperation!=NULL)
    delete archiveoperation;
  if(archiveobj==NULL)
  {
    led->setColor(QColor("orange"));
    return;
  }
  enableMenus(false);
  led->setColor(QColor("red"));
  archiveoperation = new CArchiveOperationAdd(archiveobj, progress);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString )), this, SLOT(slotMessageArchiveOperation(int, QString )));
  archiveoperation->addFilesToCurrentArchive( listfilestoadd,removefiles,actionfichiers,relativePath);
}

/** Remove the files @param filesToDelete from archive archive */
void KarchiveurApp::removeFilesFromArchive(const QString &, QStringList filesToDelete){
	if(operation==NOARCHIVE)
	{
		led->setColor(QColor("orange"));
		return;
	}

	led->setColor(QColor("red"));

	archiveobj->removeFilesFromArchive(filesToDelete);

	kdDebug()<<("Will display archiveliste")<<endl;
	displayArchiveContent();
	kdDebug()<<("removeFilesFromArchive:OK")<<endl;
	led->setColor(QColor("green"));
}

/** Open the archive @param archivename, which was selected
from the archive browser */
void KarchiveurApp::slotOpenFileFromArchiveBrowser(const QString& archivename){
  slotStatusMsg(i18n("Opening archive..."));
	led->setColor(QColor("red"));

	archivechoice->setArchiveName(navigateur->getChemin()+archivename);
  kdDebug()<<QString("slotOpenFileFromArchiveBrowser: opening %1").arg(navigateur->getChemin()+archivename)<<endl;

	displayArchiveContent();
}

/** Open the archive @param nameArchive, which was found
using the search tool */
void KarchiveurApp::slotOpenFileFromArchiveFinder(const QString& nameArchive){
	QString name;
	led->setColor(KLed::red);

	name=nameArchive;
	archivechoice->setFullPath(name.left(name.findRev('/')));
	navigateur->setPath(archivechoice->getAccessPath());
	archivechoice->setArchiveName(nameArchive);
	displayArchiveContent();
}

/** Open the archives @param names, which were sent
by the command line */
void KarchiveurApp::openFileFromCommandLine(QStringList names){
  led->setColor(QColor("red"));
  enableMenus(false);
  slotStatusMsg(i18n("Opening files..."));
  openFilesDropPaste(names);
}

/** Download a net file and returns its local filename*/
QString KarchiveurApp::openRemoteFile( const KURL& url)
{
  QString localfile;

  localfile=QDir::homeDirPath();
  localfile+="/";
  localfile+=url.fileName(true);
  kdDebug()<<QString("local file:%1*%2*").arg(localfile).arg(url.prettyURL())<<endl;
  if ( !url.isValid() )
  {
      KMessageBox::error(this, i18n("Bad URL!"));
      led->setColor(QColor("orange"));
      return NULL;
  }

  if (KIO::NetAccess::download(url, localfile, this) == false)
  {
    // this needs to be handled within KIO sooooon
    KMessageBox::error(this, i18n("Cannot download file!"));
    led->setColor(QColor("orange"));
    return NULL;
  }

  return localfile;
}

/** Displays in the status bar the size of the archive */
void KarchiveurApp::displayMessageArchiveSize(){
  QString message;
  QString filesselected=i18n(" files selected");
  if(selectedfilesStatusbarMessage.nbselectedfiles==1)
    filesselected=i18n(" file selected");

  message=IDS_STATUS_DEFAULT;
  if(archivechoice->getArchiveName()!=NULL)
    message=QString("%1  -  %2 %3").arg(IDS_STATUS_DEFAULT)
                                   .arg(archiveSize(archivechoice->getArchiveName()))
                                   .arg(i18n("bytes"));
  if(selectedfilesStatusbarMessage.nbselectedfiles>0)
  {
    message+=QString(" - %1 %2 %3 %4") .arg( selectedfilesStatusbarMessage.nbselectedfiles)
                                       .arg(filesselected)
                                       .arg(selectedfilesStatusbarMessage.sizeselectedfiles)
                                       .arg(i18n("bytes"));
    editCut->setEnabled( true );
    editCopy->setEnabled( true );
    archiveViewSelection->setEnabled( true );
    archiveDeleteSelection->setEnabled(true);
    archiveDeleteToTrashSelection->setEnabled(true);
  }
  else
  {
    archiveViewSelection->setEnabled( false );
    archiveDeleteSelection->setEnabled(false);
    archiveDeleteToTrashSelection->setEnabled(false);
    editCut->setEnabled( false );
    editCopy->setEnabled( false );
  }
  
  slotStatusMsg(message);
}

/** A drop will occur, so we have to allow it */
void KarchiveurApp::dragEnterEvent(QDragEnterEvent* e){
  e->accept(QUriDrag::canDecode(e));
}

/** someone has dropped something ! */
void KarchiveurApp::dropEvent( QDropEvent *drop){
  QStringList strliste;

	QUriDrag::decodeToUnicodeUris( drop, strliste);

  openFilesDropPaste(strliste);

}

/** Receive a message when the user try to
	* drag something. It extract the selected files,
	* and then transmit their path to the target
	* application
 	*/
void KarchiveurApp::slotDoDragAndDrop(){
  QListViewItem* vi;
  QStrList urlstotransmitt;
  QString filetoextract;
  QString futuredirectory;
  QString tmp=NULL;
  int k;


  if(operation==NOARCHIVE)
  	return;

 	//We will extract the archive to a temp folder, so let's clean it
 	process->clearArguments();
 	*process << "rm" << "-fR" <<tmpdir;
 	process->start(KProcess::Block);
 	process->clearArguments();
 	*process << "mkdir" << tmpdir;
 	process->start(KProcess::Block);

	//Extract only the selected files
	archiveobj->extractArchive(tmpdir, EXTRACT_SELECTED_FILES,tmp);

 	vi=view->firstChild();
 	for(k=0;k<view->childCount();k++)
 	{
 		if((vi->isSelected())&&(strcmp(vi->text(0),"..")!=0)&&futuredirectory!=vi->text(6))
 		{
 			futuredirectory=vi->text(6);
 			filetoextract+=vi->text(6);
			if(vi->text(6).isEmpty()||vi->text(6)=="/")
	 			filetoextract+=vi->text(0);
 			filetoextract=filetoextract.right(filetoextract.length()-1);
 			filetoextract=tmpdir+filetoextract;
			kdDebug()<<QString("*Drag of:%1*").arg(filetoextract)<<endl;
 			urlstotransmitt.append(filetoextract);
 			filetoextract="";
 		}
 		vi=vi->nextSibling();
 	}
	QDragObject *d = new QUriDrag( urlstotransmitt, this );
	d->dragCopy();
}

/** The contextual menu has been activated */
void KarchiveurApp::slotContextualMenuActivated(int i){
	switch(contextualMenu->indexOf(i))
	{
		case 0:
			slotExtractArchive();
			break;
		case 2:
			slotCopy();
			break;
		case 3:
			slotPaste();
			break;
		case 4:
			slotRemoveToTrashSelectedFiles();
			break;
		case 5:
			slotRemoveSelectedFiles();
			break;
		case 7:
			slotDisplayFile();
			break;
		case 8:
			slotDisplayFileProprieties();
			break;
	}
}

/* The user right-clicked on a file in the listview
	* so we'll display a contextual menu */
void KarchiveurApp::slotShowContextualMenu ( QListViewItem *l, const QPoint &pos, int ){
  if(archiveobj==NULL)
    return;

  QListViewItem * item = view->firstChild();
  selectedfilesStatusbarMessage.nbselectedfiles=0;
  selectedfilesStatusbarMessage.sizeselectedfiles=0;
  
  while( item ) {
    if(item->isSelected())
    {
      selectedfilesStatusbarMessage.nbselectedfiles++;
      selectedfilesStatusbarMessage.sizeselectedfiles+=item->text(1).toInt();
    }
      item = item->nextSibling();
  }
  displayMessageArchiveSize();
      

////	contextualMenu->exec(pos);
 QPopupMenu *pop = (QPopupMenu *)factory()->container("popup_list", this);
 pop->popup(pos);
}

/** Display in a dialog box some informations
concerning the selected file */
void KarchiveurApp::slotDisplayFileProprieties(){
  CProprietes propr;
  QListViewItem* l;
  int kindofCompressor;
  
  l=view->currentItem();
  archivechoice->findKindOfCompressor();
  kindofCompressor=archivechoice->getKindOfCompressor();

  propr.setFileName(l->text(0));
  propr.setFilePath(l->text(6));
  propr.setSize(l->text(1));
  if((kindofCompressor==TAR)||(kindofCompressor==TARGZ)||(kindofCompressor==TARBZ2))
          propr.setPermission(l->text(5));
  
  propr.exec();
}

void KarchiveurApp::keyPressEvent ( QKeyEvent * e )
{
  if(e->key()==Qt::Key_Control)
    controlkeypressed=true;
}

void KarchiveurApp::keyReleaseEvent ( QKeyEvent * e )
{
  if(e->key()==Qt::Key_Control)
    controlkeypressed=false;
}

/** The mouse button was clicked on the listview */
void KarchiveurApp::slotMouseClick(QListViewItem* li){
  
  if(operation==READING)
  {
          if(KMessageBox::questionYesNo(this, i18n("Stop opening of current archive?"), i18n("KArchiver")) == KMessageBox::Yes)
          {
                  archiveobj->stopReading();
                  if((operation==STANDBY)&&(archiveobj!=NULL))
                    archivechoice->setName(archiveobj->getArchiveName());//restore name of previous valide archive,which has been stored in CArchiveChoice
                  else
                  {
                    archiveobj=NULL;
                    operation=NOARCHIVE;
                  }
                  return;
          }
  }
  else if(operation==EXTRACTING)
  {
          if(KMessageBox::questionYesNo(this, i18n("Stop extracting current archive?"), i18n("KArchiver")) == KMessageBox::Yes)
          {
                  archiveobj->stopExtracting();
                  slotMessageArchiveOperation(CANNOT_PERFORM_OPERATION, i18n("Extraction stopped"));
          }
  }
  else if( (operation==STANDBY) && (li!=0) )
  {
    if(supportDisplayByDirs() && viewbydirectories && archiveobj->isDirectory(li->text(0)) && (!controlkeypressed))
    {
      selectedfilesStatusbarMessage.nbselectedfiles=0;
      selectedfilesStatusbarMessage.sizeselectedfiles=0;
      archiveobj->displayArchiveContent(li->text(0), li->text(6));//Display content of selected directory
    }
    else if((strcmp(li->text(0),"..")!=0) && (strcmp(li->text(1),".")!=0))//not a directory
    {
      QListViewItem * item = view->firstChild();
      selectedfilesStatusbarMessage.nbselectedfiles=0;
      selectedfilesStatusbarMessage.sizeselectedfiles=0;
      
      while( item ) {
        if(item->isSelected())
        {
          selectedfilesStatusbarMessage.nbselectedfiles++;
          selectedfilesStatusbarMessage.sizeselectedfiles+=item->text(1).toInt();
        }
         item = item->nextSibling();
      }
    }
    displayMessageArchiveSize();

  }
}

/** The user selected paste */
void KarchiveurApp::slotPaste(){
	QStringList urls;

	if ( QUriDrag::decodeToUnicodeUris(QApplication::clipboard()->data(), urls) )
		openFilesDropPaste(urls);
}

/** Some files were dropped or pasted; they have
been decoded in strliste, so just open them ! */
void KarchiveurApp::openFilesDropPaste( QStringList strliste )
{
  archiveoperation = new  CArchiveOperationProcessInputFiles( archiveobj,  progress, preferedArchiveType, includewholepath,  ADD_AND_REPLACE_FILES );
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  enableMenus(false);
  archiveoperation->processPastedFiles(strliste, viewbydirectories);
}

/** The user try to copy a karchiveur's file to
the clipboard */
void KarchiveurApp::slotCopy(){
  QListViewItem* vi;
  QStrList urlstotransmitt;
  QString filetoextract;
  QString futuredirectory;
  QString tmp=NULL;
  int k;

  if(operation==NOARCHIVE)
    return;

 	//We will extract the archive to a temp folder, so let's clean it
 	process->clearArguments();
 	*process << "rm" << "-fR" <<tmpdir;
 	process->start(KProcess::Block);
 	process->clearArguments();
 	*process << "mkdir" << tmpdir;
 	process->start(KProcess::Block);

	//Extract only the selected files
	archiveobj->extractArchive(tmpdir, EXTRACT_SELECTED_FILES,tmp);

 	vi=view->firstChild();
 	for(k=0;k<view->childCount();k++)
 	{
 		if((vi->isSelected())&&(strcmp(vi->text(0),"..")!=0)&&futuredirectory!=vi->text(6))
 		{
 			futuredirectory=vi->text(6);
 			filetoextract+=vi->text(6);
			if(vi->text(6).isEmpty()||vi->text(6).isNull()||vi->text(6)==" ")
	 			filetoextract+=vi->text(0);
 			filetoextract=filetoextract.right(filetoextract.length()-1);
 			filetoextract=tmpdir+filetoextract;
//	 			filetoextract+=vi->text(0);
 			urlstotransmitt.append(filetoextract);
			kdDebug()<<QString("In copy:addURL:%1*").arg(filetoextract)<<endl;
 			filetoextract="";
 		}
 		vi=vi->nextSibling();
 	}

	kapp->clipboard()->setData( new QUriDrag( urlstotransmitt, this ));

}

/** extract the selected files to trash and then
remove them from the archive */
void KarchiveurApp::slotRemoveToTrashSelectedFiles(){
  slotStatusMsg(i18n("Move all selected files to trash..."));

	if(operation!=NOARCHIVE)
	{
		int i;
		int k;
		QListViewItem *li=view->firstChild();
		QString txt;
		QString repTrash;
		KConfig conf("kdeglobals"); //To get the "trash" directory

		if(li==0)
		{
                  led->setColor(QColor("orange"));
                  slotStatusMsg(i18n("First select a file"));
		  return;
		}

		led->setColor(QColor("red"));
		repTrash=QDir::homeDirPath();
		repTrash+="/Desktop/Trash";
		conf.setGroup("Paths");
		repTrash=conf.readPathEntry("Trash",repTrash);
		k=view->childCount();

		for(i=0;i<k;i++)
		{
			if(view->isSelected(li))
			{
				txt=li->text(6);
				txt=txt.remove(0,1);
				txt+=li->text(0);
				archiveobj->extractArchive(repTrash, 9,txt);
			}
			if(i!=k)
				li=li->nextSibling();
		}
	}

	slotRemoveSelectedFiles();

	led->setColor(QColor("green"));
  slotStatusMsg(IDS_STATUS_DEFAULT);
}

/** the user has "cutted" some files in the current
archive */
void KarchiveurApp::slotCutt(){
	QListViewItem* vi;
	QStrList urlstotransmitt;
	QString filetoextract;
	QString futuredirectory;
	QString tmp=NULL;
  int k;

  kdDebug()<<("IN:slotCutt\n");
  if(operation==NOARCHIVE)
  	return;

 	//We will extract the archive to a temp folder, so let's clean it
 	process->clearArguments();
 	*process << "rm" << "-fR" <<tmpdir;
 	process->start(KProcess::Block);
 	process->clearArguments();
 	*process << "mkdir" << tmpdir;
 	process->start(KProcess::Block);

	//Extract only the selected files
	archiveobj->extractArchive(tmpdir, EXTRACT_SELECTED_FILES,tmp);

 	vi=view->firstChild();
 	for(k=0;k<view->childCount();k++)
 	{
 		if((vi->isSelected())&&(strcmp(vi->text(0),"..")!=0)&&futuredirectory!=vi->text(6))
 		{
 			futuredirectory=vi->text(6);
 			filetoextract+=vi->text(6);
			if(vi->text(6).isEmpty()||vi->text(6).isNull()||vi->text(6)==" ")
	 			filetoextract+=vi->text(0);
 			filetoextract=filetoextract.right(filetoextract.length()-1);
 			filetoextract=tmpdir+filetoextract;
//	 			filetoextract+=vi->text(0);
 			urlstotransmitt.append(filetoextract);
			kdDebug()<<QString("In cutt:addURL:%1*").arg(filetoextract)<<endl;
 			filetoextract="";
 		}
 		vi=vi->nextSibling();
 	}
	QDragObject *d = new QUriDrag( urlstotransmitt, this );

	kapp->clipboard()->setData( d);

	KMessageBox::information(this, i18n("Paste these files in any application NOW."), "karchiveur");
	slotRemoveSelectedFiles();

}

/** select all the files in the current archive */
void KarchiveurApp::slotSelectAll(){
	view->selectAll(true);
}

/** unselect all files in the current archive */
void KarchiveurApp::slotUnselectAll(){
	view->selectAll(false);
}

/** find in this archive all items matching the
name name */
void KarchiveurApp::slotFindInThisArchive( QString name ){
	QListViewItem *vi;
	int k;

	kdDebug()<<QString("Find:%1*%2*").arg(name).arg(indexfind)<<endl;
 	vi=view->firstChild();
 	for(k=0;k<view->childCount();k++)
 	{
 		if((k>indexfind)&&(vi->text(0).contains(name)!=0))
 		{
			view->setCurrentItem(vi);
			vi->setSelected(true);
			indexfind=k;
			return;
 		}
 		vi=vi->nextSibling();
 	}

 	KMessageBox::information(this, i18n("No more occurrences found!"), "karchiveur");
 	indexfind=-1;
}
/** the user closed the find in this archive tool */
void KarchiveurApp::slotCloseFind(){
	find->hide();
	indexfind=-1;
}

/** Use the find in this archive tool */
void KarchiveurApp::slotUseFind(){
	find->show();
	indexfind=-1;
}

void KarchiveurApp::slotSetPassword(){
	 QCString pass=archiveobj->getPassword();
     int rc = KPasswordDialog::getPassword(pass,i18n("Please enter the password of this archive"));
     if(rc==KPasswordDialog::Accepted)
        archiveobj->setPassword(pass);
}

/** a process in CArchive has ended. So make
the necessary with the operation variable */
void KarchiveurApp::slotEndProcess(){
  kdDebug()<<QString("KarchiveurApp::slotEndProcess: operation= %1").arg(operation)<<endl;
  operation=STANDBY;
}

/** Initialize the actions and the GUI */
void KarchiveurApp::initActions(){
  fileNewWindow=new KAction( i18n("New &Window"), kapp->miniIcon(), 0, this, SLOT(slotFileNewWindow()), actionCollection(),"file_new_window");
  fileNewArchive=new KAction( i18n("Create &new Archive"), "karchiver", 0, this, SLOT(slotFileNew()), actionCollection(),"file_new_archive" );
  fileOpenArchive=new KAction( i18n("&Open..."), "fileopen", 0, this, SLOT(slotFileOpen()), actionCollection(),"file_open_archive" );
  fileSearchArchive=new KAction( i18n("&Search for Archives"), "find", 0, this, SLOT(slotFileSearch()), actionCollection(),"file_search_archive" );
  fileCryptArchive=new KAction( i18n("&Encrypt Current Archive"), "encrypted", 0, this, SLOT(slotCryptArchive()), actionCollection(), "file_crypt_archive" );
  fileMailArchive=new KAction( i18n("&Mail To..."), "mail_generic", 0, this, SLOT(slotMailArchive()), actionCollection(), "file_mail_archive" );
  fileOpenRecent = KStdAction::openRecent( this, SLOT(slotFileOpenRecent(const KURL&)), actionCollection(), "file_open_recent_kar");
  fileConvertArchive=new KAction( i18n("Convert &As"), "filesave", 0, this, SLOT(slotFileSaveAs()), actionCollection(),"file_convert_archive" );
  fileMakeSfxArchive=new KAction( i18n("Make exe archive"), "exec", 0, this, SLOT(slotMakeSfxArchive()), actionCollection(),"file_sfx_archive" );
  fileInfosArchive=new KAction( i18n("&View Archive Info"), "documentinfo", 0, this, SLOT( slotDisplayArchiveInfos()), actionCollection(),"file_infos_archive" );
  filePrint = KStdAction::print( this, SLOT(slotFilePrint()), actionCollection(), "file_print_kar");
  fileCloseArchive=new KAction( i18n("C&lose Window"), "fileclose", 0, this, SLOT( slotFileClose()), actionCollection(),"file_close_archive" );
  fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());

  ///////////////////////////////////////////////////////////////////
  // menuBar entry archiveMenu
  archiveExtract=new CAnimatedAction(i18n("E&xtract To"), "extract", 0, this, SLOT( slotExtractArchive()), actionCollection(),"archive_extract" );
  archiveAdd=new KAction(i18n("&Add Files"), "compress", 0, this, SLOT( slotAddFiles()), actionCollection(),"archive_add" );
  archiveView=new KAction(i18n("&View File"), "exec", 0, this, SLOT( slotDisplayFile()), actionCollection(),"archive_view" );
  archiveViewSelection=new KAction(i18n("View &Selected"), "view", 0, this, SLOT( slotDisplaySelectedFiles()), actionCollection(),"archive_view_selection" );
  archiveDeleteSelection=new KAction(i18n("&Delete Files"), "editcut", 0, this, SLOT( slotRemoveWithUndoSelectedFiles()), actionCollection(),"archive_delete_selection" );
  archiveSetPassword=new KAction(i18n("Set &Password"), "encrypted", 0, this, SLOT( slotSetPassword()), actionCollection(),"archive_password" );
  archiveDeleteToTrashSelection=new KAction(i18n("&Move to Trash"), "edittrash", 0, this, SLOT( slotRemoveToTrashSelectedFiles()), actionCollection(),"archive_delete_to_trash_selection" );
  archiveWizard=new KAction(i18n("Launch the &Wizard"), "wizard", 0, this, SLOT( slotWizard()), actionCollection(),"archive_wizard" );
  archiveProprieties=new KAction(i18n("&Properties"), 0, this, SLOT( slotDisplayFileProprieties()), actionCollection(),"archive_proprieties" );
  archiveScan=new KAction( i18n("&Scan for viruses"), "find", 0, this, SLOT( slotScanArchive()), actionCollection(),"archive_scan" );
  archiveTest=new KAction(i18n("&Test and repair Archive"), "documentinfo", 0, this, SLOT( slotTestArchive()), actionCollection(),"archive_test" );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry editMenu
  editUndo=KStdAction::undo( this, SLOT( slotUndo()), actionCollection(),"edit_undo_kar" );
  editRedo=KStdAction::redo( this, SLOT( slotRedo()), actionCollection(),"edit_redo_kar" );
  editCut=KStdAction::cut( this, SLOT( slotCutt()), actionCollection(),"edit_cut_kar" );
  editCopy=KStdAction::copy( this, SLOT( slotCopy()), actionCollection(),"edit_copy_kar" );
  editPaste=KStdAction::paste( this, SLOT( slotPaste()), actionCollection(),"edit_paste_kar" );
  editSelectAll=KStdAction::selectAll( this, SLOT( slotSelectAll()), actionCollection(),"edit_select_all_kar" );
  editUnselectAll=new KAction(i18n("Select &None"), 0, this, SLOT( slotUnselectAll()), actionCollection(),"edit_unselect_all" );
  editFind=new KAction(i18n("&Find..."), "find", 0, this, SLOT( slotUseFind()), actionCollection(),"edit_find_kar" );

  ///////////////////////////////////////////////////////////////////
  // menuBar entry diskMenu
	diskSplit=new KAction(i18n("&Split/Unsplit"), 0, this, SLOT(slotDiskCutUncut()), actionCollection(), "disk_split");

  ///////////////////////////////////////////////////////////////////
  // menuBar entry viewMenu
  viewToolbar=new KToggleAction(i18n("&Toolbar"), 0, this, SLOT(slotViewToolBar()), actionCollection(), "view_toolbar" );
  viewStatusbar=new KToggleAction(i18n("&Statusbar"), 0, this, SLOT(slotViewStatusBar()), actionCollection(), "view_statusbar" );
  viewArchiveBrowser=new KToggleAction(i18n("&File Browser"), 0, this, SLOT(slotViewButineur()), actionCollection(), "view_archive_browser" );
  KStdAction::configureToolbars(this, SLOT(slotEditToolbars()), actionCollection(), "view_edit_toolbar" );
  viewUpdate=new KAction(i18n("Check for &Updates"), kapp->miniIcon(), 0, this, SLOT(slotCheckUpdate()), actionCollection(), "view_update" );
  viewPreferences=new KAction(i18n("&Configure KArchiver"), "configure", 0, this, SLOT(slotSetOptions()), actionCollection(), "view_preferences" );

  fileNewWindow->setToolTip(i18n("Open a new application window"));
  fileNewArchive->setToolTip(i18n("Create a new archive"));
  fileOpenArchive->setToolTip(i18n("Open an archive"));
  fileSearchArchive->setToolTip(i18n("Search for an existing archive"));
  fileConvertArchive->setToolTip(i18n("Convert the current archive format to..."));
  fileMakeSfxArchive->setToolTip(i18n("Make a multi-platform self-extractible archive"));
  archiveDeleteSelection->setToolTip(i18n("Delete the selected files"));
  archiveSetPassword->setToolTip(i18n("Set a password for this archive"));
  archiveExtract->setToolTip(i18n("Extract to..."));
  archiveAdd->setToolTip(i18n("Add files to this archive"));
  archiveView->setToolTip(i18n("View the current file"));
  archiveViewSelection->setToolTip(i18n("View all selected files"));
  fileInfosArchive->setToolTip(i18n("Display info about the current archive"));
  archiveWizard->setToolTip(i18n("Launch a wizard to configure & compile this archive..."));
  diskSplit->setToolTip(i18n("Split archive into 1.4Mb blocks"));
  viewArchiveBrowser->setToolTip(i18n("Enable / disable the file browser"));
  viewPreferences->setToolTip(i18n("Modify the options of KArchiver"));
  editCut->setToolTip(i18n("Move the files that were copied to the clipboard"));
  editCopy->setToolTip(i18n("Extract selected files and copy them to the clipboard"));
  editPaste->setToolTip(i18n("Add the files that were copied to the clipboard"));
  editFind->setToolTip(i18n("Find a filename in this archive"));
  editUndo->setToolTip(i18n("Undo the last operation"));
  editRedo->setToolTip(i18n("Redo the last operation"));

  createGUI();
}

void KarchiveurApp::slotEditToolbars()
{
  KEditToolbar dlg(actionCollection());
  if (dlg.exec())
    createGUI();
}

/** Extract the archives @param archives to the directory @param extractdir.
	If @param promptdir is true, ignore pervious parameter and show the extract dialog,
        to let the user	choosing the right extract dir
        If @param promptdir == false && if @param  chooseExtractDirFromArchive == true && the
        archive's files are not contained in a top subdir, create this topsubdir when extracting */
void KarchiveurApp::extractFromCommandLine( QStringList archives, bool promptdir, QCString extractdir, bool chooseExtractDirFromArchive )
{
  archiveoperation = new  CArchiveOperationExtractMultiFiles( archiveobj,  progress );
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("Extracting files..."));
  enableMenus(false);
  archiveoperation->extractMultiFiles(archives, promptdir, extractdir, chooseExtractDirFromArchive);
}

/** Check wether karchiver is lauched for the first time. If so, launch
a wizard to configure the main features */
void KarchiveurApp::checkFirstInstall(){
  int rep=1;
  QString arkpath, arkpath2, arkpath3;
  CFirstInstallWizard firstinstallwizard(this, "1st_install", true);

  config->setGroup("First Time Install");
  if( config->readBoolEntry("first run", true) )
  	if( firstinstallwizard.exec() )
  {
	  config->setGroup("First Time Install");
	  config->writeEntry("first run", false);

		config->setGroup("OPTIONS");
		if(firstinstallwizard.rb_wind_like->isChecked())
			config->writeEntry("SelectionMode",1);
		else
			config->writeEntry("SelectionMode",0);

		config->setGroup("LASTDIRS");
		if(firstinstallwizard.rb_extract_lastdir->isChecked())
			rep=0;
		else if(firstinstallwizard.rb_extract_homedir->isChecked())
			rep=1;
		else if(firstinstallwizard.rb_extract_shelldir->isChecked())
			rep=2;
		config->writeEntry("OptionExtraction",rep);

		if(firstinstallwizard.rb_open_lastdir->isChecked())
			rep=0;
		else if(firstinstallwizard.rb_open_homedir->isChecked())
			rep=1;
		else if(firstinstallwizard.rb_open_shelldir->isChecked())
			rep=2;
		config->writeEntry("OptionOuverture",1);

		if(firstinstallwizard.cb_use_karchiver->isChecked())
		{
			arkpath=KGlobal::dirs()->findResource("data", "konqueror/servicemenus/arkservicemenu.desktop");
			arkpath3=KGlobal::dirs()->findResourceDir("data", "konqueror/servicemenus/arkservicemenu.desktop");
			arkpath2=KGlobal::dirs()->findResource("services", "ark_part.desktop");
			kdDebug()<<QString("Will gzip %1\n").arg(arkpath2);
			KShellProcess proc;

      proc.clearArguments();
      proc << "kdesu";
      proc << "-c";
      proc << "'";
      if(!arkpath.isEmpty()) //for backward compatibility
      {
        proc << "mkdir";
        proc << arkpath3+"konqueror/servicemenus/ark";
        proc << ";";
        proc << "mv";
        proc << arkpath;
        proc << arkpath3+"konqueror/servicemenus/ark";
        proc << ";";
      }
      proc << "gzip" << arkpath2;
      proc << "'";
      proc.start(KProcess::Block);

      KMessageBox::information(this, i18n("Has gzipped ark files:\n")+arkpath+"\n"+arkpath2, i18n("KArchiver setup"));
		}
	config->sync();
	readOptions();
	}
}

/** Automaticaly create a compressed file containing @param files
  * @param preferedtype is an archive extension (.zip...) that indicated
  * which type of archive should be created. If @param preferedtype is Null,
  * use KArchiver's default archive type
If there is a single file or directory, create an archive
named files.tar.gz. Otherwise, prompt the user for a name */
void KarchiveurApp::createArchiveFromCommandLine( QStringList files, QString preferedtype ){

  archiveoperation = new  CArchiveOperationProcessInputFiles( archiveobj,  progress, preferedArchiveType, includewholepath,  ADD_AND_REPLACE_FILES );
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  led->setColor(QColor("red"));
  slotStatusMsg(i18n("Creating archive..."));
  enableMenus(false);
  archiveoperation->createArchiveFromCommandLine(files, preferedtype);  
}

/** crypt current archive */
void KarchiveurApp::cryptArchive(){
	CGPG crypt;
kdDebug() << QString("Will crypt %1").arg(archivechoice->getArchiveName()) << endl;
	crypt.setArchiveToProcess( archivechoice->getArchiveName());
	crypt.cryptArchive();
}

/** crypt current archive using gpg */
void KarchiveurApp::slotCryptArchive(){
	cryptArchive();
}

void KarchiveurApp::slotMailArchive() {
  led->setColor(QColor("red"));
  enableMenus(false);
  archiveoperation = new CArchiveOperationMail(archiveobj, progress, tmpdir);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString )), this, SLOT(slotMessageArchiveOperation(int, QString )));
  archiveoperation->mailCurrentArchive();
}

/** enable or disable the KAction menus */
void KarchiveurApp::enableMenus(bool b)
{
  const bool enabledandselected =b  && (selectedfilesStatusbarMessage.nbselectedfiles>0);
  fileCryptArchive->setEnabled(b);
  fileMailArchive->setEnabled(b);
  fileConvertArchive->setEnabled(b);
  fileInfosArchive->setEnabled(b);
  filePrint->setEnabled(b);
  fileCloseArchive->setEnabled(b);
  fileMakeSfxArchive->setEnabled(b);

  archiveExtract->setEnabled(b);
  archiveAdd->setEnabled(b);
  archiveView->setEnabled(b);
  archiveSetPassword->setEnabled(b);
  if((archiveobj!=NULL)&&(!archiveobj->supportPassword()))
    archiveSetPassword->setEnabled(false);
  archiveViewSelection->setEnabled( enabledandselected );
  archiveDeleteSelection->setEnabled(enabledandselected);
  archiveDeleteToTrashSelection->setEnabled(enabledandselected);
  archiveWizard->setEnabled(b);
  archiveProprieties->setEnabled(b);
  archiveTest->setEnabled(b);
  archiveScan->setEnabled(b);

  editUndo->setEnabled(b && (!listUndo.isEmpty()) );
  editRedo->setEnabled(b && (!listRedo.isEmpty())) ;
  editCut->setEnabled(b && (selectedfilesStatusbarMessage.nbselectedfiles>0) );
  editCopy->setEnabled(b && (selectedfilesStatusbarMessage.nbselectedfiles>0) );
  editPaste->setEnabled(b);
  editSelectAll->setEnabled(b);
  editUnselectAll->setEnabled(b);
  editFind->setEnabled(b);
  if(viewbydirectories && supportDisplayByDirs() && b)
      cbdirectories->show();
  else
      cbdirectories->hide();
  
  if(archiveobj==NULL)
    b=true;

    fileNewArchive->setEnabled(b);
    fileOpenArchive->setEnabled(b);
    fileOpenRecent->setEnabled(b);
    diskSplit->setEnabled(b);

}

/** displays the configuration wizard(if 1st launch & bool=true) and the tip of the day*/
void KarchiveurApp::configurationWizardAndTip( bool use_configuration_wizard )
{
  if(use_configuration_wizard)
    checkFirstInstall();
  tipofday = new KTipofDay(this,"Tip_Of_Day_Dialog");
  if(displaytipofday)
    tipofday->show();
}

void KarchiveurApp::slotCheckUpdate()
{
  CUpdate update;
  update.checkUpdate();
}

/** Test the archive for errors */
void KarchiveurApp::slotTestArchive()
{
  archiveoperation = new CArchiveOperationTest(archiveobj, progress, tmpdir);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  archiveoperation->testArchive();
}

/** Scan archive for viruses */
void KarchiveurApp::slotScanArchive()
{
  archiveoperation = new CArchiveOperationScanForVirus(archiveobj, progress, tmpdir);
  connect(archiveoperation, SIGNAL(operationEnded(int, QString)), this, SLOT(slotMessageArchiveOperation(int , QString )));
  archiveoperation->scanArchive();
}

/** User wants to go to a subfolder */
void KarchiveurApp::slotComboDirectoriesActivated(int index){
  QString path;
  int i;

  if(!supportDisplayByDirs())
    return;

  for(i=0;i<=index;i++)
  {
    path+=cbdirectories->text(i);
    if(!path.endsWith("/"))
        path+="/";
  }
  path=path.left(path.length()-1);
  kdDebug()<<QString("Going to %1, basepath=%2").arg(cbdirectories->text(index)).arg(path)<<endl;
  selectedfilesStatusbarMessage.nbselectedfiles=0;
  selectedfilesStatusbarMessage.sizeselectedfiles=0;
  displayMessageArchiveSize();
  archiveobj->displayArchiveContent(cbdirectories->text(index),path);
}

/** return true if KIO supports current kind of archive */
bool KarchiveurApp::supportDisplayByDirs(){
  if((archiveobj==NULL)||(cbdirectories==NULL))
    return false;
  
  return archiveobj->supportDisplayByDirs();
}

/** Messages from CArchiveOperation and childs */
void KarchiveurApp::slotMessageArchiveOperation(int message, QString messagetext)
{
  kdDebug()<<QString("KarchiveurApp::slotMessageArchiveOperation %1 %2").arg(message).arg(messagetext)<<endl;
  slotStatusMsg(messagetext);
  
  switch(message)
  {
    case CONVERSION_ACHIEVED:
    {
      slotStatusMsg(messagetext);
      led->setColor(QColor("green"));
      QString s=i18n("Delete : %1?").arg(archivechoice->getArchiveName());
      if(KMessageBox::questionYesNo(this, s, "karchiver")==1)
      {
        process->clearArguments();
        *process << "rm" << "-f" <<archivechoice->getArchiveName();
        process->start(KProcess::Block);
      }
      led->setColor(QColor("green"));
      
      archivechoice->setArchiveName(archiveoperation->getArchiveName());
      
      //displayArchiveContent();
      break;
    }
    case DISPLAY_ACHIEVED:
    {
      archiveobj=archiveoperation->getArchiveObject();//Fetch back the archive object, as a new one has been created in CArchiveOperation...
      archivechoice->setArchiveName(archiveobj->getArchiveName());
      connect(archiveobj, SIGNAL(archiveReadEnded()), this, SLOT(slotEndProcess()));
      connect(archiveobj, SIGNAL(sourceSoftwareSpotted()), this, SLOT(slotAskForCompilation()));
      archiveSetPassword->setEnabled(archiveobj->supportPassword());
      QFileInfo fiarchive(archiveobj->getArchiveName());
      navigateur->setPath(fiarchive.dirPath());
      led->setColor(QColor("green"));
      animation->stopAnimation();
      setCaption(fiarchive.fileName() + " "+i18n("in")+" "+fiarchive.dirPath());
      displayMessageArchiveSize();
      if(archiveobj->isSourceSoftware())
        slotAskForCompilation();
      break;
    }
    case MULTI_EXTRACTION_ACHIEVED:
    {
      slotFileQuit();
      break;
    }
    case SFX_ARCHIVE_CREATED:
    {
      KMessageBox::information(this, messagetext, "Karchiver Sfx");
      led->setColor(QColor("green"));
      break;
    }
    case EXTRACT_ACHIEVED:
    case VIEW_FILES_ACHIEVED:
    case ADD_FILES_ACHIEVED:
    case WIZARD_ACHIEVED:
    case TEST_ACHIEVED:
    {
      led->setColor(QColor("green"));
      break;
    }
    case SCAN_ACHIEVED:
    {
      KMessageBox::information(this, messagetext, i18n("Scan result"));
      led->setColor(QColor("green"));
      break;
    }
    case SPLIT_ACHIEVED:
    {
      KMessageBox::information(this,messagetext,"KArchiver");
      led->setColor(QColor("green"));
      break;
    }
    case UNSPLIT_ACHIEVED:
    {
      archivechoice->setArchiveName(messagetext);
      break;
    }
    case PASTE_FILES_ACHIEVED:
    case ARCHIVE_CREATION_ACHIEVED:
    {
      archiveobj=archiveoperation->getArchiveObject();//Fetch back the archive object, as a new one has been created in CArchiveOperation...
      kdDebug()<<QString("Setting archivename to %1").arg(archiveobj->getArchiveName())<<endl;
      archivechoice->setArchiveName(archiveobj->getArchiveName());
      break;
    }
    case CREATE_ARCHIVE_CMD_LINE_ACHIEVED:
    {
      led->setColor(QColor("green"));
      slotStatusMsg(messagetext);
      kdDebug()<<QString("Will leave the application")<<endl;
      delete archiveoperation;
      archiveoperation=NULL;
      close();
      break;
    }
    case ADD_FILES_ANOTHER_ARCHIVE_ACHIEVED:
    {
      slotStatusMsg(i18n("Opening a new application window..."));
      kdDebug()<<QString("in ADD_FILES_ANOTHER_ARCHIVE_ACHIEVED")<<endl;
      if(archiveobj!=NULL)
      {
        delete archiveobj;
        kdDebug()<<"Object deleted"<<endl;
        archiveobj=NULL;
      }
      kdDebug()<<QString("Archive name=%1").arg(archivechoice->getArchiveName())<<endl;
      /*KarchiveurApp *new_window= new KarchiveurApp();
      new_window->show();
      QStringList archname;
      archname.append(messagetext);
      kdDebug()<<QString("Displaying %1 in a new window and also %2").arg(messagetext).arg(archivechoice->getArchiveName())<<endl;
      new_window->openFileFromCommandLine(archname);*/
      kdDebug()<<QString("Starting karchiver with argument %1").arg(messagetext)<<endl;
      KProcess proc;
      proc << "karchiver" << messagetext;
      proc.start(KProcess::DontCare);
      break;
    }
    case OPEN_ARCHIVE_IN_NEW_WINDOW:
    {
      slotStatusMsg(i18n("Opening a new application window..."));

      KarchiveurApp *new_window= new KarchiveurApp();
      new_window->show();
      QStringList archname;
      archname.append(messagetext);
      kdDebug()<<QString("Displaying %1 in a new window").arg(messagetext)<<endl;
      new_window->openFileFromCommandLine(archname);
      break;
    }
    default:
    case CANNOT_PERFORM_OPERATION:
    {
      led->setColor(QColor("orange"));
      animation->stopAnimation();
      slotStatusMsg(messagetext);
      break;
    }
  }
  operation=STANDBY;
  delete archiveoperation;
  archiveoperation=NULL;
  progress->reset();
  if(archiveobj!=NULL)
    enableMenus(true);
  else
  {
    operation=NOARCHIVE;
    enableMenus(false);
  }
  
  if((message==ARCHIVE_CREATION_ACHIEVED)||(message==ADD_FILES_ACHIEVED)||(message==ADD_FILES_ANOTHER_ARCHIVE_ACHIEVED)||(message==CONVERSION_ACHIEVED)||(message==UNSPLIT_ACHIEVED))
  {
    displayArchiveContent();
  }
}

/** Ask if KArchiver has to compile this autoconf-based archived software */
void KarchiveurApp::slotAskForCompilation()
{
  if(KMessageBox::questionYesNo(this, i18n("Current archive seems to be an autoconf-based source software. KArchiver can compile and install it for you. Do it now ?"), i18n("Compile this software ?"), KStdGuiItem::yes(), KStdGuiItem::no(), i18n("Don't look for source softwares again"))==KMessageBox::Yes)
    slotWizard();
}
#include "karchiveur.moc"
