/*
 * previewdialog.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <gtkmm/stock.h>
#include <gtkmm/toolbar.h>
#include "previewdialog.hh"
#include "macros.h"
#include "dialogs.hh"
#include "utils.hh"
#include <cassert>

PreviewDialog::PreviewDialog(Print& print) :
  _print(print),
  adj(1, 1, print.get_pages()),
  pages(adj),
  label(""),
  back(Gtk::Stock::GO_BACK),
  forward(Gtk::Stock::GO_FORWARD),
  rewind(Gtk::Stock::GOTO_FIRST),
  ff(Gtk::Stock::GOTO_LAST) {
  dialog.set_title(_("Print preview"));

  label.set_text(Utils::substitute(_("of %i"), print.get_pages()));

  Gtk::VBox *box = dialog.get_vbox();

  rewind.set_relief(Gtk::ReliefStyle(Gtk::RELIEF_NONE));
  back.set_relief(Gtk::ReliefStyle(Gtk::RELIEF_NONE));
  forward.set_relief(Gtk::ReliefStyle(Gtk::RELIEF_NONE));
  ff.set_relief(Gtk::ReliefStyle(Gtk::RELIEF_NONE));

  hbox.pack_start(rewind, false, false, 5);
  hbox.pack_start(back, false, false, 5);
  hbox.pack_start(forward, false, false, 5);
  hbox.pack_start(ff, false, false, 5);

  box->pack_start(hbox, false, false);
  hbox.pack_start(pages, false, false);
  hbox.pack_start(label, false, false, 20);
  box->pack_start(window, true, true);

  window.set_policy (Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);

  window.add(image);
  dialog.add_button(Gtk::Stock::PRINT, Gtk::RESPONSE_OK);
  dialog.add_button(Gtk::Stock::CLOSE, Gtk::RESPONSE_CANCEL);

  _ok = load(1);

  Glib::RefPtr<Gdk::Screen> sc = Gdk::Screen::get_default();

  window.set_size_request(sc->get_width()/2, sc->get_height()/2);

  pages.signal_value_changed().connect(sigc::mem_fun(*this, &PreviewDialog::on_signal_value_changed));

  back.signal_clicked().connect(sigc::mem_fun(*this, &PreviewDialog::on_back_clicked));
  forward.signal_clicked().connect(sigc::mem_fun(*this, &PreviewDialog::on_forward_clicked));
  ff.signal_clicked().connect(sigc::mem_fun(*this, &PreviewDialog::on_ff_clicked));
  rewind.signal_clicked().connect(sigc::mem_fun(*this, &PreviewDialog::on_rewind_clicked));

  recalculate_gui();

  dialog.show_all();
}

PreviewDialog::~PreviewDialog() {

}

bool PreviewDialog::run() {
  return dialog.run() == Gtk::RESPONSE_OK ? true : false;
}

bool PreviewDialog::load(int x) {
  assert (x <= _print.get_pages());
  std::string out, err;
  if (!_print.get_page(x, out, err)) {
    katoob_error(err);
    return false;
  }
  // TODO: We should separate this into its own function oneday.
  GError *er = NULL;
  GdkPixbufLoader* loader = gdk_pixbuf_loader_new_with_type("png", &er);
  if (!loader) {
    std::string err(er->message);
    katoob_error(err);
    g_error_free(er);
    return false;
  }
  er = NULL;

  if (!gdk_pixbuf_loader_write(loader, reinterpret_cast<const guchar *>(out.c_str()), out.size(), &er)) {
    std::string err(er->message);
    katoob_error(err);
    g_error_free(er);
    gdk_pixbuf_loader_close(loader, NULL);
    g_object_unref (loader);
    return false;
  }
  er = NULL;
  if (!gdk_pixbuf_loader_close(loader, &er)) {
    std::string err(er->message);
    g_object_unref (loader);
    katoob_error(err);
    g_error_free(er);
    return false;
  }

  GdkPixbuf *pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
  if (!pixbuf) {
    g_object_unref (loader);
    katoob_error(_("Failed to display image"));
    return false;
  }
  g_object_ref(pixbuf);
  g_object_unref (loader);
  pix = Glib::wrap(pixbuf);
  image.set(pix);
  return true;
}

void PreviewDialog::on_signal_value_changed() {
  load(pages.get_value_as_int());
  recalculate_gui();
}

void PreviewDialog::on_back_clicked() {
  pages.set_value(pages.get_value_as_int()-1);
}

void PreviewDialog::on_rewind_clicked() {
  pages.set_value(1);
}

void PreviewDialog::on_forward_clicked() {
  pages.set_value(pages.get_value_as_int()+1);
}

void PreviewDialog::on_ff_clicked() {
  double s, e;
  pages.get_range(s, e);
  pages.set_value(e);
}

void PreviewDialog::recalculate_gui() {
  int c;
  double s, e;

  pages.get_range(s, e);
  c = pages.get_value_as_int();
  back.set_sensitive(!(c == 1));
  rewind.set_sensitive(!(c == 1));
  forward.set_sensitive(!(c == e));
  ff.set_sensitive(!(c == e));
}

bool PreviewDialog::ok() {
  return _ok;
}
