/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_proxy.h"
#include <glib/gi18n.h>

#include "kz-window.h"
#include "utils/gtk-utils.h"
#include "kazehakase.h"
#include "kz-profile.h"
#include "kz-proxy-item.h"

#include <stdio.h>

#define DATA_KEY "KzPrefsProxy::info"

enum
{
	COLUMN_TERMINATOR = -1,
	COLUMN_SELECTED,
	COLUMN_NAME,
	COLUMN_HTTP_HOST,
	COLUMN_HTTP_PORT,
	COLUMN_HTTPS_HOST,
	COLUMN_HTTPS_PORT,
	COLUMN_FTP_HOST,
	COLUMN_FTP_PORT,
	COLUMN_USE_SAME_PROXY,
	COLUMN_NO_PROXIES_ON,
	COLUMN_EDITABLE,
	N_COLUMNS
};

typedef struct _KzPrefsProxy KzPrefsProxy;

static GtkWidget  *prefs_proxy_create             (void);
static void        prefs_proxy_response           (GtkWidget *widget,
						   gint       response);
static void        prefs_proxy_delete_proxy_list  (void);
static void        prefs_proxy_set_values         (KzPrefsProxy *prefsui);
static void        prefs_proxy_set_sensitive      (KzPrefsProxy *prefsui);

static KzPrefsWinPageEntry prefs_entry =
{
	/* .path          = */ N_("/Proxy"),
	/* .priority_hint = */ 0,
	/* .ui_level      = */ KZ_UI_LEVEL_ALL,
	/* .create        = */ prefs_proxy_create,
	/* .response      = */ prefs_proxy_response,
};

struct _KzPrefsProxy
{
	GtkWidget     *main_vbox,
		      *proxy_vbox,
		      *tree_view,
		      *name,
		      *use_proxy,
		      *http_host, *http_port,
		      *https_host, *https_port,
		      *ftp_host, *ftp_port,
		      *use_same_proxy,
		      *no_proxies_on,
		      *up, *down, *new, *add, *remove;
	GtkListStore  *store;
	GList	      *proxy_list;
	gboolean       changed;     /* proxy selection changed */
	gboolean       edited;      /* proxy setting edited */
};

static GList *proxy_list;

static void
prefs_proxy_destroy (gpointer data)
{
	KzPrefsProxy *prefsui = data;

	prefs_proxy_delete_proxy_list();
	g_free(prefsui);
}

static void
prefs_proxy_delete_proxy_list()
{
	GList *node;
	for (node = proxy_list; node; node = node->next)
	{
		g_object_unref(node->data);
		node->data = NULL;
	}
	g_list_free(proxy_list);
	proxy_list = NULL;
}

static void
cb_up_button (GtkButton *button, KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, prev;
	GtkTreePath *treepath;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(prefsui->tree_view));
	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
		return;

	treepath = gtk_tree_model_get_path(model, &iter);
	if (!gtk_tree_path_prev(treepath)) goto ERROR;
	gtk_tree_model_get_iter(model, &prev, treepath);

	gtk_list_store_swap(prefsui->store, &iter, &prev);

	prefsui->edited = TRUE;
ERROR:
	gtk_tree_path_free(treepath);

	prefs_proxy_set_sensitive (prefsui);
}

static void
cb_down_button (GtkButton *button, KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(prefsui->tree_view));
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

	next = iter;
	if (!gtk_tree_model_iter_next(model, &next)) return;

	gtk_list_store_swap(prefsui->store, &iter, &next);

	prefsui->edited = TRUE;
	prefs_proxy_set_sensitive (prefsui);
}

static gboolean
toggle_selected (GtkTreeModel *model,
		 GtkTreePath *path,
		 GtkTreeIter *iter,
		 gpointer data)
{
	gboolean sel;

	if (strcmp((const gchar *)data,
		   gtk_tree_path_to_string(path)) == 0)
		sel = TRUE;
	else
		sel = FALSE;
	gtk_list_store_set(GTK_LIST_STORE(model), iter,
			   COLUMN_SELECTED, sel,
			   COLUMN_TERMINATOR);
	return FALSE;
}

static void
cb_selected_col_toggled (GtkCellRendererToggle *cell,
		     const gchar *path_str,
		     KzPrefsProxy *prefsui)
{
	gtk_tree_model_foreach(GTK_TREE_MODEL(prefsui->store),
			       toggle_selected,
			       (gpointer)path_str);

	prefs_proxy_set_values(prefsui);
	prefs_proxy_set_sensitive(prefsui);
	prefsui->changed = TRUE;
}

static void
cb_name_col_edited (GtkCellRendererText *cell,
		const gchar *path_str,
		const gchar *new_text,
		KzPrefsProxy *prefsui)
{
	GtkTreeIter iter;
	gboolean selected;

	gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(prefsui->store),
					    &iter, path_str);
	gtk_list_store_set(prefsui->store, &iter,
			   COLUMN_NAME, new_text,
			   COLUMN_TERMINATOR);

	prefs_proxy_set_values(prefsui);
	prefs_proxy_set_sensitive(prefsui);

	/* check if the name of selected proxy is changed */
	gtk_tree_model_get(GTK_TREE_MODEL(prefsui->store), &iter,
			   COLUMN_SELECTED,       &selected,
			   COLUMN_TERMINATOR);
	if (selected)
		prefsui->changed = TRUE;

	prefsui->edited = TRUE;
}

static void
cb_cursor_changed (GtkTreeView *tree_view,
		   KzPrefsProxy *prefsui)
{
	g_return_if_fail(prefsui);

	prefs_proxy_set_values(prefsui);
	prefs_proxy_set_sensitive(prefsui);
}

#define CB_ENTRY_CHANGED(name, col) \
static void								\
cb_##name##_changed(GtkEditable *editable, KzPrefsProxy *prefsui)	\
{									\
	GtkTreeSelection *selection;					\
	GtkTreeModel *model;						\
	GtkTreeIter iter;						\
									\
	g_return_if_fail(prefsui);					\
									\
	selection = gtk_tree_view_get_selection				\
		(GTK_TREE_VIEW(prefsui->tree_view));			\
	if (gtk_tree_selection_get_selected (selection, &model, &iter))	\
	{								\
		const gchar *str;					\
									\
		str = gtk_entry_get_text(GTK_ENTRY(prefsui->name));	\
		gtk_list_store_set(GTK_LIST_STORE(model), &iter,	\
				   COLUMN_##col, str,			\
				   COLUMN_TERMINATOR);			\
		prefsui->edited = TRUE;					\
	}								\
									\
	prefs_proxy_set_sensitive (prefsui);				\
}

CB_ENTRY_CHANGED(name,          NAME)
CB_ENTRY_CHANGED(http_host,     HTTP_HOST)
CB_ENTRY_CHANGED(https_host,    HTTPS_HOST)
CB_ENTRY_CHANGED(ftp_host,      FTP_HOST)
CB_ENTRY_CHANGED(no_proxies_on, NO_PROXIES_ON)

#define CB_SPIN_VALUE_CHANGED(name, col) \
static void								\
cb_##name##_value_changed(GtkEditable *editable, KzPrefsProxy *prefsui)	\
{									\
	GtkTreeSelection *selection;					\
	GtkTreeModel *model;						\
	GtkTreeIter iter;						\
									\
	g_return_if_fail(prefsui);					\
									\
	selection = gtk_tree_view_get_selection				\
		(GTK_TREE_VIEW(prefsui->tree_view));			\
	if (gtk_tree_selection_get_selected (selection, &model, &iter))	\
	{								\
		guint val;						\
									\
		val = gtk_spin_button_get_value_as_int			\
			(GTK_SPIN_BUTTON(prefsui->name));		\
		gtk_list_store_set(GTK_LIST_STORE(model), &iter,	\
				   COLUMN_##col, val,			\
				   COLUMN_TERMINATOR);			\
		prefsui->edited = TRUE;					\
	}								\
									\
	prefs_proxy_set_sensitive (prefsui);				\
}

CB_SPIN_VALUE_CHANGED(http_port,  HTTP_PORT)
CB_SPIN_VALUE_CHANGED(https_port, HTTPS_PORT)
CB_SPIN_VALUE_CHANGED(ftp_port,   FTP_PORT)

static void
cb_use_proxy_toggled(GtkToggleButton *editable,
		      KzPrefsProxy *prefsui)
{
	g_return_if_fail(prefsui);
	prefsui->edited = TRUE;
	prefs_proxy_set_sensitive (prefsui);
}


static void
cb_use_same_proxy_toggled(GtkToggleButton *editable,
			  KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;	
	GtkTreeIter iter;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection
		(GTK_TREE_VIEW(prefsui->tree_view));
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gboolean val;

		val = gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON(prefsui->use_same_proxy));
		gtk_list_store_set(GTK_LIST_STORE(model), &iter,
				   COLUMN_USE_SAME_PROXY, val,
				   COLUMN_TERMINATOR);
		prefsui->edited = TRUE;	
	}

	prefs_proxy_set_values (prefsui);
	prefs_proxy_set_sensitive (prefsui);
}


static void
cb_new_button (GtkButton *button, KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(prefsui->tree_view));
	gtk_tree_selection_unselect_all(selection);

	gtk_entry_set_text(GTK_ENTRY(prefsui->name), "");
	gtk_entry_set_text(GTK_ENTRY(prefsui->http_host), "");
	gtk_entry_set_text(GTK_ENTRY(prefsui->https_host), "");
	gtk_entry_set_text(GTK_ENTRY(prefsui->ftp_host), "");
	gtk_entry_set_text(GTK_ENTRY(prefsui->no_proxies_on), "");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->http_port), 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->https_port), 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->ftp_port), 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prefsui->use_same_proxy), FALSE);

	prefs_proxy_set_sensitive(prefsui);
}

static void
cb_add_button (GtkButton *button, KzPrefsProxy *prefsui)
{
	GtkTreeIter iter;
	GtkTreePath *treepath;
	const gchar *name, *http_host, *https_host, *ftp_host, *no_proxies_on;
	guint http_port, https_port, ftp_port;
	gboolean use_same_proxy;

	g_return_if_fail(prefsui);

	name           = gtk_entry_get_text(GTK_ENTRY(prefsui->name));
	http_host      = gtk_entry_get_text(GTK_ENTRY(prefsui->http_host));
	http_port      = gtk_spin_button_get_value_as_int
			(GTK_SPIN_BUTTON(prefsui->http_port));
	https_host     = gtk_entry_get_text(GTK_ENTRY(prefsui->https_host));
	https_port     = gtk_spin_button_get_value_as_int
			(GTK_SPIN_BUTTON(prefsui->https_port));
	ftp_host       = gtk_entry_get_text(GTK_ENTRY(prefsui->ftp_host));
	ftp_port       = gtk_spin_button_get_value_as_int
			(GTK_SPIN_BUTTON(prefsui->ftp_port));
	use_same_proxy = gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON(prefsui->use_same_proxy));
	no_proxies_on  = gtk_entry_get_text(GTK_ENTRY(prefsui->no_proxies_on));

	gtk_list_store_append(prefsui->store, &iter);
	gtk_list_store_set(prefsui->store, &iter,
			   COLUMN_NAME,           name,
			   COLUMN_HTTP_HOST,      http_host,
			   COLUMN_HTTP_PORT,      http_port,
			   COLUMN_HTTPS_HOST,     https_host,
			   COLUMN_HTTPS_PORT,     https_port,
			   COLUMN_FTP_HOST,       ftp_host,
			   COLUMN_FTP_PORT,       ftp_port,
			   COLUMN_USE_SAME_PROXY, use_same_proxy,
			   COLUMN_NO_PROXIES_ON,  no_proxies_on,
			   COLUMN_EDITABLE,       TRUE,
			   COLUMN_TERMINATOR);

	treepath = gtk_tree_model_get_path(GTK_TREE_MODEL(prefsui->store),
					   &iter);
	gtk_tree_view_set_cursor(GTK_TREE_VIEW(prefsui->tree_view),
				 treepath, NULL, FALSE);
	gtk_tree_path_free(treepath);

	prefsui->edited = TRUE;

	prefs_proxy_set_sensitive (prefsui);
}

static void
cb_remove_button (GtkButton *button, KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;
	GtkTreePath *treepath;
	gboolean select;
	gboolean selected;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(prefsui->tree_view));
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

	/* get next row to select */
	next = iter;
	select = gtk_tree_model_iter_next(model, &next);
	if (select)
	{
		treepath = gtk_tree_model_get_path(model, &next);
	}
	else
	{
		treepath = gtk_tree_model_get_path(model, &iter);
		select = gtk_tree_path_prev(treepath);
	}
	if (select)
		gtk_tree_view_set_cursor(GTK_TREE_VIEW(prefsui->tree_view),
					 treepath, NULL, FALSE);
	gtk_tree_path_free(treepath);

	/* check if the selected proxy is removed */
	gtk_tree_model_get(GTK_TREE_MODEL(prefsui->store), &iter,
			   COLUMN_SELECTED,       &selected,
			   COLUMN_TERMINATOR);
	if (selected)
		prefsui->changed = TRUE;

	/* remove the row */
	gtk_list_store_remove(prefsui->store, &iter);

	prefsui->edited = TRUE;

	prefs_proxy_set_sensitive (prefsui);
}

static GtkWidget *
prefs_proxy_create (void)
{
	KzPrefsProxy *prefsui = g_new0(KzPrefsProxy, 1);
	GtkWidget *main_vbox, *hbox, *vbox, *table, *frame;
	GtkWidget *scrwin, *tree_view, *button, *arrow;
	GtkListStore *store;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *column;
	GtkWidget *frame_vbox, *label, *entry, *spin, *check;
	GList *node;
	gchar *selected;
	gboolean use_proxy = FALSE;

	prefsui->main_vbox = main_vbox = gtk_vbox_new(FALSE, 0);
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_proxy_destroy);

	label = kz_prefs_ui_utils_create_title(_("Proxy"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	frame = gtk_frame_new("");
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);
	
	/* use_proxy */
	KZ_CONF_GET("Global", "use_proxy", use_proxy, BOOL);
	prefsui->use_proxy = check =
		gtk_check_button_new_with_label(_("Use proxy"));
	gtk_frame_set_label_widget(GTK_FRAME(frame), check);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prefsui->use_proxy),
				     use_proxy);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_use_proxy_toggled), prefsui);
	gtk_widget_show(check);

	/* proxy list */
	prefsui->proxy_vbox = frame_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), frame_vbox);
	gtk_widget_show(frame_vbox);
	
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(frame_vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_container_set_border_width(GTK_CONTAINER(scrwin), 5);
	gtk_box_pack_start(GTK_BOX(hbox), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	store = gtk_list_store_new(N_COLUMNS,
				   G_TYPE_BOOLEAN,	/* selected */
				   G_TYPE_STRING,	/* name */
				   G_TYPE_STRING,	/* http_host */
				   G_TYPE_UINT,		/* http_port */
				   G_TYPE_STRING,	/* https_host */
				   G_TYPE_UINT,		/* https_port */
				   G_TYPE_STRING,	/* ftp_host */
				   G_TYPE_UINT,		/* ftp_port */
				   G_TYPE_BOOLEAN,	/* use_same_proxy */
				   G_TYPE_STRING,	/* no_proxies_on */
				   G_TYPE_BOOLEAN);	/* editable */
	prefsui->store = store;

	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	prefsui->tree_view = tree_view;

	/* Selected column */
	cell = gtk_cell_renderer_toggle_new();
	g_signal_connect(cell, "toggled",
			 G_CALLBACK(cb_selected_col_toggled), prefsui);
	gtk_cell_renderer_toggle_set_radio
		(GTK_CELL_RENDERER_TOGGLE(cell), TRUE);
	column = gtk_tree_view_column_new_with_attributes
			(_("Use"), cell,
			 "active", COLUMN_SELECTED,
			 "activatable", COLUMN_EDITABLE,
			 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

	/* Name column */
	cell = gtk_cell_renderer_text_new();
	g_signal_connect(cell, "edited",
			 G_CALLBACK(cb_name_col_edited), prefsui);
	column = gtk_tree_view_column_new_with_attributes
			(_("Proxy Name"), cell,
			 "text",     COLUMN_NAME,
			 "editable", COLUMN_EDITABLE,
			 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

	/* insert proxy data */
	selected = KZ_CONF_GET_STR("Global", "proxy_name");
	prefsui->proxy_list = kz_proxy_get_list();
	for (node = prefsui->proxy_list; node; node = node->next)
	{
		KzProxyItem *item = KZ_PROXY_ITEM(node->data);
		GtkTreeIter iter;
		gchar *name, *http_host, *https_host, *ftp_host, *no_proxies_on;
		guint http_port, https_port, ftp_port;
		gboolean use_same_proxy;
		gboolean sel;

		g_object_get(G_OBJECT(item),
			     "name", &name,
			     "http_host", &http_host,
			     "http_port", &http_port,
			     "https_host", &https_host,
			     "https_port", &https_port,
			     "ftp_host", &ftp_host,
			     "ftp_port", &ftp_port,
			     "use_same_proxy", &use_same_proxy,
			     "no_proxies_on", &no_proxies_on,
			     NULL);

		if (selected && strcmp(selected, name) == 0)
			sel = TRUE;
		else
			sel = FALSE;

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
				   COLUMN_SELECTED, sel,
				   COLUMN_NAME, name,
				   COLUMN_HTTP_HOST, http_host,
				   COLUMN_HTTP_PORT, http_port,
				   COLUMN_HTTPS_HOST, https_host,
				   COLUMN_HTTPS_PORT, https_port,
				   COLUMN_FTP_HOST, ftp_host,
				   COLUMN_FTP_PORT, ftp_port,
				   COLUMN_USE_SAME_PROXY, use_same_proxy,
				   COLUMN_NO_PROXIES_ON, no_proxies_on,
				   COLUMN_EDITABLE, TRUE,
				   COLUMN_TERMINATOR);
		g_free(http_host);
		g_free(https_host);
		g_free(ftp_host);
		g_free(no_proxies_on);
	}

	gtk_container_add(GTK_CONTAINER(scrwin), tree_view);
	gtk_widget_show(tree_view);

	/* button box */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 0);
	gtk_widget_show(vbox);

	/* up button */
	prefsui->up = button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_widget_show(arrow);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 5);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_up_button), prefsui);
	gtk_widget_show(button);

	/* down button */
	prefsui->down = button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_widget_show(arrow);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 5);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_down_button), prefsui);
	gtk_widget_show(button);

	/* table */
	table = gtk_table_new(3, 8, FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_box_pack_start(GTK_BOX(frame_vbox), table, FALSE, FALSE, 0);
	gtk_widget_show(table);

	/* proxy name */
	label = gtk_label_new(_("Proxy Name"));
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 0, 1,
			 GTK_FILL, 0,
			 2, 4);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 1, 2, 0, 1,
			 GTK_FILL, 0,
			 2, 4);
	gtk_widget_show(hbox);

	prefsui->name = entry = gtk_entry_new();
	gtk_widget_set_size_request(entry, 100, -1);
        gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
	g_signal_connect(entry, "changed",
			 G_CALLBACK(cb_name_changed), prefsui);
	gtk_widget_show(entry);

	/* table headers */
	label = gtk_label_new(_("Protocol"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	gtk_widget_show(label);

	label = gtk_label_new(_("Server name or IP address"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 2, 3);
	gtk_widget_show(label);

	label = gtk_label_new(_("Port number"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 2, 3);
	gtk_widget_show(label);

	/* http */
	label = gtk_label_new(_("HTTP"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
	gtk_widget_show(label);

	prefsui->http_host = entry = gtk_entry_new();
	gtk_widget_set_size_request(entry, 250, -1);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 3, 4);
	g_signal_connect(entry, "changed",
			 G_CALLBACK(cb_http_host_changed), prefsui);
	gtk_widget_show(entry);

	prefsui->http_port = spin = gtk_spin_button_new_with_range(0, 65535, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), spin, 2, 3, 3, 4);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_http_port_value_changed), prefsui);
	gtk_widget_show(spin);

	/* https */
	label = gtk_label_new(_("HTTPS"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
	gtk_widget_show(label);

	prefsui->https_host = entry = gtk_entry_new();
	gtk_widget_set_size_request(entry, 250, -1);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 4, 5);
	g_signal_connect(entry, "changed",
			 G_CALLBACK(cb_https_host_changed), prefsui);
	gtk_widget_show(entry);

	prefsui->https_port = spin = gtk_spin_button_new_with_range(0, 65535, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), spin, 2, 3, 4, 5);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_https_port_value_changed), prefsui);
	gtk_widget_show(spin);

	/* ftp */
	label = gtk_label_new(_("FTP"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);
	gtk_widget_show(label);

	prefsui->ftp_host = entry = gtk_entry_new();
	gtk_widget_set_size_request(entry, 250, -1);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 5, 6);
	g_signal_connect(entry, "changed",
			 G_CALLBACK(cb_ftp_host_changed), prefsui);
	gtk_widget_show(entry);

	prefsui->ftp_port = spin = gtk_spin_button_new_with_range(0, 65535, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), spin, 2, 3, 5, 6);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_ftp_port_value_changed), prefsui);
	gtk_widget_show(spin);

	/* use_same_proxy */
	prefsui->use_same_proxy = check = gtk_check_button_new_with_label
		(_("Use http proxy for all other protocols"));
	gtk_table_attach_defaults(GTK_TABLE(table), check, 0, 3, 6, 7);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_use_same_proxy_toggled), prefsui);
	gtk_widget_show(check);

	/* no_proxies_on */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 3, 7, 8,
			 GTK_FILL, 0,
			 2, 4);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("No proxy for"));
        gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	prefsui->no_proxies_on = entry = gtk_entry_new();
	gtk_widget_set_size_request(entry, 200, -1);
        gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 5);
	g_signal_connect(entry, "changed",
			 G_CALLBACK(cb_no_proxies_on_changed), prefsui);
	gtk_widget_show(entry);

	/* button box */
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start(GTK_BOX(frame_vbox), hbox, FALSE, FALSE, 0);
        gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
        gtk_widget_show(hbox);

	/* clear button */
	button = gtk_button_new_from_stock(GTK_STOCK_NEW);
	prefsui->new = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_new_button), prefsui);
	gtk_widget_show(button);

	/* add button */
	button = gtk_button_new_from_stock(GTK_STOCK_ADD);
	prefsui->add = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_add_button), prefsui);
	gtk_widget_show(button);

	/* remove button */
	button = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
	prefsui->remove = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_remove_button), prefsui);
	gtk_widget_show(button);

        g_signal_connect(tree_view, "cursor-changed",
                         G_CALLBACK(cb_cursor_changed), prefsui);

	prefs_proxy_set_values(prefsui);
	prefs_proxy_set_sensitive(prefsui);

        prefsui->edited = FALSE;
        prefsui->changed = FALSE;

	g_free(selected);
	g_object_unref(store);
	return main_vbox;
}

#define KZ_CONF_PROXY_SET(s, k, v, t)					\
		  kz_profile_set_value(KZ_GET_PROXY,			\
				       s, k, &(v), sizeof(v),		\
				       KZ_PROFILE_VALUE_TYPE_##t)
#define KZ_CONF_PROXY_SET_STR(s, k, v)					\
		  kz_profile_set_value(KZ_GET_PROXY,			\
				       s, k,				\
				       (gchar *)v,			\
				       strlen((gchar *)v) + 1,		\
				       KZ_PROFILE_VALUE_TYPE_STRING)	

static void
prefs_proxy_save_proxy (KzPrefsProxy *prefsui)
{
	GtkTreeModel *model = GTK_TREE_MODEL(prefsui->store);
	GtkTreeIter iter;
	gboolean exist;
	gchar *name, *http_host, *https_host, *ftp_host, *no_proxies_on;
	guint http_port, https_port, ftp_port;
	gboolean use_proxy, use_same_proxy;
	GList *list, *node;

	if (!prefsui->edited) return;

	use_proxy = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(prefsui->use_proxy));
	KZ_CONF_SET("Global", "use_proxy", use_proxy, BOOL);

	list = kz_profile_enum_section(KZ_GET_PROXY);
	for (node = list; node; node = g_list_next(node))
		kz_profile_delete_section(KZ_GET_PROXY, node->data);
	g_list_free(list);

	for (exist = gtk_tree_model_get_iter_first(model, &iter);
	     exist;
	     exist = gtk_tree_model_iter_next(model, &iter))
	{
		gtk_tree_model_get(model, &iter,
				   COLUMN_NAME,           &name,
				   COLUMN_HTTP_HOST,      &http_host,
				   COLUMN_HTTP_PORT,      &http_port,
				   COLUMN_HTTPS_HOST,     &https_host,
				   COLUMN_HTTPS_PORT,     &https_port,
				   COLUMN_FTP_HOST,       &ftp_host,
				   COLUMN_FTP_PORT,       &ftp_port,
				   COLUMN_USE_SAME_PROXY, &use_same_proxy,
				   COLUMN_NO_PROXIES_ON,  &no_proxies_on,
				   COLUMN_TERMINATOR);

		if (http_host)
		{
			KZ_CONF_PROXY_SET_STR(name, "http_host", http_host);
			KZ_CONF_PROXY_SET(name, "http_port", http_port, INT);
		}

		KZ_CONF_PROXY_SET(name, "use_same_proxy", use_same_proxy, BOOL);

		if (no_proxies_on)
			KZ_CONF_PROXY_SET_STR(name, "no_proxies_on", no_proxies_on);

		if (use_same_proxy)
			goto ENSURE;

		if (https_host)
		{
			KZ_CONF_PROXY_SET_STR(name, "https_host", https_host);
			KZ_CONF_PROXY_SET(name, "https_port", https_port, INT);
		}
		if (ftp_host)
		{
			KZ_CONF_PROXY_SET_STR(name, "ftp_host", ftp_host);
			KZ_CONF_PROXY_SET(name, "ftp_port", ftp_port, INT);
		}
	ENSURE:
		g_free(name);
		g_free(http_host);
		g_free(https_host);
		g_free(ftp_host);
		g_free(no_proxies_on);
	}
}

static void
prefs_proxy_select_proxy(KzPrefsProxy *prefsui)
{
	GtkTreeModel *model = GTK_TREE_MODEL(prefsui->store);
	GtkTreeIter iter;
	gboolean exist;
	gboolean selected = FALSE;
	gchar *name, *http_host;
	guint http_port;

	if (!prefsui->changed || !prefsui->edited) return;

	for (exist = gtk_tree_model_get_iter_first(model, &iter);
	     exist;
	     exist = gtk_tree_model_iter_next(model, &iter))
	{
		gtk_tree_model_get(model, &iter,
				   COLUMN_NAME,           &name,
				   COLUMN_SELECTED,       &selected,
				   COLUMN_HTTP_HOST, &http_host,
				   COLUMN_HTTP_PORT, &http_port,
				   COLUMN_TERMINATOR);
		if (selected)
			break;

		g_free(name);
		g_free(http_host);
	}

	if (selected)
	{
		KZ_CONF_SET_STR("Global", "proxy_name", name);
		g_free(name);
		g_free(http_host);
	}
	else
	{
		kz_profile_delete_key(KZ_GET_GLOBAL_PROFILE, "Global", "proxy_name");
	}
}

static void
prefs_proxy_apply(KzPrefsProxy *prefsui)
{
	prefs_proxy_save_proxy(prefsui);
	prefs_proxy_select_proxy(prefsui);
}

static void
prefs_proxy_response (GtkWidget *widget, gint response)
{
	KzPrefsProxy *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
		prefs_proxy_apply(prefsui);
		prefsui->edited = FALSE;
		prefsui->changed = FALSE;
		break;
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}

static void
prefs_proxy_set_sensitive (KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;
	gboolean select;
	gboolean can_up = FALSE, can_down = FALSE;
	gboolean can_add = FALSE, can_new = TRUE;
	gboolean use_proxy = TRUE, use_same_proxy;
	const gchar *name;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection
		(GTK_TREE_VIEW(prefsui->tree_view));
	select = gtk_tree_selection_get_selected
		(selection, &model, &iter);

	if (select)
	{
		GtkTreePath *treepath;

		next = iter;
		can_down = gtk_tree_model_iter_next(model, &next);

		treepath = gtk_tree_model_get_path(model, &iter);
		can_up = gtk_tree_path_prev(treepath);
		gtk_tree_path_free(treepath);
	}

	name = gtk_entry_get_text(GTK_ENTRY(prefsui->name));
	if (name && *name)
		can_add = TRUE;

	if ((!name || !*name) && !select)
		can_new = FALSE;

	use_proxy = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(prefsui->use_proxy));
	use_same_proxy = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON(prefsui->use_same_proxy));

	gtk_widget_set_sensitive(prefsui->proxy_vbox, use_proxy);
	
	gtk_widget_set_sensitive(prefsui->name,      TRUE);
	
	gtk_widget_set_sensitive(prefsui->https_host,
				 !use_same_proxy);
	gtk_widget_set_sensitive(prefsui->https_port,
				 !use_same_proxy);
	gtk_widget_set_sensitive(prefsui->ftp_host,
				 !use_same_proxy);
	gtk_widget_set_sensitive(prefsui->ftp_port,
				 !use_same_proxy);

	gtk_widget_set_sensitive(prefsui->up,     can_up);
	gtk_widget_set_sensitive(prefsui->down,   can_down);
	gtk_widget_set_sensitive(prefsui->new,    can_new);
	gtk_widget_set_sensitive(prefsui->add,    can_add);
	gtk_widget_set_sensitive(prefsui->remove, select);
}

static void
prefs_proxy_set_values (KzPrefsProxy *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *name, *http_host, *https_host, *ftp_host, *no_proxies_on;
	guint http_port, https_port, ftp_port;
	gboolean use_same_proxy;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection
		(GTK_TREE_VIEW(prefsui->tree_view));
	if (!gtk_tree_selection_get_selected(selection, &model, &iter))
	{
		name           = g_strdup(gtk_entry_get_text
					  (GTK_ENTRY(prefsui->name)));
		http_host      = g_strdup(gtk_entry_get_text
					  (GTK_ENTRY(prefsui->http_host)));
		https_host     = g_strdup(gtk_entry_get_text
					  (GTK_ENTRY(prefsui->https_host)));
		ftp_host       = g_strdup(gtk_entry_get_text
					  (GTK_ENTRY(prefsui->ftp_host)));
		no_proxies_on  = g_strdup(gtk_entry_get_text
					  (GTK_ENTRY(prefsui->no_proxies_on)));
		http_port      = gtk_spin_button_get_value
				 (GTK_SPIN_BUTTON(prefsui->http_port));
		https_port     = gtk_spin_button_get_value
				 (GTK_SPIN_BUTTON(prefsui->https_port));
		ftp_port       = gtk_spin_button_get_value
				 (GTK_SPIN_BUTTON(prefsui->ftp_port));
		use_same_proxy = gtk_toggle_button_get_active
				 (GTK_TOGGLE_BUTTON(prefsui->use_same_proxy));
	} else {

	gtk_tree_model_get(model, &iter,
			   COLUMN_NAME, &name,
			   COLUMN_HTTP_HOST, &http_host,
			   COLUMN_HTTP_PORT, &http_port,
			   COLUMN_HTTPS_HOST, &https_host,
			   COLUMN_HTTPS_PORT, &https_port,
			   COLUMN_FTP_HOST, &ftp_host,
			   COLUMN_FTP_PORT, &ftp_port,
			   COLUMN_USE_SAME_PROXY, &use_same_proxy,
			   COLUMN_NO_PROXIES_ON, &no_proxies_on,
			   COLUMN_TERMINATOR);
	}

	gtk_entry_set_text(GTK_ENTRY(prefsui->name), name);

	gtk_entry_set_text(GTK_ENTRY(prefsui->http_host),
			   http_host != NULL ? http_host : "");
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->http_port),
				  (gdouble)http_port);

	if (use_same_proxy)
	{
		gtk_entry_set_text(GTK_ENTRY(prefsui->https_host), "");
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->https_port), 0);
		gtk_entry_set_text(GTK_ENTRY(prefsui->ftp_host), "");
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->ftp_port), 0);
	}
	else
	{
		gtk_entry_set_text(GTK_ENTRY(prefsui->https_host),
				   https_host != NULL ? https_host : "");
		gtk_entry_set_text(GTK_ENTRY(prefsui->ftp_host),
				   ftp_host != NULL ? ftp_host : "");
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->https_port),
					  (gdouble)https_port);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(prefsui->ftp_port),
					  (gdouble)ftp_port);
	}
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prefsui->use_same_proxy),
				     use_same_proxy);
	gtk_entry_set_text(GTK_ENTRY(prefsui->no_proxies_on), no_proxies_on);

	g_free(name);
	g_free(http_host);
	g_free(https_host);
	g_free(ftp_host);
	g_free(no_proxies_on);
}

KzPrefsWinPageEntry *
prefs_proxy_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
