/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>
   Copyright (C) 2008 Christian Weilbach <christian_weilbach@web.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "importmediadialog.h"

#include <QTreeWidgetItem>
#include <QTreeWidget>
#include <QFileInfo>

#include <kdebug.h>
#include <kicon.h>
#include <kio/job.h>
#include <kmessagebox.h>

#include "kbloggermedia.h"
#include "itemsmanager.h"

namespace KBlogger
{

ImportMediaWidget::ImportMediaWidget(QWidget *parent): QWidget(parent),
        blackListedUrls(0)
{
    kDebug();
    setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);
}

ImportMediaWidget::~ImportMediaWidget()
{
    kDebug();
}

void ImportMediaWidget::addUrl(const QString& url, const QString& blogname, const QString& postId)
{
    kDebug();
    Q_ASSERT( !url.isEmpty() );
    Q_ASSERT( !blogname.isEmpty() );

    QStringList ignoredUrls;
    ignoredUrls = ItemsManager::self()->getIgnoredMediaUrls(blogname);

    if ( ignoredUrls.contains(url, Qt::CaseInsensitive) ) {
        blackListedUrls++;
        blacklistLabel->setText(i18n("There are %1 urls blacklisted not showed here.", blackListedUrls));
        return;
    }

    //Blogger.com thumbnails look like this:
    //  http://xxx.blogger.com/_lJiR72o3Bww/R1MvGt1yWmI/AAAAAAAAAT4/sXtF1aY1HQY/s320/IMG_9261.JPG
    // TODO check for www.blogspot.com urls
//     QRegExp bloggerThumbnailRx("http://[\\d\\w]*.blogger.com/([\\d\\w]*/){1,4}s320/IMG_\\d*.JPG");

    QTreeWidgetItem *item;
    // FIXME this does not work reliably look at the note in handleJobResult
//     if ( !urls.isEmpty() && ( url.contains("thumb") || //Generic
//             url.contains(bloggerThumbnailRx ) )
//        ) { //NOTE Improve it!
//         item = new QTreeWidgetItem( urls.last() ); //Thumbnail
//     } else

    item = new QTreeWidgetItem( mediaTreeWidget ); //No Thumbnail
    item->setText(0, url );
    item->setIcon(0, KIcon("document-import") );
    item->setText(1, blogname );
    item->setText(2, postId );
    item->setText(3, i18n("Not downloaded yet"));
    mediaTreeWidget->insertTopLevelItem( 0, item );
    urls += item;
}

void ImportMediaWidget::close()
{
    kDebug();

    // clean things up
    QList<QTreeWidgetItem*>::ConstIterator it = urls.begin();
    QList<QTreeWidgetItem*>::ConstIterator end = urls.end();
    for( ; it != end; it++ ){
        mediaTreeWidget->removeItemWidget( *it, 0 );
        delete( *it );
    }
    urls.clear();
}

void ImportMediaWidget::accept()
{
    kDebug();
    //Download the urls in the media cache
    QList<QTreeWidgetItem*>::ConstIterator it = urls.begin();
    QList<QTreeWidgetItem*>::ConstIterator end = urls.end();
    for( ; it != end; it++ ){
        createMediaFromUrl( *it );
    }
}

void ImportMediaWidget::createMediaFromUrl(QTreeWidgetItem* item)
{
    KUrl url = item->text(0);
    kDebug() << url;
    Q_ASSERT( url.isValid() );

    KIO::StoredTransferJob* job = KIO::storedGet( url, KIO::NoReload, KIO::HideProgressInfo );

    jobsMap[ job ] = item;

    connect( job, SIGNAL( result( KJob* ) ),
             this, SLOT( handleFileJobResult( KJob* ) ) );
}

void ImportMediaWidget::handleFileJobResult(KJob* job)
{
    kDebug();
    if( !job ){
        kError() << "The job died.";
        return;
    }

    QString mediaMimetype = dynamic_cast<KIO::StoredTransferJob*>(job)->mimetype();
    QByteArray mediaData = dynamic_cast<KIO::StoredTransferJob*>(job)->data();
    QTreeWidgetItem* item = jobsMap[ job ];
    Q_ASSERT( item );

    if ( job->error() ) {
        item->setIcon(0, KIcon("dialog-cancel") );
        item->setText(3, i18n("Read-Error.") );
        kDebug() << "Read-Error: " << job->errorString();
        delete job;
        return;
    }
    delete job;
    jobsMap.remove( job );

    //Settings Vars
    KUrl mediaUrl( item->text(0) );
    const QString filename = QFileInfo(mediaUrl.path()).fileName();
    const QString blogname = item->text(1);

    //Test: media Size < 1Mb TODO why? i would do at least 2Mb
    if ( (mediaData.count() / 1024) > 1024 ) { //Max 1 Mb
        item->setIcon(0, KIcon("dialog-cancel") );
        item->setText(3, i18n("Size > 1Mb: Url Blacklisted"));
        ItemsManager::self()->addToIgnoredMediaUrls( mediaUrl, blogname );
        return;
    }

    if ( mediaMimetype == "text/html" ) {
        item->setIcon(0, KIcon("dialog-cancel") );
        item->setText(3, i18n("Error 404: Url blacklisted"));
        ItemsManager::self()->addToIgnoredMediaUrls( mediaUrl, blogname );
        return;
    }

    KBloggerMedia *media;
    media = new KBloggerMedia( mediaData,
                                    filename,
                                    mediaMimetype,
                                    blogname
                          );

    if ( !mediaUrl.isLocalFile() )
        media->setUrl(mediaUrl);

    if ( media->isCached() ) {
        item->setIcon(0, KIcon("dialog-ok-apply") );
        item->setText(3, QString("%1 Kb").arg(mediaData.count() / 1024));

        //Thumbnail managing FIXME this does not work, since the jobs may not
        // call this function in the correct order
//         if ( prev_media && filename.contains("thumb") ) {
//             prev_media->setThumbnail(media);
//             media->setFullSizeImage(prev_media);
//             prev_media = 0; // this is a thumbnail
//         } else {
//             prev_media = media;
//         }

         //Add Media in the MediaListWidget
        ItemsManager::self()->addMedia(media);
    } else {
        item->setIcon(0, KIcon("dialog-cancel") );
        item->setText(3, i18n("CacheFile writing Error"));
    }

    if( jobsMap.isEmpty() ) {
        close();
    }
}


} //Namespace

#include "importmediadialog.moc"
