/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "medialist.h"

#include <QTreeWidget>
#include <QFile>
#include <QDataStream>
#include <QMenu>

#include <kdebug.h>
#include <kblog/blog.h>
// #include <kblog/blogmedia.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kblog/blogmedia.h>
#include <kdatetime.h>
#include <kactioncollection.h>
#include <kapplication.h>

#include "kblogger.h"
#include "itemsmanager.h"
#include "kbloggermedia.h"
#include "mainwidget.h"

#define DIR_TO_SAVE_MEDIA "media/"

#define MAGIC_KEY 0xA1B2C3D5
#define VERSION 002

namespace KBlogger
{
/**
 *  mediaListWidget
 */

MediaListWidget::MediaListWidget( const QString& blogname, QWidget* parent): ItemList(parent)
{
    kDebug();
    setupUi(this);
    setBlogname(blogname);
}

MediaListWidget::~MediaListWidget()
{
    kDebug();
}

void MediaListWidget::appendMedia( KBloggerMedia* kbMedia )
{
    kDebug();
    Q_ASSERT( kbMedia );

    //Sets status string
    QString status;

    KLocalizedString ks = ki18nc( "a list of the local posts in a tree view", "Local" );

    switch ( kbMedia->status() ) {
    case KBlog::BlogMedia::New: {
        status = ks.toString();
    }
    ; break;
    case KBlog::BlogMedia::Created: {
        status = i18n("Uploaded");
    }
    ; break;
    case KBlog::BlogMedia::Error: {
        status = i18n("Error");
    }
    ; break;

    };

    //TEST
    /*
    QMapIterator<QTreeWidgetItem*, KBloggerMedia*> i(mMediaMap);

    while (i.hasNext()) {

        QTreeWidgetItem *treeItem;
        KBloggerMedia *media;
        i.next();
        treeItem = i.key();
        media = i.value();
        kDebug() << "treeItem" << treeItem
                 << "media" << media;
        if ( media->name() == kbMedia->name() ) { //TODO Use Checksum?
            //if ( kbMedia->status() != KBlog::BlogMedia::Created ) return;
            mMediaMap.remove(treeItem);
            delete treeItem;
            break;
        }
    }*/

    //Creates QTreeWidgetItem for kbMedia
    QTreeWidgetItem *kbMediaItem(0);
    KBloggerMedia* fullSizeMedia(0);
    //NOTE kbMedia is a Thumbnail if have a fullSizeImage
    fullSizeMedia = kbMedia->fullSizeImage();

    if ( fullSizeMedia ) {
        kbMediaItem = new QTreeWidgetItem( mMediaMap.key(fullSizeMedia) );
    } else {
        kbMediaItem = new QTreeWidgetItem(listTree);
    }

    kbMediaItem->setIcon (Name, kbMedia->icon() );
    kbMediaItem->setText(Name, kbMedia->name() );
    kbMediaItem->setText(CreationDate,
                         kbMedia->creationDateTime().dateTime().toString("ddd, dd MMM yyyy, HH:mm") );
    kbMediaItem->setText( Size, kbMedia->sizeString() );
    kbMediaItem->setText(Status, status );

    //Appends kbMedia to QMap(key sorted) and Qlist (sorted)
    mMediaMap[kbMediaItem] = kbMedia;
    mSortedMediaList << kbMedia;
}

KBloggerMedia* MediaListWidget::getCurrentMedia()
{
    kDebug();
    if ( !listTree->currentItem() ) return 0;
    return mMediaMap[ listTree->currentItem() ];
}

KBloggerMedia* MediaListWidget::getCurrentThumbnail()
{
    kDebug();
    if( !listTree->currentItem() ) return 0;
    if( !listTree->currentItem()->child( 0 ) ) return 0;
    return mMediaMap[ listTree->currentItem()->child( 0 ) ];
}

int MediaListWidget::count()
{
    return mMediaMap.count();
}


/**
 *   MediaList
 */
MediaList::MediaList( const QString& blogname,
                      Application* kbloggerParent,
                      QWidget* parent ): MediaListWidget(blogname, parent),
        mKblogger(kbloggerParent)
{
    kDebug();

    ItemsManager::self(this)->addInTheWidgetsList(blogname, this);
    mFilename = QString(blogname).prepend(DIR_TO_SAVE_MEDIA);
    load();

    //Connect
    connect(listTree, SIGNAL( itemSelectionChanged () ),
            SLOT( previewItem() ));
    connect(listTree, SIGNAL(  itemDoubleClicked(QTreeWidgetItem*, int ) ),
            SLOT( openSelectedMedia() ));
    connect(listTree, SIGNAL( customContextMenuRequested(const QPoint&) ),
            SLOT( showPopup(const QPoint&) ));
}

MediaList::~MediaList()
{
    kDebug();
    QList<QTreeWidgetItem*> treeList = mMediaMap.keys();
    QList<QTreeWidgetItem*>::ConstIterator it = treeList.begin();
    QList<QTreeWidgetItem*>::ConstIterator end = treeList.end();
    for( ; it != end; it++ ){
        delete mMediaMap[ (*it) ];
        delete ( *it );
    }
}

void MediaList::appendMedia(KBloggerMedia* kbMedia)
{
    kDebug();
    if ( kbMedia->checksum() == 0 ) return;
    MediaListWidget::appendMedia( kbMedia );

    //
    MainWidget::self()->updateItemNumbers();
}

void MediaList::save()
{
    kDebug();
    Q_ASSERT( !mFilename.isEmpty() );

    //Disable User Input
    KApplication::kApplication()->processEvents(QEventLoop::ExcludeUserInputEvents);

    //Sets file to save
    QString fileToSave;
    fileToSave = KStandardDirs::locateLocal("appdata", mFilename , true);
    kDebug() << "saving file: " << fileToSave;

    //Opens the file in write mode
    QFile file( fileToSave );
    file.open(QIODevice::WriteOnly);
    QDataStream out(&file);   // we will serialize the data into the file

    //Writes header
    out << (quint32)MAGIC_KEY; //"magic number"
    out << (qint32)VERSION; //binary file version

    //Writes File Version
    out.setVersion(QDataStream::Qt_4_0);

    //Writes the number of media to save.
    out << mMediaMap.count();

    //Writes mPostMap data;
    foreach (KBloggerMedia* media, mSortedMediaList) {
        Q_ASSERT( media );
        kDebug() << "name=" << media->name();
        kDebug() << "statusInt =" << media->status();
        kDebug() << "checksum =" << media->checksum();

        out << media->name();
        out << media->mimetype();
        out << media->status();
        out << media->url(); //uploadedMediaUrl
        out << media->checksum();
    }

    //Save Ignored Media Urls
    out << mIgnoredMediaUrls;

    file.close();

    //Enable User input
    KApplication::kApplication()->processEvents();
}

void MediaList::load()
{
    kDebug();
    if (mFilename.isEmpty()) {
        kDebug() << "\tmFilename is empty" << endl;
        return;
    }
    mMediaMap.clear(); //FIXME Memory Leak?

    QString fileToOpen = KStandardDirs::locate("appdata", mFilename);
    if (fileToOpen.isEmpty()) {
        kDebug() << "\t the file " <<  mFilename << " doesn't exist" << endl;
        return;
    }
    QFile file( fileToOpen );
    file.open(QIODevice::ReadOnly);
    QDataStream in(&file);

    // Read and check the header
    quint32 magic;
    in >> magic;
    if (magic != MAGIC_KEY) {
        kDebug() << "BAD MAGIC KEY" << endl;
        return;
    }

    // Read the version
    qint32 version;
    in >> version;
    if (version != VERSION) {
        kDebug() << "BAD VERSION" << endl;
        return;
    }

    // Read the number of entries
    qint32 entriesNumber;
    in >> entriesNumber;
    kDebug() << "there are " << entriesNumber << " media saved" << endl;
    //Getting data from file, clear the listTree Widget and insert the readed items in the widget.

    listTree->clear();

    //Thumbnail managing: this var contains fullSizeImage ptr
    KBloggerMedia *prev_media = 0;

    for ( int i = 0; i < entriesNumber; ++i ) {

        QString name, mimetype;
        KUrl uploadedMediaUrl; //Contains the url of a uploaded Media
        int statusInt;
        quint16 checksum;

        //Read data
        in >> name;
        in >> mimetype;
        in >> statusInt;
        in >> uploadedMediaUrl;
        in >> checksum;

        kDebug() << "New Media loaded - checksum" << QString("%1").arg(checksum, 4, 16);

        //create Media
        KBloggerMedia *media;
        media = new KBloggerMedia(name, mimetype, blogname(), checksum);
        media->setStatus( static_cast<KBlog::BlogMedia::Status>(statusInt) );
        media->setUrl(uploadedMediaUrl);

        //Thumbnail managing
        if ( prev_media && name.contains("thumbnail") ) {
            prev_media->setThumbnail(media);
            media->setFullSizeImage(prev_media);
            prev_media = 0;
        } else {
            prev_media = media;
        }
        appendMedia( media );
    }

    //Load Ignored Media Urls
    in >> mIgnoredMediaUrls;

    //Close QFile
    file.close();
}

void MediaList::updateSearchLine()
{
    kDebug();
    Q_ASSERT(mKblogger);
    mKblogger->updateSearchLine(listTree);
}

void MediaList::openSelectedMedia()
{
    kDebug();
    KBloggerMedia* media = getCurrentMedia();
    if (media)
        media->run();
}

QList<KBloggerMedia*> MediaList::getMediaList()
{
    kDebug();
    return mMediaMap.values();
}

void MediaList::removeLocalItem( KBloggerMedia *media )
{
    kDebug();
    QTreeWidgetItem* currentItem;

    if (!media) {
        //Gets the ptr of selected media in the widget
        media = getCurrentMedia();
        Q_ASSERT( media );
        //Gets the ptr of the selected QTreeWidgetItem
        currentItem = listTree->currentItem();
    } else {
        QMapIterator<QTreeWidgetItem*, KBloggerMedia*> i(mMediaMap);
        while (i.hasNext()) {
            i.next();
            if ( i.value() == media ) {
                currentItem = i.key();
                break;
            }
        }
        Q_ASSERT( currentItem );
    }

    //Thumbnail managing: Settings ptr
    KBloggerMedia* fsi;
    KBloggerMedia* thumb;
    fsi = media->fullSizeImage();
    thumb = media->thumbnail();

    //Thumbnail managing: Remove Thumbnail
    if ( thumb ) {
        thumb->setFullSizeImage(0);
        removeLocalItem(thumb);
    }

    //Informs the user that he's deleting a uploaded media
    if ( media->status() != KBlog::BlogMedia::New ) {
        KMessageBox::sorry( this, i18n("You are deleting a cached file.\nKBlogger can not remove the file on the server") );
    }

    //Remove cached file
    if (! media->removeCachedFile() ) {
        KMessageBox::error( this, i18n("Impossible remove the cached file"));
        return;
    }

    //Remove from the Map and List
    {
        KBloggerMedia* mediaToRemove;
        mediaToRemove = mMediaMap.value(currentItem);
        Q_ASSERT(mediaToRemove);

        mSortedMediaList.removeAll(mediaToRemove);
        mMediaMap.remove(currentItem);
    }

    //Remove QTreeWidgetItem
    delete currentItem;
    delete media;

    //Thumbnail managing: Remove Full Size Image
    //NOTE: Remove his thumbnail first because thumbnail QTreeWidgetItem
    //     is a child of Full Size Image QTreeWidgetItem.
    if ( fsi ) {
        fsi->setThumbnail(0);
        removeLocalItem(fsi);
    }

    MainWidget::self()->updateItemNumbers();
}

void MediaList::addToIgnoredMediaUrls(const KUrl &url)
{
    kDebug();
    Q_ASSERT( url.isValid() );
    mIgnoredMediaUrls << url.url();
}

QStringList MediaList::getIgnoredMediaUrls()
{
    kDebug();
    return mIgnoredMediaUrls;
}

void MediaList::previewItem()
{
    kDebug();
    KBloggerMedia* media = getCurrentMedia();
    if ( !media ) return;
    MainWidget::self()->viewEntry(media);
}

void MediaList::showPopup(const QPoint& pos)
{
    kDebug();

    QTreeWidgetItem * item = 0 ;
    item = listTree->itemAt(pos);
    if (!item) return;

    QMenu menu(i18n("Post menu"), listTree);

    KActionCollection *aktionCollection = MainWidget::self()->actionCollection();
    const QList< QActionGroup * > actionGroups =  aktionCollection->actionGroups();
    QActionGroup* actionGroup;
    for (int i = 0; i < actionGroups.size(); ++i) {
        if ( actionGroups.at(i)->objectName() == "mediaActionGroup" ) {
            actionGroup = actionGroups.at(i);
            break;
        }
    }

    QList<QAction *> actions = actionGroup->actions();
    for (int i = 0; i < actions.size(); ++i) {
        if ( actions.at(i) )
            menu.addAction( actions.at(i) );
    }

    menu.exec(QCursor::pos());
}



}


#include "medialist.moc"
