/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "dbusutil.h"

#include <klocale.h>

QString DBusUtil::niceType( QString const & type )
throw( QDBusXmlError ) {
    QString niceType;

    unsigned int numStructs = 0;
    bool inDict = false;
    for ( unsigned int i = 0; i < type.length(); i++ ) {
        QString old = niceType;
        QString thisType;
        switch ( type[i].latin1() ) {
            case 0:
                return niceType;
                break;

            case 'y':
                thisType = "BYTE";
                break;

            case 'b':
                thisType = "BOOLEAN";
                break;

            case 'n':
                thisType = "INT16";
                break;

            case 'q':
                thisType = "UINT16";
                break;

            case 'i':
                thisType = "INT32";
                break;

            case 'u':
                thisType = "UINT32";
                break;

            case 'x':
                thisType = "INT64";
                break;

            case 't':
                thisType = "UINT64";
                break;

            case 'd':
                thisType = "DOUBLE";
                break;

            case 's':
                thisType = "STRING";
                break;

            case 'o':
                thisType = "OBJECT_PATH";
                break;

            case 'g':
                thisType = "SIGNATURE";
                break;

            // Structs are handled later also
            case ')':
                thisType = " )";
                if ( numStructs == 0 )
                    throw QDBusXmlError(
                      i18n(
                        "struct closed while not in struct in argument "
                        "type"
                      )
                    );
                numStructs--;
                break;

            case 'v':
                thisType = "VARIANT";
                break;

            // Dict entries are handled later also
            case '}':
                thisType = " >";
                if ( !inDict )
                    throw QDBusXmlError(
                      i18n( "dict closed while not in dict in argument type" )
                    );
                inDict = false;
                break;
        }

        // If we've got a simple type, just append it
        if ( !thisType.isNull() ) {

            if ( inDict ) {
                if ( type[i+1].latin1() != '}' )
                    thisType += ", ";
            }
            // If we're in a struct, put a comma and space on the end,
            // as long as next character isn't ')'
            else if ( numStructs > 0 ) {
                if ( (i + 1) >= type.length() )
                    throw QDBusXmlError(
                      i18n( "Unterminated struct found in argument type" )
                    );

                if ( type[i+1].latin1() != ')' )
                    thisType += ", ";
            }
            
            niceType += thisType;
        }

        // Now do array, struct, dict.
        switch ( type[i].latin1() ) {
            case '(':
                // Can only have primitives in dict
                if ( inDict )
                    throw QDBusXmlError(
                      i18n( "struct found within dict in argument type" )
                    );
                
                numStructs++;
                niceType += "STRUCT of ( ";
                break;



            case '{':
                // Can only have primitives in dict
                if ( inDict )
                    throw QDBusXmlError(
                      i18n( "dict found within dict in argument type" )
                    );

                // Check that we are only 2 elements
                if ( (i + 3) >= type.length() )
                    throw QDBusXmlError(
                      i18n( "Unterminated dict found in argument type" )
                    );
                if ( type[i+1] == '}' )
                    throw QDBusXmlError(
                      i18n( "dict with zero elements found" )
                    );
                if ( type[i+2] == '}' )
                    throw QDBusXmlError(
                      i18n( "dict with one element found" )
                    );
                if ( type[i+3] != '}' )
                    throw QDBusXmlError(
                      i18n(
                        "Unterminated dict or dict with more than two "
                        "elements or dict with non-primitive elements found"
                      )
                    );

                inDict = true;
                niceType += "DICT of < ";
                break;



            case 'a':
                niceType += "ARRAY of ";
                break;

            case 'e':
            case 'r':
                // These are in the spec, but I really don't understand how
                // they fit into the message format... there are no examples :-(
                throw QDBusXmlError(
                  "'e' and 'r' aren't correctly interpreted yet.  Fix me!"
                );
                break;

            } // switch ( m_type[i].latin1()


            if ( niceType == old )
                throw QDBusXmlError(
                  i18n(
                    "Unrecognised argument type '%1' encountered"
                  ).arg( type[i] )
                );
        } // for

    return niceType;
}

QMutex DBusUtil::mutex( true );
