/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004 by Michael Buesch                            *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "kwalletif.h"
#include "pwmexception.h"
#include "listobjselectwnd.h"
#include "pwmdoc.h"

#include <klocale.h>
#include <kmessagebox.h>


KWalletIf::KWalletIf(QWidget *_parentView)
 : QObject()
{
	wallet = 0;
	parentView = _parentView;
}

KWalletIf::~KWalletIf()
{
	delWalletObj();
}

void KWalletIf::delWalletObj()
{
	delete_ifnot_null(wallet);
	curEntries.clear();
}

bool KWalletIf::kwalletImport(PwMDoc *doc)
{
	importing = true;
	if (!openWallet()) {
		return false;
	}
	if (!selectFolder()) {
		delWalletObj();
		return false;
	}
	if (!selectEntries()) {
		delWalletObj();
		return false;
	}
	if (!doImport(doc)) {
		delWalletObj();
		return false;
	}
	delWalletObj();
	return true;
}

bool KWalletIf::kwalletExport(PwMDoc *doc)
{
	importing = false;
	if (!openWallet()) {
		return false;
	}
	if (!doExport(doc)) {
		delWalletObj();
		return false;
	}
	delWalletObj();
	return true;
}

bool KWalletIf::doImport(PwMDoc *doc)
{
	PWM_ASSERT(doc);
	QString val;
	PwMDataItem dta;
	PwMerror err;
	QStringList::iterator listBegin = curEntries.begin(),
			      listEnd = curEntries.end(),
			      listI = listBegin;
	while (listI != listEnd) {
		if (wallet->readPassword(*listI, val)) {
			KMessageBox::error(parentView,
					   i18n("This does not seem to be a password entry;\n"
						"did you select the correct folder?"),
					   i18n("Not Password Entry"));
			return false;
		}
//		printDebug(string("KWalletIf: importing: ") + val.latin1());
		dta.clear();
		dta.desc = (*listI).latin1();
		dta.pw = val.latin1();
		err = doc->addEntry(wallet->currentFolder(), &dta, true);
		if (err != e_success) {
			/* we simply skip the entry, if it already exists
			 * and don't return an error.
			 */
			if (err != e_entryExists) {
				KMessageBox::error(parentView,
						   i18n("Could not import data;\n"
							"an error occurred while writing "
							"the data to the document."),
						   i18n("Import Error"));
				doc->flagDirty();
				return false;
			}
		}
		++listI;
	}
	doc->flagDirty();
	return true;
}

bool KWalletIf::doExport(PwMDoc *doc)
{
	PWM_ASSERT(doc);
	vector<string> catList;
	doc->getCategoryList(&catList);
	vector<string>::iterator catI = catList.begin(),
				 catEnd = catList.end();
	int numEntr, i;
	PwMDataItem d;
	d.clear();
	doc->unlockAll_tempoary();
	while (catI != catEnd) {
		if (!wallet->createFolder(catI->c_str())) {
			KMessageBox::error(parentView,
					   i18n("Could not export data;\n"
						"an error occurred while writing "
						"the data to the wallet."),
					   i18n("Export Error"));
			doc->unlockAll_tempoary(true);
			return false;
		}
		if (!wallet->setFolder(catI->c_str())) {
			KMessageBox::error(parentView,
					   i18n("Could not export data;\n"
						"an error occurred while writing "
						"the data to the wallet."),
					   i18n("Export Error"));
			doc->unlockAll_tempoary(true);
			return false;
		}
		numEntr = doc->numEntries(catI->c_str());
		for (i = 0; i < numEntr; ++i) {
			if (!doc->getEntry(catI->c_str(), i, &d)) {
				doc->unlockAll_tempoary(true);
				return false;
			}
//			printDebug(string("KWalletIf: exporting: ") + d.desc);
			if (wallet->writePassword(d.desc.c_str(), d.pw.c_str())) {
				KMessageBox::error(parentView,
						   i18n("Could not export data;\n"
							"an error occurred while writing "
							"the data to the wallet."),
						   i18n("Export Error"));
				doc->unlockAll_tempoary(true);
				return false;
			}
		}
		++catI;
	}
	doc->unlockAll_tempoary(true);
	return true;
}

bool KWalletIf::openWallet()
{
	PWM_ASSERT(!wallet);
	QStringList walletList(Wallet::walletList());
	if (!walletList.size())
		return false;
	ListObjSelectWnd selWnd(i18n("Select Wallet"),
				importing ?
				 i18n("Please select the wallet you want to import "
				      "some data from:")
				:
				 i18n("Please select the wallet you want to export "
				      "the current document to:"),
				parentView);
	QStringList::iterator listBegin = walletList.begin(),
			      listEnd = walletList.end(),
			      listI = listBegin;
	while (listI != listEnd) {
		selWnd.addObj(*listI);
		++listI;
	}
	selWnd.selectObj(Wallet::LocalWallet());
	if (selWnd.exec() != 1)
		return false;
	wallet = Wallet::openWallet(selWnd.getSelected());
	if (!wallet || !wallet->isOpen()) {
		delWalletObj();
		return false;
	}
	printDebug(string("KWalletIf: opened wallet: ") + selWnd.getSelected().latin1());
	return true;
}

bool KWalletIf::selectFolder()
{
	PWM_ASSERT(wallet);
	QStringList folderList(wallet->folderList());
	if (!folderList.size()) {
		KMessageBox::error(parentView,
				   i18n("No folders found in this wallet."),
				   i18n("No Folder"));
		return false;
	}
	ListObjSelectWnd selWnd(i18n("Select Folder"),
				i18n("Please select the folder of the wallet "
				     "you want to open."),
				parentView);
	QStringList::iterator listBegin = folderList.begin(),
			      listEnd = folderList.end(),
			      listI = listBegin;
	while (listI != listEnd) {
		selWnd.addObj(*listI);
		++listI;
	}
	selWnd.selectObj(Wallet::PasswordFolder());
	if (selWnd.exec() != 1)
		return false;
	if (!wallet->setFolder(selWnd.getSelected()))
		return false;
	printDebug(string("KWalletIf: selected folder: ") + selWnd.getSelected().latin1());
	return true;
}

bool KWalletIf::selectEntries()
{
	PWM_ASSERT(wallet);
	QStringList entryList(wallet->entryList());
	if (!entryList.size()) {
		KMessageBox::error(parentView,
				   i18n("This folder does not contain any entries."),
				   i18n("No Entries"));
		return false;
	}
	ListObjSelectWnd selWnd(i18n("Select entries"),
				i18n("Please select the entries you want to import."),
				parentView, true);
	QStringList::iterator listBegin = entryList.begin(),
			      listEnd = entryList.end(),
			      listI = listBegin;
	while (listI != listEnd) {
		if (wallet->entryType(*listI) == Wallet::Password)
			selWnd.addObj(*listI);
		++listI;
	}
	if (selWnd.exec() != 1)
		return false;
	curEntries = selWnd.getSelectedList();
	if (!curEntries.size()) {
		return false;
	}
	return true;
}

#include "kwalletif.moc"
