/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#ifndef __SERIALIZER_H
#define __SERIALIZER_H

#include "pwmdoc.h"

#include <qcstring.h>
#include <qdom.h>

#include <vector>

using std::vector;

/** This serializes its input data into
  * the PwManager-XML-datastream, that becomes
  * encrypted and maybe compressed
  */
class Serializer
{
public:
	/** construct an empty serializer document */
	Serializer();
	/** construct a serializer document and parse "buffer" */
	Serializer(const QCString &buffer);
	/** destructor */
	virtual ~Serializer();

	/** clears all data */
	void clear();
	/** parse the given data buffer */
	bool parseXml(const QCString &buffer);
	/** returns the current XML data */
	QCString getXml();
	/** serialize "dta" and store it as XML data */
	bool serialize(const vector<PwMCategoryItem> &dta);
	/** deserialize the (parsed) XML data and store it in "dta" */
	bool deSerialize(vector<PwMCategoryItem> *dta);
	/** sets the initial default lockStat we should assign */
	void setDefaultLockStat(bool stat)
			{ defaultLockStat = stat; }

protected:
	/** main data holder */
	QDomDocument *domDoc;
	/** default lockStat to assign */
	bool defaultLockStat;
	/** Should we encode all data with base64?
	  * Please note that this is NOT backward compatible
	  * with versions before 1.2.4.
	  */
	bool base64Data;

protected:
	/** check if this is valid PwManager XML data */
	bool checkValid();
	/** read the categories in the node "n" */
	bool readCategories(const QDomNode &n,
			    vector<PwMCategoryItem> *dta);
	/** read the entries in the node "n" */
	bool readEntries(const QDomNode &n,
			 vector<PwMDataItem> *dta);
	/** extract the data out of the given item at "n" */
	bool extractEntry(const QDomNode &n,
			  PwMDataItem *dta);
	/** generates a new root node and sets all initial parameters */
	QDomElement genNewRoot();
	/** add new categories to the XML data stream in e */
	bool addCategories(QDomElement *e,
			   const vector<PwMCategoryItem> &dta);
	/** add the given new entries to the XML data stream in e */
	bool addEntries(QDomElement *e,
			const vector<PwMDataItem> &dta);
	/** do serialize and write the given entry to the XML stream */
	bool writeEntry(QDomElement *e,
			const PwMDataItem &_dta);
	/** escape illegal characters out of the given entry data string */
	QString escapeEntryData(QString dta);
	/** un-escape illegal characters out of the given entry data string.
	  * Be sure to set base64Data correctly before calling this function!
	  */
	QString unescapeEntryData(QString dta);
};

#endif // __SERIALIZER_H
