# -*- coding: utf-8 -*-
# KeyJnoteGUI: A frontend to KeyJnote, an eye-candy presentation programm
# Copyright (C) 2006  Sebastian Wiesner <basti.wiesner@gmx.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Contains widgets used by KeyJnoteGUI"""

__revision__ = '$Id: kjnguiwidgets.py 108 2007-04-10 17:28:47Z lunar $'

from qt import Qt, QObject, SIGNAL, QStringList, QVBoxLayout, QTime
from kdecore import i18n, KGlobalSettings
from kdeui import KDialogBase
from kfile import KFile, KImageFilePreview

# KeyJnoteGUI modules
from presentationwidget import PresentationWidget
from settingswidget import SettingsWidget

from settings import Geometry, standard_resolutions


# TODO: have rotation available in the gui


class PresentationUI(PresentationWidget):
    """Reimplements PresentationWidget to extend its features"""

    def __init__(self, parent):
        PresentationWidget.__init__(self, parent)
        QObject.connect(self.source_pdf, SIGNAL('toggled(bool)'),
                        self.pdf_toggled)
        QObject.connect(self.source_directory, SIGNAL('toggled(bool)'),
                        self.directory_toggled)
        QObject.connect(self.transitions.availableListBox(),
                        SIGNAL('onItem(QListBoxItem *)'),
                        self.transition_on_item)
        QObject.connect(self.transitions.selectedListBox(),
                        SIGNAL('onItem(QListBoxItem *)'),
                        self.transition_on_item)
        # enable preview for cursor images
        dlg = self.cursor_file.fileDialog()
        self.cursor_preview_widget = KImageFilePreview(dlg)
        dlg.setPreviewWidget(self.cursor_preview_widget)
        # XXX: extend this filter to include all supported formats
        self.cursor_file.setFilter('image/png')

    def transition_on_item(self, item):
        """Handles the onItem signals of the listboxes of the action
        selector widget transition_input. It shows a short description of
        the transition in the status bar"""
        # TODO: Would be nice, to have this translated
        doc = self.parent().wrapper.transition_doc(str(item.text()))
        self.parent().statusBar().message(doc)
        
    def directory_toggled(self, on):
        """Invoked, if the directory radio button in the source box is
        toggled.
        :param on: True, if it is checked, False otherwise"""
        if on:
            self.source.setMode(KFile.Directory | KFile.LocalOnly |
                                KFile.ExistingOnly)

    def pdf_toggled(self, on):
        """Invoked, if the pdf radio button in the source box is toggled.
        :param on: True, if it is checked, False otherwise"""
        if on:
            self.source.setMode(KFile.File | KFile.LocalOnly |
                                KFile.ExistingOnly)
            self.source.setFilter('application/pdf')

    def load(self, settings):
        """Loads `settings`"""
        self.source.setURL(settings.source)
        if settings.source_type == 'directory':
            self.source_directory.setChecked(True)
        else:
            self.source_pdf.setChecked(True)
        self.fullscreen.setChecked(settings.fullscreen)
        # fill geometry box and load the screen geometry setting
        geometries = standard_resolutions
        desktop_geometry = Geometry(KGlobalSettings.desktopGeometry(self))
        if desktop_geometry not in geometries:
            geometries.append(desktop_geometry)
        if settings.screen_geometry not in geometries:
            geometries.append(settings.screen_geometry)
        geometries.sort(reverse=True)
        self.geometry.insertStrList([str(g) for g in geometries])
        self.geometry.setCurrentText(str(settings.screen_geometry))
        self.scale.setChecked(settings.scale)
        self.supersample.setChecked(settings.supersample)
        self.cursor_file.setURL(settings.cursor_file)
        self.wrap.setChecked(settings.wrap)
        self.auto.setChecked(settings.auto)
        self.auto_time.setValue(settings.auto_time)
        self.cache.setChecked(settings.cache)
        self.memory.setChecked(settings.memcache)
        self.background.setChecked(settings.background_render)
        self.extensions.setChecked(settings.useext)
        time = QTime.fromString(settings.duration, Qt.ISODate)
        self.duration.setChecked(time.isValid())
        self.duration_time.setTime(time)
        self.poll.setValue(settings.poll)
        selected_box = self.transitions.selectedListBox()
        available_box = self.transitions.availableListBox()
        available_box.insertStrList(self.parent().wrapper.transitions)
        for name in settings.selected:
            item = available_box.findItem(name, Qt.CaseSensitive |
                                      Qt.ExactMatch)
            if item:
                # the stored transition exists
                text = item.text()
                available_box.removeItem(available_box.index(item))
                selected_box.insertItem(text)

    def save(self, settings):
        """Saves `settings`"""
        settings.source = str(self.source.url())
        if self.source_directory.isChecked():
            settings.source_type = 'directory'
        else:
            settings.source_type = 'pdf'
        settings.poll = self.poll.value()
        settings.fullscreen = self.fullscreen.isChecked()
        settings.screen_geometry = Geometry(self.geometry.currentText())
        settings.scale = self.scale.isChecked()
        settings.supersample = self.supersample.isChecked()
        settings.cursor_file = str(self.cursor_file.url())
        settings.wrap = self.wrap.isChecked()
        settings.auto = self.auto.isChecked()
        settings.auto_time = self.auto_time.value()
        settings.cache = self.cache.isChecked()
        settings.memcache = self.memory.isChecked()
        settings.background_render = self.background.isChecked()
        settings.useext = self.extensions.isChecked()
        str_format = ''
        if self.duration.isChecked():
            str_format = str(self.duration_time.time().toString(Qt.ISODate))
        settings.duration = str_format
        selected = QStringList()
        listbox = self.transitions.selectedListBox()
        for i in xrange(0, listbox.count()):
            selected.append(listbox.text(i))
        settings.selected = selected
        

class SettingsDialog(KDialogBase):
    """Settings dialog."""

    def __init__(self, parent, settings):
        """Creates a new instance"""
        buttons = KDialogBase.Ok | KDialogBase.Apply | KDialogBase.Cancel
        KDialogBase.__init__(self, KDialogBase.Plain, i18n('Preferences'),
                             buttons, KDialogBase.Ok, parent)
        # create gui
        page = self.plainPage()
        QVBoxLayout(page).setAutoAdd(True)
        self._settingswidget = SettingsWidget(page)
        self.settings = settings
        QObject.connect(self, SIGNAL('okClicked()'), self.save)
        QObject.connect(self, SIGNAL('applyClicked()'), self.save)
        self.load()

    def load(self):
        """Loads settings"""
        self._settingswidget.executable.setURL(self.settings.kjn_executable)
        self._settingswidget.use_cmdline.setChecked(
            self.settings.use_cmdline_wrapper)

    def save(self):
        """Saves settings"""
        value = str(self._settingswidget.executable.url())
        self.settings.kjn_executable = value
        value = self._settingswidget.use_cmdline.isChecked()
        self.settings.use_cmdline_wrapper = value
        self.settings.save()

    def show(self):
        """Shows the dialog"""
        # reload settings whenever the dialog is shown
        self.load()
        KDialogBase.show(self)
