/*
    This file is part of the KDE project.
    Copyright (c) 2006-2007 Friedrich W. H. Kossebau <kossebau@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301  USA
*/

// qt specific
#include <qlayout.h>
#include <qlabel.h>
#include <qimage.h>
#include <qpopupmenu.h>
#include <qevent.h>
// kde specific
#include <klocale.h>
#include <kiconloader.h>
#include <kglobalsettings.h>
#include <kglobal.h>
// khalkhicore specific
#include <services.h>
#include <propertyadapter.h>
#include <allpropertiesglobalactionserviceclient.h>
#include <statusserviceclient.h>
// khalkhigui specific
#include <propertyallactionservicemenufiller.h>
#include <propertyalldataactionservicemenufiller.h>
#include <allpropertiesglobalactionservicemenufiller.h>
#include <allpropertiesglobaldataactionservicemenufiller.h>
// lib specific
#include "card.h"
#include "cardview.h"


namespace Khalkhi {


class GlobalServiceClientCardProxy : public AllPropertiesGlobalActionServiceClient
{
public:
    GlobalServiceClientCardProxy( CardView *V ) : View( V ) {}
    virtual ~GlobalServiceClientCardProxy() {}

public: // interface
    /** returns the person for which the services are requested */
    virtual const KABC::Addressee &person() const { return View->person(); }

public: // slots interface
    /** called if the service switched */
    virtual void onActionServiceStateChange( const PropertyActionService &/*Service*/, int /*Change*/,
                                             int /*ItemIndex*/ ) { View->updateView(); }
    virtual void onPropertyManagerChange() { View->updateView(); }
    virtual void onGlobalActionServiceSwitch( const QString &/*PropertyId*/ ) { View->updateView(); }

protected:
    CardView *View;
};


class StatusClientCardProxy : public PropertyStatusServiceClient
{
public:
    StatusClientCardProxy( CardView *V ) : View( V ) {}
    virtual ~StatusClientCardProxy() {}

public: // StatusClient interface
    virtual const KABC::Addressee &person() const { return View->person(); }
    virtual void onStateChange( const PropertyStatusService &Service, const StatusChange &Change,
                                const Status &NewStatus, int ItemIndex );
    virtual void onPropertyManagerChange()         { View->updateView(); }

protected:
    CardView *View;
};

void StatusClientCardProxy::onStateChange( const PropertyStatusService &/*Service*/,
                                           const StatusChange &/*Change*/, const Status &/*NewStatus*/, int /*ItemIndex*/ )
{ View->updateView(); }


CardView::CardView( QWidget *Parent, const char *Name )
: QTextBrowser( Parent, Name ),
  GlobalServicesProxy( new GlobalServiceClientCardProxy(this) ),
  PropertyServices( new PropertyAllActionServiceMenuFiller() ),
  Services( new AllPropertiesGlobalActionServiceMenuFiller() ),
  PropertyDropServices( new PropertyAllDataActionServiceMenuFiller() ),
  DropServices( new AllPropertiesGlobalDataActionServiceMenuFiller() ),
  StatusProxy( new StatusClientCardProxy(this) )
{
    setWrapPolicy( QTextEdit::AtWordBoundary );
    setLinkUnderline( false );
    //setVScrollBarMode( QScrollView::AlwaysOff );
    //setHScrollBarMode( QScrollView::AlwaysOff );

    styleSheet()->item( "a" )->setColor( KGlobalSettings::linkColor() );
    setFrameStyle( QTextEdit::NoFrame );

    setAcceptDrops( true );

//     connect( this, SIGNAL(highlighted( const QString& )), SLOT(onHighLighted( const QString& )) );

}

void CardView::setPerson( const KABC::Addressee &P )
{
    Services::self()->unregisterClient( GlobalServicesProxy );
    Services::self()->unregisterClient( StatusProxy );

    Person = P;

    Services::self()->registerClient( StatusProxy );
    Services::self()->registerClient( GlobalServicesProxy );

    updateView();
}


void CardView::updateView()
{
    setText( Person.isEmpty() ? QString::null : createCard() );
}


void CardView::setSource( const QString &Link )
{
    QStringList L = QStringList::split( ":", Link, true );
    if( L[1].startsWith("//") )
        L[1] = L[1].mid(2);

    Services::self()->execute( Person, L[1], L[2].toInt(), L[0] );
}


QPopupMenu *CardView::createPopupMenu( const QPoint& Pos )
{
    // for card content
    QPopupMenu *Menu = QTextBrowser::createPopupMenu( Pos );

    // add kde icon
    int MenuId = Menu->idAt( 0 );
    Menu->changeItem( MenuId, SmallIconSet("editcopy"), Menu->text(MenuId) );

    Menu->insertSeparator();

    // add type services
    QString Link = anchorAt( Pos, AnchorHref );
    if( Link.isEmpty() )
        Link = anchorAt( Pos, AnchorName );

    int Inserted;
    if( !Link.isEmpty() )
    {
        QStringList L = QStringList::split( ":", Link, true );
        PropertyServices->set( Person, L[1], L[2].toInt() );

        Inserted = PropertyServices->fillMenu( Menu,Menu->count() );
    }
    else
    {
        Services->set( Person );
        Inserted = Services->fillMenu( Menu, Menu->count() );
    }
    if( Inserted > 0 )
        Menu->insertSeparator();

    return Menu;
}


static QString createLink( const QString &PropertyId, const QString &Text, int Id, const QString &ServiceId )
{
    QString Link = QString::fromLatin1( "%1:%2:%3" ).arg( ServiceId, PropertyId, QString::number(Id) );
    return QString::fromLatin1( "<a %1=\"%2\">%3</a>" ).arg(
        QString::fromLatin1(ServiceId.isNull()?"name":"href"), Link, Text );
}

static void fill( Card *Card, const KABC::Addressee &Person )
{
    const QString &UID = Person.uid();
    const PropertyManagerList &Managers = Services::self()->propertyManagers();

    PropertyManagerList::ConstIterator ManagerIt = Managers.begin();
    for( ; ManagerIt != Managers.end(); ++ManagerIt )
    {
        const PropertyAdapter *Adapter = (*ManagerIt)->propertyAdapter();
        const PropertyActionServiceList &ActionServices = (*ManagerIt)->mainActionServices();
        const PropertyStatusServiceList &StatusServices = (*ManagerIt)->statusServices();
        const QString &PropertyId = Adapter->id();

        const int ItemSize = Adapter->numberOfItems( Person );
        for( int ItemIndex = 0; ItemIndex<ItemSize; ++ItemIndex )
        {
            QString ItemId = QString::number( ItemIndex );
            // get data
            PropertyItem PropertyItem = Adapter->propertyItemOf( Person, ItemIndex );
            QString LabelText = PropertyItem.data( LabelTextRole ).asString();
            QString ItemText = PropertyItem.data( DisplayTextRole ).asString();

            // find first global action
            QString ActionServiceId;
#if 0
// TODO: before we searched for the first service that supported the item (better for fax or phone with number)
            if( ActionService->isAvailableFor(Person,ItemIndex) )
                ActionServiceId = ActionService->id();
#endif
            for( PropertyActionServiceList::ConstIterator ActionServiceIt = ActionServices.begin();
                 ActionServiceIt != ActionServices.end(); ++ActionServiceIt )
                if( (*ActionServiceIt)->supports(Person,ItemIndex) )
                {
                    // get id if it's available
                    if( (*ActionServiceIt)->isAvailableFor(Person,ItemIndex) )
                        ActionServiceId = (*ActionServiceIt)->id();
                    break;
                }

            ItemText = createLink( PropertyId, ItemText, ItemIndex, ActionServiceId );

            // get status
            QString StatusText;
            PropertyStatusServiceList::ConstIterator StatusServiceIt = StatusServices.begin();
            for( ; StatusServiceIt != StatusServices.end(); ++StatusServiceIt )
            {
                if( !(*StatusServiceIt)->supports(Person,ItemIndex) )
                    continue;

                Status Status = (*StatusServiceIt)->status( Person, ItemIndex, 0 );
                QString Text = Status.data( DisplayTextRole ).asString();
                QImage Icon = Status.data( DisplayIconRole ).asImage();

                // enrich text with icon
                if( !Icon.isNull() )
                {
                    QString IconId = QString::fromLatin1( "%1_tipicon_%2_%3_%4" )
                      .arg( (*StatusServiceIt)->id(), UID, PropertyId, ItemId );
                    QMimeSourceFactory::defaultFactory()->setImage( IconId, Icon );
                    Text = QString::fromLatin1( "<img src=\"%1\">&nbsp;%2" ).arg( IconId, Text );
                }

                StatusText.append( Text );
            }

            // add to card
            Card->appendItem( LabelText, ItemText, StatusText );
        }
    }
}

QString CardView::createCard()
{
    if( Person.isEmpty() )
        return QString::null;

    Card Card;

    Card.initiate();

    QString ImageURL = QString::fromLatin1( "person_image_%1" ).arg( Person.uid() );
    KABC::Picture Photo = Person.photo();
    if ( Photo.isIntern() && !Photo.data().isNull() ) // TODO: care for extern pictures
        QMimeSourceFactory::defaultFactory()->setImage( ImageURL, Photo.data() );
    else
        QMimeSourceFactory::defaultFactory()->setPixmap( ImageURL,
            KGlobal::iconLoader()->loadIcon("personal",KIcon::NoGroup,50) );

    Card.appendHeader( ImageURL, Person.realName(), Person.role(), Person.organization() );

    fill( &Card, Person );

    Card.finalize();

    return Card.data();
}


void CardView::contentsDragEnterEvent( QDragEnterEvent *Event )
{
    //if( CardDropServices->canHandle(Event) )
        Event->accept();
}


void CardView::contentsDragLeaveEvent( QDragLeaveEvent */*Event*/ )
{
}


void CardView::contentsDragMoveEvent( QDragMoveEvent *Event )
{
    QPoint Pos = Event->pos();

    bool Dropable;

    // update drop options
    QString Link = anchorAt( Pos, AnchorHref );
    if( Link.isEmpty() )
        Link = anchorAt( Pos, AnchorName );

    if( !Link.isEmpty() )
    {
        QStringList L = QStringList::split( ":", Link, true );
        PropertyDropServices->set( Person, L[1], L[2].toInt(), Event );
        Dropable = PropertyDropServices->serviceAvailableForData();
    }
    else
    {
        DropServices->set( Person, Event );
        Dropable = DropServices->serviceAvailableForData();
    }

    Event->accept( Dropable );
}


void CardView::contentsDropEvent( QDropEvent *Event )
{
    // for card content
    QPopupMenu *Menu = new QPopupMenu();

    // add type services
    QPoint Pos = Event->pos();
    QString Link = anchorAt( Pos, AnchorHref );
    if( Link.isEmpty() )
        Link = anchorAt( Pos, AnchorName );

    int Inserted;
    if( !Link.isEmpty() )
    {
        QStringList L = QStringList::split( ":", Link, true );
        PropertyDropServices->set( Person, L[1], L[2].toInt(), Event );
        Inserted = PropertyDropServices->fillMenu( Menu, 0 );
    }
    else
    {
        DropServices->set( Person, Event );
        Inserted = DropServices->fillMenu( Menu, 0 );
    }

    if( Inserted > 0 )
    {
        Menu->insertSeparator();
        Menu->insertItem( SmallIcon("cancel"), i18n("&Cancel") );

        Menu->exec( mapToGlobal(Event->pos()) );
    }
    delete Menu;
}
#if 0
void CardView::onHighLighted( const QString &Link )
{
    QStringList L = QStringList::split( ":", Link, true );
    if( L[1].startsWith("//") )
        L[1] = L[1].mid(2);

   //TODO: get service entry
   // Services::self()->execute( Person, L[1], L[2].toInt(), L[0] );

    emit actionHighlighted( ActionText );
}
#endif
CardView::~CardView()
{
    Services::self()->unregisterClient( GlobalServicesProxy );
    Services::self()->unregisterClient( StatusProxy );

    delete GlobalServicesProxy;
    delete PropertyServices;
    delete PropertyDropServices;
    delete Services;
    delete DropServices;
    delete StatusProxy;
}

}

#include "cardview.moc"
