/* $Id: we_input.c,v 1.13 2007/07/23 21:53:25 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <libintl.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gmodule.h>

#include "simocombobox.h"

#include "kildclient.h"
#include "perlscript.h"


/***********************
 * Function prototypes *
 ***********************/
static void          multi_line_toggled_cb(GtkToggleButton *button,
                                           gpointer         data);
static void          input_size_changed_cb(GtkSpinButton *button,
                                           gpointer       data);
static void          entryfont_set_cb(GtkFontButton *font_btn,
                                      gpointer data);
/* Glade signals */
G_MODULE_EXPORT void flood_prevention_toggled_cb(GtkToggleButton *chkbox,
                                                 gpointer data);



GtkWidget *
get_we_input_tab(GladeXML *gladexml)
{
  GtkWidget *panel;

  panel    = glade_xml_get_widget(gladexml, "we_panel_input");

  return panel;
}


void
fill_we_input_tab(World *world, GladeXML *gladexml)
{
  GtkWidget *chkRepeatCommands;
  GtkWidget *chkCmdEcho;
  GtkWidget *chkStoreCommands;
  GtkWidget *spnCommandsToSave;
  GtkWidget *txtCommandSeparator;
  GtkWidget *radSingleLine;
  GtkWidget *radMultiLine;
  GtkWidget *spnNLines;
  GtkWidget *chkAutoCompletion;
  GtkWidget *spnMinPrefix;
  GtkWidget *chkSpell;
  GtkWidget *txtSpellLanguage;
  GtkWidget *entryfont_btn;
  GtkWidget *chkFloodPrevention;
  GtkWidget *spnMaxCommands;
  GtkWidget *txtPreventionCommand;

  if (!world) {
    return;
  }

  /* Get widgets */
  chkRepeatCommands    = glade_xml_get_widget(gladexml, "chkRepeatCommands");
  chkCmdEcho           = glade_xml_get_widget(gladexml, "chkCmdEcho");
  chkStoreCommands     = glade_xml_get_widget(gladexml, "chkStoreCommands");
  spnCommandsToSave    = glade_xml_get_widget(gladexml, "spnCommandsToSave");
  radSingleLine        = glade_xml_get_widget(gladexml, "radSingleLine");
  radMultiLine         = glade_xml_get_widget(gladexml, "radMultiLine");
  chkAutoCompletion    = glade_xml_get_widget(gladexml, "chkAutoCompletion");
  spnMinPrefix         = glade_xml_get_widget(gladexml, "spnMinPrefix");
  spnNLines            = glade_xml_get_widget(gladexml, "spnNLines");
  chkSpell             = glade_xml_get_widget(gladexml, "chkSpell");
  txtSpellLanguage     = glade_xml_get_widget(gladexml, "txtSpellLanguage");
  txtCommandSeparator  = glade_xml_get_widget(gladexml, "txtCommandSeparator");
  entryfont_btn        = glade_xml_get_widget(gladexml, "entryfont_btn");
  chkFloodPrevention   = glade_xml_get_widget(gladexml, "chkFloodPrevention");
  spnMaxCommands       = glade_xml_get_widget(gladexml, "spnMaxCommands");
  txtPreventionCommand = glade_xml_get_widget(gladexml,
                                              "txtPreventionCommand");

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkRepeatCommands),
                               world->repeat_commands);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkCmdEcho),
                               world->cmd_echo);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkStoreCommands),
			       world->store_commands);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnCommandsToSave),
                            world->commands_to_save);
  gtk_entry_set_text(GTK_ENTRY(txtCommandSeparator),
                     world->command_separator);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnNLines),
                            world->input_n_lines_saved);
  set_input_line_controls(world, radSingleLine, radMultiLine, spnNLines);
  g_signal_connect(G_OBJECT(radSingleLine), "toggled",
                   G_CALLBACK(multi_line_toggled_cb), world);
  g_signal_connect(G_OBJECT(radMultiLine), "toggled",
                   G_CALLBACK(multi_line_toggled_cb), world);
  g_signal_connect(G_OBJECT(spnNLines), "value-changed",
                   G_CALLBACK(input_size_changed_cb), world);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkAutoCompletion),
                               world->autocompletion);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnMinPrefix),
                            world->autocompl_minprefix);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkSpell),
                               world->spell);
  if (world->spell_language) {
    gtk_entry_set_text(GTK_ENTRY(txtSpellLanguage), world->spell_language);
  }

  gtk_font_button_set_font_name(GTK_FONT_BUTTON(entryfont_btn),
                                world->entryfont);
  g_signal_connect(G_OBJECT(entryfont_btn), "font-set",
                   G_CALLBACK(entryfont_set_cb), world);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(chkFloodPrevention),
                               world->flood_prevention);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnMaxCommands),
                            world->max_equal_commands);
  gtk_widget_set_sensitive(spnMaxCommands, world->flood_prevention);
  gtk_widget_set_sensitive(txtPreventionCommand, world->flood_prevention);
  if (world->flood_prevention_command) {
    gtk_entry_set_text(GTK_ENTRY(txtPreventionCommand),
                       world->flood_prevention_command);
  }
}



void
update_we_input_parameters(World *world)
{
  GladeXML  *gladexml;
  GtkWidget *chkRepeatCommands;
  GtkWidget *chkCmdEcho;
  GtkWidget *chkFloodPrevention;
  GtkWidget *spnMaxCommands;
  GtkWidget *txtPreventionCommand;
  GtkWidget *chkStoreCommands;
  GtkWidget *spnCommandsToSave;
  GtkWidget *chkAutoCompletion;
  GtkWidget *spnMinPrefix;
  GtkWidget *chkSpell;
  GtkWidget *txtSpellLanguage;
  GtkWidget *txtCommandSeparator;

  GtkEntryCompletion *completion;

  const gchar *newval;

  /* Get widgets */
  gladexml             = glade_get_widget_tree(world->dlgEditWorld);
  chkRepeatCommands    = glade_xml_get_widget(gladexml, "chkRepeatCommands");
  chkCmdEcho           = glade_xml_get_widget(gladexml, "chkCmdEcho");
  chkStoreCommands     = glade_xml_get_widget(gladexml, "chkStoreCommands");
  spnCommandsToSave    = glade_xml_get_widget(gladexml, "spnCommandsToSave");
  chkAutoCompletion    = glade_xml_get_widget(gladexml, "chkAutoCompletion");
  spnMinPrefix         = glade_xml_get_widget(gladexml, "spnMinPrefix");
  chkSpell             = glade_xml_get_widget(gladexml, "chkSpell");
  txtSpellLanguage     = glade_xml_get_widget(gladexml, "txtSpellLanguage");
  txtCommandSeparator  = glade_xml_get_widget(gladexml, "txtCommandSeparator");
  chkFloodPrevention   = glade_xml_get_widget(gladexml, "chkFloodPrevention");
  spnMaxCommands       = glade_xml_get_widget(gladexml, "spnMaxCommands");
  txtPreventionCommand = glade_xml_get_widget(gladexml,
                                              "txtPreventionCommand");

  world->repeat_commands =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkRepeatCommands));
  world->cmd_echo =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkCmdEcho));

  world->store_commands =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkStoreCommands));
  world->commands_to_save =
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spnCommandsToSave));

  world->autocompletion =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkAutoCompletion));
  world->autocompl_minprefix =
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spnMinPrefix));
  if (world->gui) {
    completion = simo_combo_box_get_completion(world->gui->cmbEntry);
    if (world->autocompletion) {
      gtk_entry_completion_set_minimum_key_length(completion,
                                                  world->autocompl_minprefix);
    } else {
      gtk_entry_completion_set_minimum_key_length(completion,
                                                  9999);
    }
  }

  world->spell = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkSpell));
  g_free(world->spell_language);
  newval = gtk_entry_get_text(GTK_ENTRY(txtSpellLanguage));
  if (strcmp(newval, "") != 0) {
    world->spell_language = g_strdup(newval);
  } else {
    world->spell_language = NULL;
  }
  if (world->gui) {
    worldgui_configure_spell(world->gui);
  }

  strcpy(world->command_separator,
         gtk_entry_get_text(GTK_ENTRY(txtCommandSeparator)));

  world->flood_prevention =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chkFloodPrevention));
  world->max_equal_commands =
    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spnMaxCommands));

  newval = gtk_entry_get_text(GTK_ENTRY(txtPreventionCommand));
  if (!world->flood_prevention_command
      || strcmp(world->flood_prevention_command, newval) != 0) {
    g_free(world->flood_prevention_command);
    world->flood_prevention_command = g_strdup(newval);
  }
}


static
void
multi_line_toggled_cb(GtkToggleButton *button, gpointer data)
{
  World     *world  = (World *) data;
  GladeXML  *gladexml;
  GtkWidget *radSingleLine;

  /* Avoid infinite loop */
  if (world->updating_controls) {
    return;
  }

  gladexml      = glade_get_widget_tree(GTK_WIDGET(button));
  radSingleLine = glade_xml_get_widget(gladexml, "radSingleLine");

  /* If there is a gui, setting the size generates a signal that is caught,
     and world->input_n_lines is updated in that handler. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radSingleLine))) {
    if (world->gui) {
      simo_combo_box_set_n_lines(world->gui->cmbEntry, 1);
    } else {
      world->input_n_lines = 1;
    }
  } else {
    if (world->gui) {
      simo_combo_box_set_n_lines(world->gui->cmbEntry,
                                 world->input_n_lines_saved);
    } else {
      world->input_n_lines = world->input_n_lines_saved;
    }
  }
}


static
void
input_size_changed_cb(GtkSpinButton *button, gpointer data)
{
  World     *world  = (World *) data;
  GladeXML  *gladexml;
  GtkWidget *radMultiLine;

  /* Avoid infinite loop */
  if (world->updating_controls) {
    return;
  }

  gladexml     = glade_get_widget_tree(GTK_WIDGET(button));
  radMultiLine = glade_xml_get_widget(gladexml, "radMultiLine");

  world->input_n_lines_saved = gtk_spin_button_get_value_as_int(button);
  if (world->input_n_lines_saved > 10) {
    world->input_n_lines_saved = 10;
  }
  if (world->input_n_lines_saved < 2) {
    world->input_n_lines_saved = 2;
  }

  if (world->gui) {
    simo_combo_box_set_n_lines(world->gui->cmbEntry,
                               world->input_n_lines_saved);
  } else {
    world->input_n_lines = world->input_n_lines_saved;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radMultiLine), TRUE);
}


static
void
entryfont_set_cb(GtkFontButton *font_btn, gpointer data)
{
  PangoFontDescription *fontDesc;
  World                *world  = (World *) data;
  WorldGUI             *gui    = world->gui;

  free(world->entryfont);
  world->entryfont = g_strdup(gtk_font_button_get_font_name(font_btn));

  if (gui) {
    fontDesc = pango_font_description_from_string(world->entryfont);
    simo_combo_box_set_entry_font(gui->cmbEntry, fontDesc);
    pango_font_description_free(fontDesc);
  }
}


void
flood_prevention_toggled_cb(GtkToggleButton *chkbox, gpointer data)
{
  GladeXML  *gladexml;
  GtkWidget *spnMaxCommands;
  GtkWidget *txtPreventionCommand;
  gboolean   enabled;

  /* Get widgets */
  gladexml             = glade_get_widget_tree(GTK_WIDGET(chkbox));
  spnMaxCommands       = glade_xml_get_widget(gladexml, "spnMaxCommands");
  txtPreventionCommand = glade_xml_get_widget(gladexml,
                                              "txtPreventionCommand");
  enabled = gtk_toggle_button_get_active(chkbox);

  gtk_widget_set_sensitive(spnMaxCommands, enabled);
  gtk_widget_set_sensitive(txtPreventionCommand, enabled);
}


void
set_input_line_controls(World *world,
                        GtkWidget *radSingleLine,
                        GtkWidget *radMultiLine,
                        GtkWidget *spnNLines)
{
  GladeXML *gladexml;

  /* If the controls have not been passed as argument. */
  if (!radSingleLine) {
    if (!world->dlgEditWorld) {
      return;
    }

    gladexml = glade_get_widget_tree(world->dlgEditWorld);
    radSingleLine = glade_xml_get_widget(gladexml, "radSingleLine");
    radMultiLine = glade_xml_get_widget(gladexml, "radMultiLine");
    spnNLines = glade_xml_get_widget(gladexml, "spnNLines");
  }

  world->updating_controls = TRUE;
  if (world->input_n_lines == 1) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radSingleLine), TRUE);
  } else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radMultiLine), TRUE);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spnNLines),
                              world->input_n_lines);
  }
  world->updating_controls = FALSE;
}
