#include <string.h>
#include <time.h>

#include <gconf/gconf-client.h>
#include <glade/glade.h>
#include <gtk/gtk.h>

#include "kphtmloutputdialog.h"
#include "kpmainwindow.h"
#include "kpguiutils.h"

#include "../html/kphtmlwriter.h" 
#include "../kptraininglog.h"
#include "../kipina-i18n.h"

static void kp_html_output_dialog_class_init  (KPHtmlOutputDialogClass *klass);
static void kp_html_output_dialog_init        (KPHtmlOutputDialog *dialog);
static void kp_html_output_dialog_finalize    (GObject *object);
static void output_folder_browse_clicked_cb   (GtkButton *button, 
                                               KPHtmlOutputDialog *dialog);
static void gen_button_clicked_cb             (GtkButton *button, 
                                               KPHtmlOutputDialog *dialog);

typedef struct KPHtmlOutputDialogPrivateData_
{
  GtkWidget *output_folder_button;
  GtkWidget *output_folder_entry;
  
  GtkWidget *cancel_button;
  GtkWidget *gen_button;
} KPHtmlOutputDialogPrivateData;

#define KP_HTML_OUTPUT_DIALOG_PRIVATE_DATA(widget) (((KPHtmlOutputDialogPrivateData*) \
        (KP_HTML_OUTPUT_DIALOG (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_html_output_dialog_get_type (void)
{
  static GType kp_html_output_dialog_type = 0;

  if (kp_html_output_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPHtmlOutputDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_html_output_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPHtmlOutputDialog),
      0,
      (GInstanceInitFunc) kp_html_output_dialog_init,
      NULL,
    };

    kp_html_output_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                         "KPHtmlOutputDialog",
                                                         &our_info, 0);
  }

  return kp_html_output_dialog_type;
}

static void
kp_html_output_dialog_class_init (KPHtmlOutputDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_html_output_dialog_finalize;
}



static void
kp_html_output_dialog_init (KPHtmlOutputDialog *dialog)
{
  KPHtmlOutputDialogPrivateData *p_data;
  GConfClient *client;
  gchar *last_dir;
  GladeXML *xml;

  xml = kp_gui_load ("html_output_dialog", "html_output_dialog");

  dialog->private_data = g_new0 (KPHtmlOutputDialogPrivateData, 1);
  p_data = KP_HTML_OUTPUT_DIALOG_PRIVATE_DATA (dialog);
  
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Html Statistics"));

  p_data->cancel_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                              GTK_STOCK_CANCEL,
                                              GTK_RESPONSE_CANCEL);

  p_data->gen_button = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                              GTK_STOCK_EXECUTE,
                                              GTK_RESPONSE_OK);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     KP_W (xml, "html_output_dialog"));
  
  /* Put widgets to private data */
  p_data->output_folder_button = KP_W (xml, "output_folder_button");
  p_data->output_folder_entry = KP_W (xml, "output_folder_entry");

  g_signal_connect (G_OBJECT (p_data->gen_button), "clicked",
                    G_CALLBACK (gen_button_clicked_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->output_folder_button), "clicked",
                    G_CALLBACK (output_folder_browse_clicked_cb), dialog);
  g_signal_connect_swapped (G_OBJECT (p_data->cancel_button), "clicked",
                            G_CALLBACK (gtk_widget_destroy), dialog);
  
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
 
  client = gconf_client_get_default ();
  last_dir = gconf_client_get_string (client, KP_CONF_HTML_OUTPUT_DIR, NULL);

  if (last_dir != NULL)
    gtk_entry_set_text (GTK_ENTRY (p_data->output_folder_entry), last_dir);
  
  g_object_unref (G_OBJECT (xml));
}


static void
kp_html_output_dialog_finalize (GObject *object)
{
  KPHtmlOutputDialog *dialog;
  g_return_if_fail (KP_IS_HTML_OUTPUT_DIALOG (object));
  dialog = KP_HTML_OUTPUT_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

  

GtkWidget *
kp_html_output_dialog_new (void)
{
  KPHtmlOutputDialogPrivateData *p_data;
  KPHtmlOutputDialog *dialog;

  dialog = g_object_new (KP_TYPE_HTML_OUTPUT_DIALOG, NULL);

  p_data = KP_HTML_OUTPUT_DIALOG_PRIVATE_DATA (dialog);
  
  return GTK_WIDGET (dialog);
}

 
static void
gen_button_clicked_cb (GtkButton *button, KPHtmlOutputDialog *dialog)
{
  KPHtmlOutputDialogPrivateData *p_data;
  GConfClient *client;
  KPStatusbar *sbar;
  KPHtmlWriter *writer;
  KPTrainingLog *log;
  const gchar *dir;
  GError *error;
  
  p_data = KP_HTML_OUTPUT_DIALOG_PRIVATE_DATA (dialog);

  log = kp_main_window_get_log ();
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  if (kp_training_log_get_size (log) == 0) {
    kp_gui_err_message (GTK_WINDOW (dialog), "Log is empty!");
    return;
  }
    
  dir = gtk_entry_get_text (GTK_ENTRY (p_data->output_folder_entry));
 
  writer = kp_html_writer_new ();
  error = NULL;
   
  if (kp_html_writer_write (writer, log, dir, &error) == FALSE) {
    kp_gui_err_message (GTK_WINDOW (dialog), 
                       "Generating Html Statistics failed!\n(%s)",
                       (error->message) ? error->message : _("Unknown error"));

    if (error)
      g_error_free (error);
  } else {
    /* Remember the dir next time */
    client = gconf_client_get_default ();
    gconf_client_set_string (client, KP_CONF_HTML_OUTPUT_DIR, dir, NULL);
       
    sbar = kp_main_window_get_statusbar ();
    kp_statusbar_set_format_message (sbar,
                                   _("HTML statistics wrote: %s/index.html"), 
                                     dir);
    gtk_widget_destroy (GTK_WIDGET (dialog));
  }
}


static void
output_folder_browse_clicked_cb (GtkButton *button, KPHtmlOutputDialog *dialog)
{
  KPHtmlOutputDialogPrivateData *p_data;
  gchar *dir;
  
  p_data = KP_HTML_OUTPUT_DIALOG_PRIVATE_DATA (dialog);

  dir = kp_gui_get_dir (GTK_WINDOW (dialog));

  if (dir == NULL)
    return;

  gtk_entry_set_text (GTK_ENTRY (p_data->output_folder_entry), dir);
  g_free (dir);
}

