#include <string.h>

#include <glib.h>
#include <glib-object.h>

#include "kpcalendarentry.h"
#include "kptraininglog.h"
#include "kipina-i18n.h"
#include "kpcomment.h"
#include "kputil.h"

static void       kp_comment_class_init          (GObjectClass *klass,
                                                  gpointer data);
static void       kp_comment_instance_init       (GObject *object,
                                                  gpointer data);
static void       kp_comment_instance_finalize   (GObject *object);

/* KPCalendarEntry virtual function implementations */
static gchar     *kp_comment_get_human_name      (KPCalendarEntry *entry);
static gchar     *kp_comment_get_icon_name       (KPCalendarEntry *entry);
G_CONST_RETURN gchar *
                  kp_comment_to_calendar_string  (KPCalendarEntry *entry);
xmlNodePtr        kp_comment_to_xml              (KPCalendarEntry *entry);
static gboolean   kp_comment_parse               (KPCalendarEntry *entry,
                                                  xmlNodePtr node);



GType
kp_comment_get_type ()
{
  static GType kp_comment_type = 0;

  if (!kp_comment_type) {
    static const GTypeInfo kp_comment_info = {
      sizeof (KPCommentClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_comment_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPComment),
      0,
      (GInstanceInitFunc) kp_comment_instance_init,
      NULL
    };

    kp_comment_type = g_type_register_static (KP_TYPE_CALENDAR_ENTRY,
                                             "KPComment",
                                             &kp_comment_info,
                                              0);
  }
  
  return kp_comment_type;
}



static void
kp_comment_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;
  KPCalendarEntryClass *entry_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_comment_instance_finalize;

  entry_class = KP_CALENDAR_ENTRY_CLASS (klass);
  entry_class->get_human_name = kp_comment_get_human_name;
  entry_class->get_icon_name = kp_comment_get_icon_name;
  entry_class->to_string = kp_comment_to_calendar_string;
  entry_class->to_xml = kp_comment_to_xml;
  entry_class->parse = kp_comment_parse;
}

static void
kp_comment_instance_init (GObject *object, gpointer data)
{
  KPComment *comment;

  comment = KP_COMMENT (object);

  comment->title = g_string_new (NULL);
  comment->text = g_string_new (NULL);
}


static void
kp_comment_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  KPComment *comment = KP_COMMENT (object);

  if (comment->title)
    g_string_free (comment->title, TRUE);
  
  if (comment->text)
    g_string_free (comment->text, TRUE);

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}


/**
 * kp_comment_new:
 * @title: The title of the comment
 * @text: The comment
 * 
 * Create a new instance of #KPComment.
 *
 * Returns: A #KPComment 
 */
KPComment *
kp_comment_new (const gchar *title, const gchar *text)
{
  GObject *comment;
  
  comment = g_object_new (kp_comment_get_type (), NULL);

  kp_comment_set_title (KP_COMMENT (comment), title);
  kp_comment_set_text (KP_COMMENT (comment), text);
  
  return KP_COMMENT (comment);
}

  
KPComment *
kp_comment_copy (KPComment *comment)
{
  KPCalendarTime *ct_copy;
  KPCalendarTime *ct;
  KPComment *copy;
  KPDate date;

  g_return_val_if_fail (KP_IS_COMMENT (comment), NULL);

  copy = kp_comment_new (kp_comment_get_title (comment),
                         kp_comment_get_text (comment));

  ct_copy = KP_CALENDAR_ENTRY (copy)->datetime;
  ct = KP_CALENDAR_ENTRY (comment)->datetime;

  kp_calendar_time_to_kpdate (ct, &date);
  kp_calendar_time_set_dmy (ct_copy, date.d, date.m, date.y);
  kp_calendar_time_set_hmst (ct_copy, ct->h, ct->m, ct->s, ct->t);
  
  return copy;
}


/**
 * kp_comment_set_title:
 * @comment: A #KPComment
 * @title: The title to set
 *
 * Set the title of the comment.
 */
void
kp_comment_set_title (KPComment *comment, const gchar *title)
{
  g_return_if_fail (KP_IS_COMMENT (comment));
  if (title)
    g_string_assign (comment->title, title);
  else
    g_string_assign (comment->title, "");
}


/**
 * kp_comment_set_text:
 * @comment: A #KPComment
 * @text: The body of the comment
 *
 * Set the body of the comment to @text.
 */
void
kp_comment_set_text (KPComment *comment, const gchar *text)
{
  g_return_if_fail (KP_IS_COMMENT (comment));
  if (text)
    g_string_assign (comment->text, text);
  else
    g_string_assign (comment->text, "");
}


/**
 * kp_comment_get_text:
 * @comment: A #KPComment
 *
 * Retrieve the body of the comment.
 *
 * Returns: A newly-allocated string that must be freed or NULL.
 */
gchar *
kp_comment_get_text (KPComment * comment)
{
  g_return_val_if_fail (KP_IS_COMMENT (comment), NULL);
  g_return_val_if_fail (comment->text != NULL, NULL);

  return g_strdup (comment->text->str);
}

 
/**
 * kp_comment_get_title:
 * @comment: A #KPComment
 *
 * Retrieve the title of the comment.
 *
 * Returns: A newly-allocated string that must be freed or NULL.
 */
gchar *
kp_comment_get_title (KPComment * comment)
{
  g_return_val_if_fail (KP_IS_COMMENT (comment), NULL);
  g_return_val_if_fail (comment->title != NULL, NULL);

  return g_strdup (comment->title->str);
}


/**
 * get_output_format_date (KPComment *wo)
 * @wo: a KPComment.
 *
 * Returns date in format CCYY-MM-DDThh:mm:ss. Result-string
 * is malloc'd, so it must be freed.
 **/
gchar *
kp_comment_get_formatted_date (KPComment *comment)
{
  KPDate date;

  g_return_val_if_fail (comment != NULL, NULL);
  g_return_val_if_fail (KP_CALENDAR_ENTRY (comment)->datetime != NULL, NULL);

  kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (comment), &date);
  g_return_val_if_fail (kp_date_valid (&date), NULL);

  return g_strdup_printf ("%.4d-%.2d-%.2dT%.2d:%.2d:%.2d", 
                           date.y, date.m, date.d,
                           KP_CALENDAR_ENTRY (comment)->datetime->h,
                           KP_CALENDAR_ENTRY (comment)->datetime->m,
                           KP_CALENDAR_ENTRY (comment)->datetime->s);
}


gchar *
kp_comment_get_icon_name (KPCalendarEntry *entry)
{
  return g_strdup ("comment.png");
}

/**
 * kp_comment_to_calendar_string:
 * @entry: A #KPCalendarEntry
 *
 * Get #KPComment as a string that can be passed directly to CalendarView or 
 * something like that.
 *
 * Returns: a newly-allocated string that must be freed by the caller.
 */
G_CONST_RETURN gchar *
kp_comment_to_calendar_string (KPCalendarEntry *entry)
{
  gchar *title;
  gchar *text;
  gchar *str;
  
  g_return_val_if_fail (KP_IS_COMMENT (entry), NULL);
  g_return_val_if_fail (KP_COMMENT (entry)->title != NULL, NULL);
  g_return_val_if_fail (KP_COMMENT (entry)->text != NULL, NULL);
 
  title = g_markup_escape_text (KP_COMMENT (entry)->title->str, -1);
  text = g_markup_escape_text (KP_COMMENT (entry)->text->str, -1);
  
  str = g_strdup_printf ("<b>%s</b>\n%s", title, text);

  g_free (title);
  g_free (text);

  return str;
}

static gchar *
kp_comment_get_human_name (KPCalendarEntry *entry)
{
  return g_strdup (_("Comment"));
}

/**
 * kp_comment_to_xml:
 * @entry: A #KPCalendarEntry
 *
 * Turns the #KPComment structure to the xmlNodePtr structure.
 *
 * The XML format looks like this:
 * <comment>
 *   <date>1012202-T0:02:00</date>
 *   <title>Otsikker</title>
 *   <text>Dumdidum</text>
 * </comment>
 *
 * Returns: A #xmlNodePtr
 */
xmlNodePtr
kp_comment_to_xml (KPCalendarEntry *entry)
{
  xmlNodePtr node;
  gchar *date;
  gchar *esc[2];
  
  date = kp_comment_get_formatted_date (KP_COMMENT (entry));

  node = xmlNewNode (NULL, BAD_CAST ("comment"));

  esc[0] = g_markup_escape_text (KP_COMMENT (entry)->title->str, -1);
  esc[1] = g_markup_escape_text (KP_COMMENT (entry)->text->str, -1);
  
  (void) xmlNewChild (node, NULL, BAD_CAST ("datetime"), BAD_CAST (date));
  (void) xmlNewChild (node, NULL, BAD_CAST ("title"), BAD_CAST (esc[0]));
  (void) xmlNewChild (node, NULL, BAD_CAST ("text"), BAD_CAST (esc[1]));

  g_free (esc[0]);
  g_free (esc[1]);
  
  return node;
}


/**
 * kp_comment_parse:
 * @entry: A #KPCalendarEntry
 * @node: A #xmlNodePtr
 *
 * Parses a workout and adds it to log. If kp_workout's type
 * is interval, interval-params are stored to params-hashtable.
 * Interval-things' keys use "__interval__"-prefix to prevent
 * name-collisions which should be impossible this way.
 *
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_comment_parse (KPCalendarEntry *entry, xmlNodePtr node)
{
  KPComment *comment;
  xmlNodePtr child;
  gchar *date;
  gchar *title;
  gchar *text;

  
  comment = KP_COMMENT (entry);
  g_return_val_if_fail (KP_IS_COMMENT (comment), FALSE);
  
  child = node->children;

  while (child) {

    /** <datetime> **/
    if (KP_TAG_MATCH (child, "datetime")) {
      date = (gchar *) xmlNodeGetContent (child);
      if (!kp_calendar_time_set_datetime (KP_CALENDAR_ENTRY (comment)->datetime, 
                                          date)) {
        g_free (date);
        g_return_val_if_reached (FALSE);
      }
      g_free (date);
    }
    /** <comment> **/
    else if (KP_TAG_MATCH (child, "title")) {
      title = (gchar *) xmlNodeGetContent (child);
      kp_comment_set_title (comment, title);
      g_free (title);
    }
    else if (KP_TAG_MATCH (child, "text") && child->type == XML_ELEMENT_NODE) {
      text = (gchar *) xmlNodeGetContent (child);

      kp_comment_set_text (comment, text);
      g_free (text);
    }
    
    child = child->next;
  } 
  title = kp_comment_get_title (comment);

  if (!title || !title[0]) {
    g_free (title);
    g_object_unref (comment);

    g_return_val_if_reached (FALSE);
  }

  return TRUE; 
}


