#ifndef __Wire_CLASS__
#define __Wire_CLASS__

#include <klogicList.h>
#include <device.h>
#include <qpixmap.h>
#include "xdevice.h"

/********************************************
* node handling of all nodes of a wire 
*********************************************/
class Wire
{
public:
	typedef enum IO_TYPE {INPUT, OUTPUT, BOTH};

	Wire();
	Wire(QPoint);
	virtual ~Wire();

	// draw wire
	void drawImage(QPainter *);
	void drawConnections(QPainter *, IO_TYPE, int = 0);
	void setColor(QColor);

	// called by XDevice
	// modify input- and output-nodes by devices (device-move)
	void setInputPosition(QPoint);
	void setInputPosition(QPoint, int);
	QPoint getInputPosition(int idx, int selonly = 0);
	void setOutputPosition(QPoint);
	void setOutputPosition(QPoint, int);
	QPoint getOutputPosition(int idx, int selonly = 0);
	// get io-position of connected wire
	QPoint getIOPosition(int idx, int selonly = 0);

	// called by NetWidget
	void garbageCollection();
	QPoint getActive();
	int activeIsInput();
	int activeIsOutput();
	bool activeIsEnd();
	int countNodes();
	int lockNode(QPoint);
	bool lockLastNode();
	int lockSelectedNode();
	void updateNode(QPoint);	// update position of active node
	void releaseNode();

	// called by Selection
	int getMinSelX();
	int getMaxSelX();
	int getMinSelY();
	int getMaxSelY();
	void moveBy(const QPoint&);
	bool isSelected();

	// constants
	static const int NODE_NOLOCK = 0;
	static const int NODE_SINGLE = 1;
	static const int NODE_FRONT = 2;
	static const int NODE_MIDDLE = 3;

	static const int WFAIL = -1;

	// import/export mothods
	void setExportDelta(int dx, int dy);
	void setExportSelected(bool);
	QString operator>> (QString &);
	bool operator<< (const QString &);

protected:
	// erase whole wire (usually use erase)
	void eraseWire(QPainter *);
	// erase active node and end nodes
	void erase(QPainter *);
	// draw all nodes
	void draw(QPainter *);
	// draw grid into rect
	void restoreArry(QPainter *p, int, int, int, int);
	void drawSolderedPoint(QPainter *, QColor, QPoint);
	void drawInverterPoint(QPainter *, QColor, QPoint);

	// wire modifying methods
	KlogicList<QPoint> *contains(QPoint);
	int lockNode(KlogicList<QPoint> *);
	int lockNode(QPoint *);
	bool nodeIsReference(QPoint *);
	bool nodeIsForeign(QPoint *);
	int nodeIsInput(QPoint *);
	int nodeIsOutput(QPoint *);
	int isPart(QPoint *);

	// operations on active (locked) node
	int createNode(const QPoint&);	// set active node
	void cutNode();			// will not destroy it!
	void removeNode();		// destroys active node
	QPoint *getActiveNode();
	bool activeIsReference();	// node is only a copy
	bool activeIsForeign();		// also part of another wire
	int getActiveConnIndex();

	// called by XWire: connection methods
	int connectInput(int);
	void disconnectInput(int);
	int connectOutput(int);
	void disconnectOutput(int);
	int connectWire(int);
	void disconnectWire(int);
	QPoint *checkConnection(QPoint *);

	// wire-part selctions
	bool select(QRect, bool all = false);
	void unselect();

	// current color of the wire
	QColor m_oColor;

	// set wire size in pixels
	void setLineSize(int);

	// erase everything before refreshing the wire next time
	void enforceCompleteDrawing();
	bool completeDrawingIsEnforced();

	virtual bool inputIsInverted(int) { return false; }
	virtual bool outputIsInverted(int) { return false; }

private:
	// *****************************************************
	// import/export
	// *****************************************************
	int string2wire(const char *);	// hist import
	int exp_dx;
	int exp_dy;
	bool exp_selected;

	// *****************************************************
	// connection detection methods
	// *****************************************************
	int derive(QPoint, QPoint, QPoint);
	int inBetween(QPoint, QPoint, QPoint);
	QPoint * align(QPoint *);

	// *****************************************************
	// connection management
	// *****************************************************
	// node of wire, which is input-node (input of a device)
	QPoint *m_poInputNode[2];

	// node of wire, which is output-node (output of a device)
	QPoint *m_poOutputNode[2];

	// connection nodes (wire-wire-connections)
	QPoint *m_poWireNode[2];

	// connection nodes of other wires (references)
	KlogicList<QPoint> m_oForeignWireNodes;

	// *****************************************************
	// moving and drawing management
	// *****************************************************
	// last position of the locked node at time of its drawing
	QPoint o_node;
	QPoint o_start_node;
	QPoint o_end_node;
	// last wire-course (node inserted if not null)
	QPoint o_ins_node1, o_ins_node2;

	// actual locked node with predecessor and successor
	QPoint *from_node;
	QPoint *current_node;
	QPoint *to_node;

	bool bMustErase;

	// *****************************************************
	// the wire
	// *****************************************************
	// all nodes of this wire
	KlogicList<QPoint> wire;

	// all currenty selected nodes of this wire
	KlogicList<QPoint> selnode;

	int m_iSize;
	int m_iOldSize;
};

#endif
