/***************************************************************************
                          emoticontheme.h - holds a collection of emoticons
                             -------------------
    begin                : Tue April 10 2007
    copyright            : (C) 2007 by Valerio Pilo
    email                : amroth@coldshock.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef EMOTICONTHEME_H
#define EMOTICONTHEME_H

#include <qobject.h>
#include <qptrlist.h>
#include <qregexp.h>

#include "emoticon.h"



/**
 * @brief Data class for a group of emoticons
 *
 * This class represents an emoticon theme, which may be a default emoticon set, or
 * a custom emoticon set. It can load, update, and save an emoticon theme.
 * Themes are loaded from, and saved to, XML definition files - compatible with other
 * clients like Kopete, even non-MSN ones.
 *
 * @author Michael Curtis, Diederik van der Boor, Valerio Pilo
 * @ingroup Root
 */
class EmoticonTheme : public QObject, public QPtrList<Emoticon>
{
  Q_OBJECT

  public:    // Public methods
    // Constructor
                                  EmoticonTheme();
    // Copy constructor
                                  EmoticonTheme( const EmoticonTheme &other );
    // Destructor
                                 ~EmoticonTheme();

    // Create a new emoticon and add it to the theme
    void                          addEmoticon( QString pictureFile, QStringList shortcuts );
    // Check if the theme contains a certain emoticon
    bool                          contains( QString shortcut );
    // Return the picture file names of all emoticons, mapped by their first shortcut code
    const QMap<QString,QString>  &getFileNames() const;
    // Return the search pattern to find emoticons in an HTML text
    const QRegExp                &getHtmlPattern() const;
    // Return the HTML replacement codes for all emoticons
    const QMap<QString,QString>  &getHtmlReplacements( bool small = false ) const;
    // Return the search pattern to find emoticons in a text
    const QRegExp                &getPattern() const;
    // Return one replacement code for the given emoticon
    const QString                &getReplacement( const QString &code, bool small = false ) const;
    // Return the replacement codes for all emoticons
    const QMap<QString,QString>  &getReplacements( bool small = false ) const;
    // Return where the picture files for this theme are located
    const QString                &getThemePath();
    // Load a theme, by creating it anew or by refreshing the current one
    bool                          loadTheme( QString themeName, bool isCustomTheme );
    // Delete a custom emoticon from the theme
    bool                          removeEmoticon( QString shortcut );
    // Save the current theme to an XML theme definition file
    bool                          saveTheme();

  public:  // Public static methods
    // Return the full path of the first emoticon of the specified theme
    static QString                getThemeIcon( QString themeDir );

  private:     // Private methods
    // Create the theme from nothing
    bool                          createTheme( QString themeDir );
    // Rebuild the search&replace caches
    void                          updateCache();
    // Update a standard theme with names for the standard MSN emoticons
    void                          updateTitles();
    // Update the currently loaded theme with new images
    bool                          updateTheme( QString themeDir );

  private:    // Private properties
    // Are we loading a theme?
    bool                          loadingTheme_;
    // Is this a custom theme?
    bool                          isCustomTheme_;
    // The HTML replace codes for emoticons
    QMap<QString,QString>         largeHtmlReplacements_;
    // The replace codes for emoticons
    QMap<QString,QString>         largeReplacements_;
    // The search pattern for html emoticon codes
    QRegExp                       patternHtml_;
    // The search pattern for text emoticon codes
    QRegExp                       patternText_;
    // The HTML replace codes for emoticons
    QMap<QString,QString>         smallHtmlReplacements_;
    // The replace codes for emoticons
    QMap<QString,QString>         smallReplacements_;
    // The full path to the currently loaded theme
    QString                       themePath_;
    // A mapping from code to emoticon picture filename
    QMap<QString,QString>         themeFileNames_;

  signals:
    // Signal that we've been changed and the UI needs to be refreshed
    void                          updated();
};



#endif
