/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include "kis_iptc_io.h"

#include <kis_debug.h>

#include <exiv2/iptc.hpp>

#include "kis_exiv2.h"

#include <kis_meta_data_store.h>
#include <kis_meta_data_entry.h>
#include <kis_meta_data_value.h>
#include <kis_meta_data_schema.h>

struct IPTCToKMD {
    QString exivTag;
    QString namespaceUri;
    QString name;
};

static const IPTCToKMD mappings[] = {
    { "Iptc.Application2.City", KisMetaData::Schema::PhotoshopSchemaUri, "City" },
    { "Iptc.Application2.Copyright", KisMetaData::Schema::DublinCoreSchemaUri, "rights" },
    { "Iptc.Application2.CountryName", KisMetaData::Schema::PhotoshopSchemaUri, "Country" },
    { "Iptc.Application2.CountryCode", KisMetaData::Schema::PhotoshopSchemaUri, "" },
    { "Iptc.Application2.Byline", KisMetaData::Schema::DublinCoreSchemaUri, "Creator" },
    { "Iptc.Application2.BylineTitle", KisMetaData::Schema::PhotoshopSchemaUri, "AuthorsPosition" },
    { "Iptc.Application2.DateCreated", KisMetaData::Schema::PhotoshopSchemaUri, "DateCreated" },
    { "Iptc.Application2.Caption", KisMetaData::Schema::DublinCoreSchemaUri, "description" },
    { "Iptc.Application2.Writer", KisMetaData::Schema::PhotoshopSchemaUri, "CaptionWriter" },
    { "Iptc.Application2.Headline", KisMetaData::Schema::PhotoshopSchemaUri, "Headline" },
    { "Iptc.Application2.SpecialInstructions", KisMetaData::Schema::PhotoshopSchemaUri, "Instructions" },
    { "Iptc.Application2.ObjectAttribute", KisMetaData::Schema::IPTCSchemaUri, "IntellectualGenre" },
    { "Iptc.Application2.TransmissionReference", KisMetaData::Schema::PhotoshopSchemaUri, "JobID" },
    { "Iptc.Application2.Keywords", KisMetaData::Schema::DublinCoreSchemaUri, "subject" },
    { "Iptc.Application2.SubLocation", KisMetaData::Schema::IPTCSchemaUri, "Location" },
    { "Iptc.Application2.Credit", KisMetaData::Schema::PhotoshopSchemaUri, "Credit" },
    { "Iptc.Application2.ProvinceState", KisMetaData::Schema::PhotoshopSchemaUri, "State" },
    { "Iptc.Application2.Source", KisMetaData::Schema::PhotoshopSchemaUri, "Source" },
    { "Iptc.Application2.Subject", KisMetaData::Schema::IPTCSchemaUri, "SubjectCode" },
    { "Iptc.Application2.ObjectName", KisMetaData::Schema::DublinCoreSchemaUri, "Title" },
    { "Iptc.Application2.Urgency", KisMetaData::Schema::PhotoshopSchemaUri, "Urgency" },
    { "Iptc.Application2.Category", KisMetaData::Schema::PhotoshopSchemaUri, "Category" },
    { "Iptc.Application2.SuppCategory", KisMetaData::Schema::PhotoshopSchemaUri, "SupplementalCategory" },
    { "", "", "" } // indicates the end of the array
};

struct KisIptcIO::Private {
    QHash<QString, IPTCToKMD> iptcToKMD;
    QHash<QString, IPTCToKMD> kmdToIPTC;
};

// ---- Implementation of KisExifIO ----//
KisIptcIO::KisIptcIO() : d(new Private)
{
}

void KisIptcIO::initMappingsTable() const
{
    // For some reason, initializing the tables in the constructor makes the it crash
    if(d->iptcToKMD.size() == 0)
    {
        for(int i = 0; not mappings[i].exivTag.isEmpty(); i++)
        {
            d->iptcToKMD[mappings[i].exivTag] = mappings[i];
            d->kmdToIPTC[
                    KisMetaData::SchemaRegistry::instance()
                        ->schemaFromUri(mappings[i].namespaceUri)
                        ->generateQualifiedName( mappings[i].name) ] = mappings[i];
        }
    }
}

bool KisIptcIO::saveTo(KisMetaData::Store* store, QIODevice* ioDevice) const
{
    initMappingsTable();
    ioDevice->open(QIODevice::WriteOnly);
    Exiv2::IptcData iptcData;
    for(QHash<QString, KisMetaData::Entry>::const_iterator it = store->begin();
        it != store->end(); ++it )
    {
        const KisMetaData::Entry& entry = *it;
        if(d->kmdToIPTC.contains(entry.qualifiedName()))
        {
            QString iptcKeyStr = d->kmdToIPTC[ entry.qualifiedName() ].exivTag;
            Exiv2::IptcKey iptcKey(qPrintable(iptcKeyStr));
            iptcData.add(iptcKey, kmdValueToExivValue( entry.value(),
                         Exiv2::IptcDataSets::dataSetType( iptcKey.tag(), iptcKey.record()) ) );
        }
    }
    Exiv2::DataBuf rawData = iptcData.copy();
    ioDevice->write( (const char*) rawData.pData_, rawData.size_);
    ioDevice->close();
    return true;
}

bool KisIptcIO::canSaveAllEntries(KisMetaData::Store* store) const
{
    Q_UNUSED(store);
    return false;
}

bool KisIptcIO::loadFrom(KisMetaData::Store* store, QIODevice* ioDevice) const
{
    initMappingsTable();
    dbgFile <<"Loading IPTC Tags";
    ioDevice->open(QIODevice::ReadOnly);
    QByteArray arr = ioDevice->readAll();
    Exiv2::IptcData iptcData;
    iptcData.load((const Exiv2::byte*)arr.data(), arr.size());
    dbgFile <<"There are" << iptcData.count() <<" entries in the IPTC section";
    for(Exiv2::IptcMetadata::const_iterator it = iptcData.begin();
        it != iptcData.end(); ++it)
    {
        dbgFile <<"Reading info for key" << it->key().c_str();
        if(d->iptcToKMD.contains(it->key().c_str()))
        {
            const IPTCToKMD& iptcToKMd = d->iptcToKMD[it->key().c_str()];
            store->addEntry(KisMetaData::Entry(
                            KisMetaData::SchemaRegistry::instance()->schemaFromUri(iptcToKMd.namespaceUri),
                            iptcToKMd.name,
                            exivValueToKMDValue(it->getValue())));
        }
    }
    return false;
}
