/***************************************************************************
 *  libbab.cc -- BabyTrans ( Babylon Translator front-end for GTK )        *
 *                                                                         *
 *  Copyright (C) 1999-2002  Frederic Jolliton -- <babytrans@tuxee.net>    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef LIBBAB_H
#define LIBBAB_H

#include <fstream>
#include <list>
#include <string>

class Babylon
{
public:
	/*
	 * A item contain 3 fields:
	 *
	 * - word
	 *
	 *   The word translated.
	 *
	 * - attrib
	 *
	 *   A string that is the hexadecimal representation
	 *   of a bit field. Each bit indicate various attribute for the
	 *   word, such as verb, pronoum, and so on. This field is not
	 *   decoded. The application must process it.
	 *
	 * - definition (as one string, even if multiple definition are available.
	 *   They are concatened.)
	 *
	 */
	struct Item {
							Item() { }
		/* Copy constructor */
							Item( const Item& other ) {
			word = other.word ;
			attrib = other.attrib ;
			definition = other.definition ;
		}
		/* Assignation operator */
		Item&				operator=( const Item& other ) {
			word = other.word ;
			attrib = other.attrib ;
			definition = other.definition ;
			return *this ;
		}
		std::string			word ;
		std::string			attrib ;
		std::string			definition ;
	} ;

	/* Define ContainerType as list of item */
	typedef std::list< Item >
						ContainerType ;
private:
	/* System dependant definitions ! */
	typedef unsigned char   u8 ;
	typedef unsigned short  u16 ;
	typedef unsigned long   u32 ;

	/* Flag reflecting the state of the instance.
	 * 'true' is dictionnary are ready (correctly opened)
	 * 'false' otherwise.
	 */
	bool				myIsOk ;

	/* File containing the index into definitions file */
	std::ifstream		myFileIdx ;

	/* File containing all the word definition */
	std::ifstream		myFileDef ;
	
	/* Path to dictionnaries */
	std::string			myPath ;
	
	/* The current error message if any */
	std::string			myErrorMsg ;

	/* Convert value used by Babylon dictionnaries to character */
	char				babToChar( int c ) ;

	/* Private implementation */
	bool				translatePriv( const std::string& word , ContainerType& ) ;

	/* Read (and decode !) a word of 'length' character */
	std::string			readWord( unsigned int length ) ;

	/* Read a byte (assuming 8 bit bytes) from file 'is' */
	u8					readU8( std::istream& is ) {
		return (u8)is.get() ;
	}

	/* Read 2 bytes from file 'is' */
	u16					readU16( std::istream& is ) {
		return readU8( is ) + (u16)( readU8( is ) ) * 256 ;
	}

	/* Read 4 bytes from file 'is' */
	u32					readU32( std::istream& is ) {
		return readU16( is ) + (u32)( readU16( is ) ) * 256 * 256 ;
	}

	/* Read a definition and return it as a Item */
	Item				makeDefinition( size_t ) ;

	/* Set current error message */
	void				setError( const std::string& err ) {
		myErrorMsg = err ;
	}
public:
						~Babylon() ;

	/* Set current path to search dictionnaries */
	void				setPath( const std::string& path ) {
		myPath = path ;
	}

	/* Open dictionnary ('filename' is the definition file filename) */
	bool				open( const std::string& filename ) ;

	/* Close the current opened dictionnary */
	void				close() ;

	/* Search the translation of a word and fill a provided list reference */
	bool				translate( const std::string& word , ContainerType& ) ;

	/* Return the last error that occured */
	std::string			getError() { return myErrorMsg ; }
} ;

#endif /* LIBBAB_H */
