// =============================================================================
//
//      --- kvi_dcc_chat_thread.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDccChatThread"

#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include "kvi_application.h"
#include "kvi_dcc_chat.h"
#include "kvi_dcc_chat_thread.h"
#include "kvi_dcc_event.h"
#include "kvi_error.h"
#include "kvi_irc_socket.h"
#include "kvi_locale.h"
#include "kvi_mutex.h"
#include "kvi_netutils.h"

// Time to sleep if no data has to be read or sent
// In DCC Chat it can be a bit longer...
#define KVI_DCC_IDLE_TIME_USECS 400

////////////////////////////////////////////////////////////////////////////////

KviDccChatThread::KviDccChatThread(KviDccChatData *data)
	: KviDccThread()
{
	// Data is not owned by the thread itself
	m_pData = data;
}

KviDccChatThread::~KviDccChatThread()
{
	// Nothing here
}

/**
 * Main DCC Chat loop
 */
void KviDccChatThread::eventLoop()
{
	// It is terminated when abort() is called
	// or if the parent object terminates this thread
	for( ; !m_bStopRequested; ) {
		checkForIncomingData();
		checkForOutgoingData();
	}
}

/**
 * Check if there is data available on m_socket,
 * read it, split into messages and post data events
 * to the parent DCC window class.
 */
void KviDccChatThread::checkForIncomingData()
{
	if( !selectForRead(m_socket) ) {
		usleep(KVI_DCC_IDLE_TIME_USECS);
		return;
	}

	// Read data (1024 + 1 byte for \0 terminator)
	char buffer[1025];
	int readLength = read(m_socket, buffer, 1024);

	if( m_bStopRequested ) return;

	if( readLength <= 0 ) {
		// Oops?
		if( readLength == 0 ) {
			abort(_i18n_("Remote end has closed the connection"), m_pData->parent);
			return;
		} else { // Error?
			if( (errno == EINTR) || (errno == EAGAIN) )
				return;
			// Yes... error :(
			int iError;
			switch( errno ) {
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error; pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// Post the error event to the parent and exit
			m_pData->tmpBuffer.sprintf(_i18n_("READ ERROR: %s"), kvi_getErrorString(iError));
			abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
			return;
		}
	}

	// Successfully read data.
	*(buffer + readLength) = '\0';           // Null-terminate the buffer
	m_pData->szPartialBuffer.append(buffer); // And merge with the latest unterminated data

	// Now split the data into messages
	// A message MUST be terminated by a LF.
	int idx = m_pData->szPartialBuffer.findFirstIdx('\n');
	while( idx != -1 ) {
		// Strip CR after the message... if any
		KviStr tmp = m_pData->szPartialBuffer.left(idx);
		if( idx > 0 ) {
			if( m_pData->szPartialBuffer.at(idx - 1) == '\r' ) {
				tmp.cutRight(1);
			}
		}
		// And now post the data event to the parent...
		KviDccEvent *ev = new KviDccEvent(KVI_DCC_EVENT_DATA, tmp.ptr());
		KviApplication::postEvent(m_pData->parent, ev);

		// Remove the stuff that has been processed...
		m_pData->szPartialBuffer.cutLeft(idx + 1);
		// And look for the next message
		idx = m_pData->szPartialBuffer.findFirstIdx('\n');
	}

	if( m_pData->szPartialBuffer.len() > 512 ) {
		// Oops... no LF in the first 512 bytes?
		// This must be a looooong message...
		// Hehehe... if I were a hacker, I could use it as
		// a system attack once a DCC chat has been estabilished.
		// Just write a script that sends an infinite sequence of senseless
		// data without LF's inside...
		// Sooner or later an unhandled ENOMEM will jump out :)
		// Anyway, KVIrc is immune now...
		KviDccEvent *ev = new KviDccEvent(
			KVI_DCC_EVENT_MSG, _i18n_("Unterminated message longer than 512 bytes: auto-splitting")
		);
		KviApplication::postEvent(m_pData->parent, ev);

		ev = new KviDccEvent(KVI_DCC_EVENT_DATA, m_pData->szPartialBuffer.ptr());
		KviApplication::postEvent(m_pData->parent, ev);
		m_pData->szPartialBuffer = "";
	}
}

void KviDccChatThread::checkForOutgoingData()
{
	// Need to lock a mutex while managing the outQueue.
	// No cancel available while mutex is locked, so
	// there should be no need for another cleanup routine.

	// Usually there is only one message per 1-2 seconds (unless
	// the user pastes something in the input line), so this stuff
	// can be done only once per loop...

	KviStr *str = 0;
	m_pData->outMutex->lock();
	str = m_pData->outQueue->head();
	m_pData->outMutex->unlock();
	if( !str ) {
		usleep(KVI_DCC_IDLE_TIME_USECS); // Take a break
		return;                          // No data to be sent
	}

	int wrLen = write(m_socket, str->ptr(), str->len());

	// Managing str is now safe... the pointer and the object are never changed
	// by other threads...
	if( wrLen == str->len() ) {
		// All data sent...
		// Send a CR/LF too
		write(m_socket, "\r\n", 2);
		// Now relock the mutex and remove the string from the queue
		m_pData->outMutex->lock();
		m_pData->outQueue->remove();
		m_pData->outMutex->unlock();
	} else {
		if( wrLen >= 0 )
			str->cutLeft(wrLen);
		else {
			// Oops... error while writing?
			if( (errno == EINTR) || (errno == EAGAIN) )
				return;
			// Yes... error :(
			int iError;
			switch( errno ) {
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				case EFAULT:       iError = KVI_ERROR_OutOfAddressSpace;     break;
				case EBADF:        iError = KVI_ERROR_BadFileDescriptor;     break;
				// Unhandled error; pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// Post the error event to the parent and exit
			m_pData->tmpBuffer.sprintf(_i18n_("WRITE ERROR: %s"), kvi_getErrorString(iError));
			abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
			return;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////

KviDccChatAcceptThread::KviDccChatAcceptThread(KviDccChatData *data)
	: KviDccChatThread(data)
{
	// Nothing here
}

KviDccChatAcceptThread::~KviDccChatAcceptThread()
{
	// Nothing here
}

/**
 * Accept a DCC CHAT REQUEST from another user on IRC.
 * A user sent us a CTCP DCC with a port and an address.
 * Now we try to contact that host and initiate a DCC Chat.
 */
void KviDccChatAcceptThread::run()
{
	// Prepare the target data
	struct in_addr     inAddress;
	struct sockaddr_in hostSockAddr;

	inAddress.s_addr        = m_pData->uAddress;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(m_pData->uPort);
	hostSockAddr.sin_addr   = inAddress;

	// Let's go...
	m_socket = socket(PF_INET, SOCK_STREAM, 0);

	if( m_socket < 0 ) {
		abort(_i18n_("Unable to create a stream socket. DCC chat failed"), m_pData->parent);
		return;
	}

	if( fcntl(m_socket, F_SETFL, O_NONBLOCK) < 0 ) {
		abort(_i18n_("Unable to create a non-blocking stream socket. DCC chat failed"), m_pData->parent);
		return;
	}

	if( m_bStopRequested ) return;
	// Ok... now try to connect
	int iError = connect((struct sockaddr *) (&hostSockAddr), sizeof(hostSockAddr));

	if( iError != KVI_ERROR_Success ) {
		m_pData->tmpBuffer.sprintf(_i18n_("CONNECT ERROR: %s"), kvi_getErrorString(iError));
		abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
		return;
	}

	// Now wait for connection...
	KviStr tmp(KviStr::Format, _i18n_("Connecting to %s on port %u"), m_pData->szAddress.ptr(), m_pData->uPort);
	KviDccEvent *msg = new KviDccEvent(KVI_DCC_EVENT_MSG, tmp.ptr());
	KviApplication::postEvent(m_pData->parent, msg);

	iError = waitForOutgoingConnection();
	if( iError != KVI_ERROR_Success ) {
		m_pData->tmpBuffer.sprintf(_i18n_("CONNECT ERROR: %s"), kvi_getErrorString(iError));
		abort(m_pData->tmpBuffer.ptr(), m_pData->parent);
		return;
	}

	msg = new KviDccEvent(KVI_DCC_EVENT_MSG, _i18n_("Connection established"));
	KviApplication::postEvent(m_pData->parent, msg);

	m_pData->tmpBuffer.sprintf(
		"%s %s %s %s", m_pData->nick.ptr(), m_pData->username.ptr(), m_pData->host.ptr(), m_pData->szAddress.ptr()
	);
	msg = new KviDccEvent(KVI_DCC_EVENT_CONNECTIONESTABLISHED, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, msg);

	eventLoop();
}

////////////////////////////////////////////////////////////////////////////////

KviDccChatRequestThread::KviDccChatRequestThread(KviDccChatData *data)
	: KviDccChatThread(data)
{
	// Nothing here
}

KviDccChatRequestThread::~KviDccChatRequestThread()
{
	// Nothing here
}

/**
 * Request a DCC from an IRC user
 */
void KviDccChatRequestThread::run()
{
	int newsock = -1;
	KviDccEvent *ev = 0;

	struct sockaddr_in sockAddress;
	sockAddress.sin_family      = AF_INET;
	sockAddress.sin_port        = htons(m_pData->uPortToListenOn);
	sockAddress.sin_addr.s_addr = INADDR_ANY;

	// Let's go...
	m_socket = socket(PF_INET, SOCK_STREAM, 0);

	if( m_socket < 0 ) {
		abort(_i18n_("Unable to create a listening socket. DCC chat failed"), m_pData->parent);
		return;
	}

	if( m_bStopRequested ) return;

	if( (bind(m_socket, (struct sockaddr *) &sockAddress, sizeof(sockAddress)) < 0) || (listen(m_socket, 100) < 0) ) {
		abort(_i18n_("Unable to set up a listening socket. DCC chat failed"), m_pData->parent);
		return;
	}

	socklen_t iSize = sizeof(sockAddress);
	getsockname(m_socket, (struct sockaddr*) &sockAddress, &iSize);
	m_pData->uPort = ntohs(sockAddress.sin_port);

	m_pData->tmpBuffer.sprintf(_i18n_("Listening on port %u"), m_pData->uPort);
	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->tmpBuffer.sprintf(
		"PRIVMSG %s :%cDCC CHAT chat %u %u%c", m_pData->nick.ptr(), 0x01, ntohl(m_pData->uAddress), m_pData->uPort, 0x01
	);
	ev = new KviDccEvent(KVI_DCC_EVENT_LISTENING, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	newsock = waitForIncomingConnection(&(m_pData->uAddress), &(m_pData->uPort), &(m_pData->szAddress));

	close(m_socket);    // Close the old socket...
	m_socket = newsock; // And set the new one

	m_pData->tmpBuffer.sprintf(_i18n_("Connected to %s on port %u"), m_pData->szAddress.ptr(), m_pData->uPort);
	ev = new KviDccEvent(KVI_DCC_EVENT_MSG, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	m_pData->tmpBuffer.sprintf(
		"%s %s %s %s", m_pData->nick.ptr(), m_pData->username.ptr(), m_pData->host.ptr(), m_pData->szAddress.ptr()
	);
	ev = new KviDccEvent(KVI_DCC_EVENT_CONNECTIONESTABLISHED, m_pData->tmpBuffer.ptr());
	KviApplication::postEvent(m_pData->parent, ev);

	eventLoop();
}
