// =============================================================================
//
//      --- kvi_editor_syntaxhighlighter.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qtextedit.h>

#include "kvi_config.h"
#include "kvi_editor_syntaxhighlighter.h"
#include "kvi_options.h"

#define SYNTAX_STATE_BEGIN_IN_COMMENT 1
#define SYNTAX_STATE_END_IN_COMMENT   2
#define SYNTAX_STATE_BEGIN_IN_TAG     4
#define SYNTAX_STATE_END_IN_TAG       8

KviEditorSyntaxHighlighter::KviEditorSyntaxHighlighter(QTextEdit *textedit)
	: QSyntaxHighlighter(textedit)
{
	m_pColors = new KviEditorColors();

	m_pColors->font = KviConfig::getFixedFont();
	m_pColors->font.setPointSize(g_pOptions->m_fntView.pointSize());
	m_pColors->htmlFont = m_pColors->font;
	m_pColors->cppFont  = m_pColors->htmlFont;

	m_pColors->background          = QColor(  0,   0,   0);
	m_pColors->cursor              = QColor(255,   0,   0);
	m_pColors->extBackground       = QColor( 40,  40,  40);
	m_pColors->normalText          = QColor( 50, 255,   0);

	m_pColors->cppBackground       = QColor(  0,   0,   0);
	m_pColors->cppBrace            = QColor(255,   0,   0);
	m_pColors->cppChar             = QColor(100, 140, 250);
	m_pColors->cppCursor           = QColor(255,   0,   0);
	m_pColors->cppEscape           = QColor( 50, 130, 240);
	m_pColors->cppExtBackground    = QColor( 40,  40,  40);
	m_pColors->cppGType            = QColor(190, 190, 190);
	m_pColors->cppGdkCall          = QColor(150, 120, 180);
	m_pColors->cppGdkMacro         = QColor(230, 170, 160);
	m_pColors->cppGdkStruct        = QColor(255, 205,  90);
	m_pColors->cppGlobalVariable   = QColor(230, 200, 110);
	m_pColors->cppGtkCall          = QColor(150, 150, 180);
	m_pColors->cppGtkMacro         = QColor(220, 170, 180);
	m_pColors->cppGtkStruct        = QColor(255, 255,  50);
	m_pColors->cppInclude          = QColor(200, 200, 200);
	m_pColors->cppKClass           = QColor(255, 255,   0);
	m_pColors->cppKeyword          = QColor(130, 130, 130);
	m_pColors->cppLineComment      = QColor( 40, 150,   0);
	m_pColors->cppMemberVariable   = QColor(190, 170,  80);
	m_pColors->cppMultilineComment = QColor( 20, 180,   0);
	m_pColors->cppNormalText       = QColor( 80, 255,   0);
	m_pColors->cppNumber           = QColor(190, 200, 255);
	m_pColors->cppOperator         = QColor(150, 150,  40);
	m_pColors->cppParenthesis      = QColor(170, 130,  30);
	m_pColors->cppPreprocessor     = QColor(255, 255, 255);
	m_pColors->cppPunctuation      = QColor(180, 180,  50);
	m_pColors->cppQClass           = QColor(255, 255,  50);
	m_pColors->cppQSignals         = QColor(255, 150,   0);
	m_pColors->cppSpecial          = QColor(150, 150, 150);
	m_pColors->cppString           = QColor( 80, 170, 250);
	m_pColors->cppSystemIdentifier = QColor(255,   0, 255);
	m_pColors->cppType             = QColor(160, 160, 160);
	m_pColors->cppXStuff           = QColor(255, 255,  90);

	m_pColors->htmlBackground      = QColor(  0,   0,   0);
	m_pColors->htmlComment         = QColor( 35, 180,   0);
	m_pColors->htmlCursor          = QColor(255,   0,   0);
	m_pColors->htmlExtBackground   = QColor( 40,  40,  40);
	m_pColors->htmlNormalText      = QColor( 20, 255,  20);
	m_pColors->htmlString          = QColor( 40, 180, 255);
	m_pColors->htmlTag             = QColor(180, 100,  30);
	m_pColors->htmlTagInternal     = QColor(180, 150,  20);

	m_mode = Normal;
	initializeCurrentMode();
}

KviEditorSyntaxHighlighter::~KviEditorSyntaxHighlighter()
{
	delete m_pColors;
	m_pColors = 0;
}

void KviEditorSyntaxHighlighter::switchMode()
{
	switch( m_mode ) {
		case Normal: setMode(CPP);    break;
		case CPP:    setMode(HTML);   break;
		case HTML: // Fall-through
		default:
			setMode(Normal);
			break;
	}
}

void KviEditorSyntaxHighlighter::setMode(ColorMode mode)
{
	if( m_mode == mode ) return;

	m_mode = mode;
	initializeCurrentMode();
	rehighlight();
}

void KviEditorSyntaxHighlighter::initializeCurrentMode()
{
	QTextEdit *w = textEdit();
	QPalette pal = w->palette();
	QFontMetrics *fm;
	switch( m_mode ) {
		case CPP:
			pal.setColor(QColorGroup::Base, m_pColors->cppBackground);
			pal.setColor(QColorGroup::Text, m_pColors->cursor);
			fm = new QFontMetrics(m_pColors->cppFont);
			break;
		case HTML:
			pal.setColor(QColorGroup::Base, m_pColors->htmlBackground);
			pal.setColor(QColorGroup::Text, m_pColors->cursor);
			fm = new QFontMetrics(m_pColors->htmlFont);
			break;
		case Normal: // Fall-through
		default:
			pal.setColor(QColorGroup::Base, m_pColors->background);
			pal.setColor(QColorGroup::Text, m_pColors->cursor);
			fm = new QFontMetrics(m_pColors->font);
			break;
	}
	w->setTabStopWidth(fm->width(' ') * KVI_TAB_WIDTH);
	w->setPalette(pal);
	delete fm;
}

KviEditorSyntaxHighlighter::ColorMode KviEditorSyntaxHighlighter::mode()
{
	return m_mode;
}

int KviEditorSyntaxHighlighter::highlightParagraph(const QString &text, int state)
{
	switch( m_mode ) {
		case CPP:
			return highlightParagraphCPP(text, state);
			break;
		case HTML:
			return highlightParagraphHTML(text, state);
			break;
		case Normal:
		default:
			return highlightParagraphNormal(text, state);
			break;
	}
}

int KviEditorSyntaxHighlighter::highlightParagraphNormal(const QString &text, int state)
{
	setFormat(0, text.length(), m_pColors->font, m_pColors->normalText);
	return state;
}

int KviEditorSyntaxHighlighter::highlightParagraphCPP(const QString &text, int state)
{
	int current  = 0;
	int begin    = 0;
	QColor *pClr = 0;

	bool bInMultilineComment = state & SYNTAX_STATE_BEGIN_IN_COMMENT;
	bool bInLineComment      = false;
	bool bInString           = false;
	bool bInChar             = false;
	bool bPreprocessor       = false;
	bool bGotInclude         = false;

	while( text[current] ) {
		switch( text[current] ) {
			case '\t':
				current++;
				break;
			case ' ':
				for( ; text[current] == ' '; current++ );
				break;
			default: {
				begin = current;
				if( bInLineComment ) {
					// We are in a line comment. Up to the end of the line!
					while( (text[current]) && (text[current] != ' ') && (text[current] != '\t') )
						current++;
					pClr = &(m_pColors->cppLineComment);
				} else if( bInMultilineComment ) {
					// Multiline comment
					while( (text[current]) && bInMultilineComment && (text[current] != ' ') && (text[current] != '\t') ) {
						if( text[current] == '*' ) {
							current++;
							if( text[current] == '/' ) {
								// End of comment
								current++;
								bInMultilineComment = false;
							}
						} else current++;
					}
					pClr = &(m_pColors->cppMultilineComment);
				} else if( bInString ) {
					// Inside a string
					if( text[current] == '\\' ) {
						// Escape sequence inside a string
						current++;
						if( (text[current]) && (text[current] != ' ') && (text[current] != '\t') ) {
							current++;
							pClr = &(m_pColors->cppEscape);
						} else pClr = &(m_pColors->cppString);
					} else {
						while( text[current] && bInString &&
						      (text[current] != ' ') && (text[current] != '\t') && (text[current] != '\\')
						) {
							if( text[current] == '"' ) {
								current++;
								bInString = false;
							} else current++;
						}
						pClr = &(m_pColors->cppString);
					}
				} else if( bInChar ) {
					// Inside a char constant
					if( text[current] == '\\' ) {
						// Escape sequence inside a string
						current++;
						if( (text[current]) && (text[current] != ' ') && (text[current] != '\t') ) {
							current++;
							pClr = &(m_pColors->cppEscape);
						} else pClr = &(m_pColors->cppChar);
					} else {
						while( (text[current]) && bInChar &&
						       (text[current] != ' ') && (text[current] != '\t') && (text[current] != '\\')
						) {
							if( text[current] == '\'' ) {
								current++;
								bInChar = false;
							} else current++;
						}
						pClr = &(m_pColors->cppChar);
					}
				} else if( bGotInclude ) {
					bGotInclude = false;
					while( (text[current]) && (text[current] != ' ') && (text[current] != '\t') )
						current++;
					pClr = &(m_pColors->cppInclude);
				} else if(
					((text[current] >= 'a') && (text[current] <= 'z')) ||
					((text[current] >= 'A') && (text[current] <= 'Z')) ||
					(text[current] == '_')
				) {
					// A text token
					current++;
					while(
						((text[current] >= 'a') && (text[current] <= 'z')) ||
						((text[current] >= 'A') && (text[current] <= 'Z')) ||
						((text[current] >= '0') && (text[current] <= '9')) ||
						(text[current] == '_')
					) {
						current++;
					}
					if( bPreprocessor ) {
						pClr = &(m_pColors->cppPreprocessor);
						bPreprocessor = false;
						if( text.mid(begin, current - begin) == "include" )
							bGotInclude = true;
					} else pClr = cppModeGetTokenColor(text.mid(begin, current - begin));
				} else if(
					(text[current] == '(') || (text[current] == ')') ||
					(text[current] == '[') || (text[current] == ']')
				) {
					// A parenthesis
					current++;
					while(
						(text[current] == '(') || (text[current] == ')') ||
						(text[current] == '[') || (text[current] == ']')
					) {
						current++;
					}
					pClr = &(m_pColors->cppParenthesis);
				} else if( (text[current] == ':') ) {
					current++;
					if( text[current] == ':' )
						current++;
					pClr = &(m_pColors->cppPunctuation);
				} else if( (text[current] == ';') || (text[current] == ',') || (text[current] == '.') ) {
					current++;
					pClr = &(m_pColors->cppPunctuation);
				} else if( (text[current] >= '0') && (text[current] <= '9') ) {
					// A parenthesis
					current++;
					while(
						((text[current] >= '0') && (text[current] <= '9')) ||
						((text[current] >= 'a') && (text[current] <= 'z')) ||
						((text[current] >= 'A') && (text[current] <= 'Z'))
					) {
						current++;
					}
					pClr = &(m_pColors->cppNumber);
				} else if( text[current] == '#' ) {
					// A preprocessor code
					current++;
					bPreprocessor = true;
					pClr = &(m_pColors->cppPreprocessor);
				} else if(
					(text[current] == '+') || (text[current] == '-') ||
					(text[current] == '*') || (text[current] == '?') ||
					(text[current] == '!') || (text[current] == '<') ||
					(text[current] == '>') || (text[current] == '^') ||
					(text[current] == '%') || (text[current] == '&') ||
					(text[current] == '|') || (text[current] == '=') ||
					(text[current] == '~') || (text[current] == '\\')
				) {
					// A parenthesis
					current++;
					while(
						(text[current] == '+') || (text[current] == '-') ||
						(text[current] == '*') || (text[current] == '?') ||
						(text[current] == '!') || (text[current] == '<') ||
						(text[current] == '>') || (text[current] == '^') ||
						(text[current] == '%') || (text[current] == '&') ||
						(text[current] == '|') || (text[current] == '=') ||
						(text[current] == '~') || (text[current] == '\\')
					) {
						current++;
					}
					pClr = &(m_pColors->cppOperator);
				} else if( (text[current] == '{') || (text[current] == '}') ) {
					// Brace
					current++;
					pClr = &(m_pColors->cppBrace);
				} else if( (text[current] == '"') ) {
					// String
					current++;
					bInString = true;
					pClr = &(m_pColors->cppString);
				} else if( (text[current] == '\'') ) {
					// String
					current++;
					bInChar = true;
					pClr = &(m_pColors->cppChar);
				} else if( (text[current] == '/') ) {
					// Comment begin?
					current++;
					if( text[current] == '/' ) {
						bInLineComment = true;
						current++;
						pClr = &(m_pColors->cppLineComment);
					} else if( text[current] == '*' ) {
						bInMultilineComment = true;
						current++;
						pClr = &(m_pColors->cppMultilineComment);
					} else pClr = &(m_pColors->cppOperator);
				} else {
					while(
						(text[current])        && (text[current] != ' ') && (text[current] != '\t') &&
						(text[current] != '{') && (text[current] != '}') && (text[current] != '/') &&
						(text[current] != '(') && (text[current] != ')') && (text[current] != '[') &&
						(text[current] != ']') && (text[current] != '"') && (text[current] != '\'') &&
						(text[current] != '+') && (text[current] != '-') && (text[current] != '*') &&
						(text[current] != '!') && (text[current] != '<') && (text[current] != '.') &&
						(text[current] != '>') && (text[current] != '^') && (text[current] != '%') &&
						(text[current] != '&') && (text[current] != '|') && (text[current] != '=') &&
						(text[current] != ',') && (text[current] != ';') && (text[current] != ':') &&
						(text[current] != '~') && (text[current] != '?') && (text[current] != '#') &&
						(text[current] != '\\')
					) {
						current++;
					}
					pClr = &(m_pColors->cppNormalText);
				}
				setFormat(begin, current - begin, m_pColors->cppFont, *pClr);
				break;
			}
		}
	}
	state = 0;
	if( bInMultilineComment )
		state = SYNTAX_STATE_BEGIN_IN_COMMENT;
	return state;
}

int KviEditorSyntaxHighlighter::highlightParagraphHTML(const QString &text, int state)
{
	int current  = 0;
	int begin    = 0;
	QColor *pClr = 0;

	bool bInComment = state & SYNTAX_STATE_BEGIN_IN_COMMENT;
	bool bInTag     = state & SYNTAX_STATE_BEGIN_IN_TAG;
	bool bInString  = false;

	while( text[current] ) {
		switch( text[current] ) {
			case '\t':
				// A tab character
				current++;
				break;
			case ' ':
				// A set of spaces
				for( ; text[current] == ' '; current++ );
				break;
			default: {
				begin = current;
				if( bInComment ) {
					if( text[current] == '-' ) {
						current++;
						if( text[current] == '-' ) {
							current++;
							if( text[current] == '>' ) {
								current++;
								bInComment = false;
								bInTag     = false;
								bInString  = false;
							} else pClr = &(m_pColors->htmlComment);
						} else pClr = &(m_pColors->htmlComment);
					} else {
						while( text[current] && (text[current] != '-') && (text[current] != ' ') && (text[current] != '\t') )
							current++;
						pClr = &(m_pColors->htmlComment);
					}
				} else if( bInTag ) {
					if( text[current] == '"' ) {
						current++;
						bInString = !bInString;
						pClr = &(m_pColors->htmlString);
					} else if( text[current] == '>' ) {
						current++;
						bInString = false;
						bInTag    = false;
						pClr = &(m_pColors->htmlTag);
					} else {
						while( text[current] && (text[current] != '>') && (text[current] != '"') )
							current++;
						pClr = bInString ? &(m_pColors->htmlString) : &(m_pColors->htmlTagInternal);
					}
				} else {
					// Normal text
					if( text[current] == '<' ) {
						current++;
						bInString = false;
						bInTag    = true;
						if( text[current] == '!' ) {
							current++;
							if( text[current] == '-' ) {
								current++;
								if( text[current] == '-' ) {
									current++;
									bInString  = false;
									bInTag     = false;
									bInComment = true;
									pClr = &(m_pColors->htmlComment);
								} else pClr = &(m_pColors->htmlTag);
							} else pClr = &(m_pColors->htmlTag);
						} else pClr = &(m_pColors->htmlTag);
					} else {
						while( text[current] && (text[current] != '<') )
							current++;
						pClr = &(m_pColors->htmlNormalText);
					}
				}
				setFormat(begin, current - begin, m_pColors->htmlFont, *pClr);
				break;
			}
		}
	}
	state = 0;
	if( bInComment )
		state = state | SYNTAX_STATE_BEGIN_IN_COMMENT;
	if( bInTag )
		state = state | SYNTAX_STATE_BEGIN_IN_TAG;
	return state;
}

QColor *KviEditorSyntaxHighlighter::cppModeGetTokenColor(const QString &token)
{
	if( (token.length() > 2) && token[token.length() - 1] == 't' ) {
		if( (token[0] != '_') && (token[token.length() - 2] == '_') )
			return &(m_pColors->cppType);
	}

	switch( token[0] ) {
		case '_':
			return &(m_pColors->cppSystemIdentifier);
			break;
		case 'a':
			if( token == "asm"  ) return &(m_pColors->cppKeyword);
			if( token == "auto" ) return &(m_pColors->cppKeyword);
			break;
		case 'b':
			if( token == "bool"  ) return &(m_pColors->cppType);
			if( token == "break" ) return &(m_pColors->cppKeyword);
			break;
		case 'c':
			if( token == "case"       ) return &(m_pColors->cppKeyword);
			if( token == "catch"      ) return &(m_pColors->cppKeyword);
			if( token == "char"       ) return &(m_pColors->cppType);
			if( token == "class"      ) return &(m_pColors->cppKeyword);
			if( token == "const"      ) return &(m_pColors->cppKeyword);
			if( token == "const_cast" ) return &(m_pColors->cppKeyword);
			if( token == "continue"   ) return &(m_pColors->cppKeyword);
			break;
		case 'd':
			if( token == "daemon"       ) return &(m_pColors->cppSpecial);
			if( token == "dec"          ) return &(m_pColors->cppKeyword);
			if( token == "default"      ) return &(m_pColors->cppKeyword);
			if( token == "defined"      ) return &(m_pColors->cppPreprocessor);
			if( token == "delete"       ) return &(m_pColors->cppKeyword);
			if( token == "dlclose"      ) return &(m_pColors->cppSpecial);
			if( token == "dlerror"      ) return &(m_pColors->cppSpecial);
			if( token == "dlopen"       ) return &(m_pColors->cppSpecial);
			if( token == "dlsym"        ) return &(m_pColors->cppSpecial);
			if( token == "do"           ) return &(m_pColors->cppKeyword);
			if( token == "domainname"   ) return &(m_pColors->cppSpecial);
			if( token == "double"       ) return &(m_pColors->cppType);
			if( token == "dup"          ) return &(m_pColors->cppSpecial);
			if( token == "dup2"         ) return &(m_pColors->cppSpecial);
			if( token == "dynamic_cast" ) return &(m_pColors->cppKeyword);
			break;
		case 'e':
			if( token == "else"    ) return &(m_pColors->cppKeyword);
			if( token == "emit"    ) return &(m_pColors->cppQSignals);
			if( token == "encrypt" ) return &(m_pColors->cppSpecial);
			if( token == "enum"    ) return &(m_pColors->cppKeyword);
			if( token == "execl"   ) return &(m_pColors->cppSpecial);
			if( token == "execle"  ) return &(m_pColors->cppSpecial);
			if( token == "execlp"  ) return &(m_pColors->cppSpecial);
			if( token == "execv"   ) return &(m_pColors->cppSpecial);
			if( token == "execve"  ) return &(m_pColors->cppSpecial);
			if( token == "execvp"  ) return &(m_pColors->cppSpecial);
			if( token == "extern"  ) return &(m_pColors->cppKeyword);
			break;
		case 'f':
			if( token == "false"   ) return &(m_pColors->cppKeyword);
			if( token == "fclose"  ) return &(m_pColors->cppSpecial);
			if( token == "fcntl"   ) return &(m_pColors->cppSpecial);
			if( token == "fdopen"  ) return &(m_pColors->cppSpecial);
			if( token == "feof"    ) return &(m_pColors->cppSpecial);
			if( token == "ferror"  ) return &(m_pColors->cppSpecial);
			if( token == "fexecve" ) return &(m_pColors->cppSpecial);
			if( token == "fflush"  ) return &(m_pColors->cppSpecial);
			if( token == "fgetc"   ) return &(m_pColors->cppSpecial);
			if( token == "fgetc"   ) return &(m_pColors->cppSpecial);
			if( token == "fgets"   ) return &(m_pColors->cppSpecial);
			if( token == "fileno"  ) return &(m_pColors->cppSpecial);
			if( token == "fixed"   ) return &(m_pColors->cppKeyword);
			if( token == "float"   ) return &(m_pColors->cppType);
			if( token == "fopen"   ) return &(m_pColors->cppSpecial);
			if( token == "for"     ) return &(m_pColors->cppKeyword);
			if( token == "fork"    ) return &(m_pColors->cppSpecial);
			if( token == "fprintf" ) return &(m_pColors->cppSpecial);
			if( token == "fputc"   ) return &(m_pColors->cppSpecial);
			if( token == "fread"   ) return &(m_pColors->cppSpecial);
			if( token == "free"    ) return &(m_pColors->cppSpecial);
			if( token == "freopen" ) return &(m_pColors->cppSpecial);
			if( token == "friend"  ) return &(m_pColors->cppKeyword);
			if( token == "fscanf"  ) return &(m_pColors->cppSpecial);
			if( token == "fseek"   ) return &(m_pColors->cppSpecial);
			if( token == "ftime"   ) return &(m_pColors->cppSpecial);
			if( token == "fwrite"  ) return &(m_pColors->cppSpecial);
			break;
		case 'g':
			if( token.length() <= 1 ) break;

			if( token[1] == '_'          ) return &(m_pColors->cppGlobalVariable);
			if( token.startsWith("gtk_") ) return &(m_pColors->cppGtkCall);
			if( token.startsWith("gdk_") ) return &(m_pColors->cppGdkCall);
			if( token == "gboolean"      ) return &(m_pColors->cppGType);
			if( token == "gchar"         ) return &(m_pColors->cppGType);
			if( token == "gconstpointer" ) return &(m_pColors->cppGType);
			if( token == "gdouble"       ) return &(m_pColors->cppGType);
			if( token == "gfloat"        ) return &(m_pColors->cppGType);
			if( token == "gint"          ) return &(m_pColors->cppGType);
			if( token == "gint16"        ) return &(m_pColors->cppGType);
			if( token == "gint32"        ) return &(m_pColors->cppGType);
			if( token == "gint64"        ) return &(m_pColors->cppGType);
			if( token == "gint8"         ) return &(m_pColors->cppGType);
			if( token == "gldouble"      ) return &(m_pColors->cppGType);
			if( token == "glong"         ) return &(m_pColors->cppGType);
			if( token == "goto"          ) return &(m_pColors->cppKeyword);
			if( token == "gpointer"      ) return &(m_pColors->cppGType);
			if( token == "gshort"        ) return &(m_pColors->cppGType);
			if( token == "gsize"         ) return &(m_pColors->cppGType);
			if( token == "gssize"        ) return &(m_pColors->cppGType);
			if( token == "guchar"        ) return &(m_pColors->cppGType);
			if( token == "guint"         ) return &(m_pColors->cppGType);
			if( token == "guint16"       ) return &(m_pColors->cppGType);
			if( token == "guint32"       ) return &(m_pColors->cppGType);
			if( token == "guint64"       ) return &(m_pColors->cppGType);
			if( token == "guint8"        ) return &(m_pColors->cppGType);
			if( token == "gulong"        ) return &(m_pColors->cppGType);
			if( token == "gushort"       ) return &(m_pColors->cppGType);
			break;
		case 'i':
			if( token == "if"       ) return &(m_pColors->cppKeyword);
			if( token == "inline"   ) return &(m_pColors->cppKeyword);
			if( token == "int"      ) return &(m_pColors->cppType);
			if( token == "internal" ) return &(m_pColors->cppKeyword);
			break;
		case 'k':
			if( token.startsWith("kvi_") ) return &(m_pColors->cppSpecial);
			break;
		case 'l':
			if( token == "long" ) return &(m_pColors->cppType);
			break;
		case 'm':
			if( token.startsWith("m_") ) return &(m_pColors->cppMemberVariable);
			if( token == "mutable"     ) return &(m_pColors->cppKeyword);
			break;
		case 'n':
			if( token == "namespace" ) return &(m_pColors->cppKeyword);
			if( token == "new"       ) return &(m_pColors->cppKeyword);
			break;
		case 'o':
			if( token == "operator" ) return &(m_pColors->cppKeyword);
			if( token == "overload" ) return &(m_pColors->cppKeyword);
			break;
		case 'p':
			if( token == "private"   ) return &(m_pColors->cppKeyword);
			if( token == "protected" ) return &(m_pColors->cppKeyword);
			if( token == "public"    ) return &(m_pColors->cppKeyword);
			break;
		case 'r':
			if( token == "register"         ) return &(m_pColors->cppKeyword);
			if( token == "reinterpret_cast" ) return &(m_pColors->cppKeyword);
			if( token == "return"           ) return &(m_pColors->cppKeyword);
			break;
		case 's':
			if( token == "short"       ) return &(m_pColors->cppType);
			if( token == "signals"     ) return &(m_pColors->cppQSignals);
			if( token == "signed"      ) return &(m_pColors->cppType);
			if( token == "sizeof"      ) return &(m_pColors->cppKeyword);
			if( token == "slots"       ) return &(m_pColors->cppQSignals);
			if( token == "static"      ) return &(m_pColors->cppKeyword);
			if( token == "static_cast" ) return &(m_pColors->cppKeyword);
			if( token == "struct"      ) return &(m_pColors->cppKeyword);
			if( token == "switch"      ) return &(m_pColors->cppKeyword);
			break;
		case 't':
			if( token == "template" ) return &(m_pColors->cppKeyword);
			if( token == "this"     ) return &(m_pColors->cppKeyword);
			if( token == "throw"    ) return &(m_pColors->cppKeyword);
			if( token == "true"     ) return &(m_pColors->cppKeyword);
			if( token == "try"      ) return &(m_pColors->cppKeyword);
			if( token == "typedef"  ) return &(m_pColors->cppKeyword);
			if( token == "typeid"   ) return &(m_pColors->cppKeyword);
			break;
		case 'u':
			if( token == "u_char"   ) return &(m_pColors->cppType);
			if( token == "u_int"    ) return &(m_pColors->cppType);
			if( token == "u_long"   ) return &(m_pColors->cppType);
			if( token == "u_short"  ) return &(m_pColors->cppType);
			if( token == "uchar"    ) return &(m_pColors->cppType);
			if( token == "uint"     ) return &(m_pColors->cppType);
			if( token == "ulong"    ) return &(m_pColors->cppType);
			if( token == "union"    ) return &(m_pColors->cppKeyword);
			if( token == "unsigned" ) return &(m_pColors->cppType);
			if( token == "ushort"   ) return &(m_pColors->cppType);
			break;
		case 'v':
			if( token == "virtual"  ) return &(m_pColors->cppKeyword);
			if( token == "void"     ) return &(m_pColors->cppType);
			if( token == "volatile" ) return &(m_pColors->cppKeyword);
			break;
		case 'w':
			if( token == "while" ) return &(m_pColors->cppKeyword);
			break;
		case 'A':
			if( token == "Atom" ) return &(m_pColors->cppXStuff);
			break;
		case 'B':
			if( token == "Bool" ) return &(m_pColors->cppXStuff);
			break;
		case 'C':
			if( token == "Cardinal" ) return &(m_pColors->cppXStuff);
			if( token == "Colormap" ) return &(m_pColors->cppXStuff);
			if( token == "Cursor"   ) return &(m_pColors->cppXStuff);
			break;
		case 'D':
			if( token == "DIR"      ) return &(m_pColors->cppType);
			if( token == "Depth"    ) return &(m_pColors->cppXStuff);
			if( token == "Display"  ) return &(m_pColors->cppXStuff);
			if( token == "Drawable" ) return &(m_pColors->cppXStuff);
			break;
		case 'F':
			if( token == "FALSE"         ) return &(m_pColors->cppKeyword);
			if( token == "FILE"          ) return &(m_pColors->cppType);
			if( token.startsWith("Font") ) return &(m_pColors->cppXStuff);
			break;
		case 'G':
			if( token[1] == '_'          ) return &(m_pColors->cppGMacro);
			if( token == "GC"            ) return &(m_pColors->cppXStuff);
			if( token.startsWith("GDK_") ) return &(m_pColors->cppGdkMacro);
			if( token.startsWith("GTK_") ) return &(m_pColors->cppGtkMacro);
			if( token.startsWith("Gdk")  ) return &(m_pColors->cppGdkStruct);
			if( token.startsWith("Gtk")  ) return &(m_pColors->cppGtkStruct);
			break;
		case 'K':
			return &(m_pColors->cppKClass);
			break;
		case 'M':
			if( token == "Mask" ) return &(m_pColors->cppXStuff);
			break;
		case 'N':
			if( token == "NULL" ) return &(m_pColors->cppKeyword);
			break;
		case 'P':
			if( token == "Pixmap" ) return &(m_pColors->cppXStuff);
			break;
		case 'Q':
			return &(m_pColors->cppQClass);
			break;
		case 'S':
			if( token == "SIGNAL"       ) return &(m_pColors->cppQSignals);
			if( token == "SLOT"         ) return &(m_pColors->cppQSignals);
			if( token == "Screen"       ) return &(m_pColors->cppXStuff);
			if( token == "ScreenFormat" ) return &(m_pColors->cppXStuff);
			if( token == "Status"       ) return &(m_pColors->cppXStuff);
			break;
		case 'T':
			if( token == "TRUE" ) return &(m_pColors->cppKeyword);
			if( token == "Time" ) return &(m_pColors->cppXStuff);
			break;
		case 'V':
			if( token == "Visual"   ) return &(m_pColors->cppXStuff);
			if( token == "VisualID" ) return &(m_pColors->cppXStuff);
			break;
		case 'W':
			if( token == "Widget" ) return &(m_pColors->cppXStuff);
			if( token == "Window" ) return &(m_pColors->cppXStuff);
			break;
		case 'X':
			return &(m_pColors->cppXStuff);
			break;
	}
	return &(m_pColors->cppNormalText);
}
