/***************************************************************************
 *
 *  $Id: kzengenre.cpp,v 1.11 2005/06/27 16:45:53 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <qpainter.h>
#include <qpixmap.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>

#include "kzenconfig.h"
#include "kzentrack.h"
#include "kzengenre.h"
#include "kzenalbum.h"
#include "kzenartist.h"

QStringList KZenGenre::fGenres;

KZenGenre::KZenGenre(const char * name)
{
  setName(name);

  fTracks = new QPtrList<KZenTrack>(); 
  fTracks->setAutoDelete(false);

  fAlbums.clear();
  fArtists.clear();
}

KZenGenre::~KZenGenre()
{
  delete fTracks;
}

void KZenGenre::setName(const char * name)
{
  fName = name;  

  if (!KZenGenre::getGenres().contains(name)) {
    KZenGenre::getGenres().append(name);
    KZenGenre::getGenres().sort();
  }
}

QStringList & KZenGenre::getGenres()
{
  return fGenres;
}

void KZenGenre::addTrack(KZenTrack * track)
{
  fTracks->append(track);
  
  if (!fAlbums.contains(track->getAlbum())) {
    fAlbums.append(track->getAlbum());
  }
  
  if (!fArtists.contains(track->getArtist())) {
    fArtists.append(track->getArtist());
  }
}

void KZenGenre::removeTrack(KZenTrack * track)
{
  fTracks->remove(track);

  int artistc = 0;
  int albumc = 0;
  KZenTrack * t;
  for (uint tr=0;tr<fTracks->count();tr++) {
    t = fTracks->at(tr);
    if (t->getArtist()==track->getArtist()) artistc++;
    if (t->getAlbum()==track->getAlbum()) albumc++;
  }
  if (artistc==0) {
    fArtists.remove(track->getArtist());
  }
  if (albumc==0) {
    fAlbums.remove(track->getAlbum());
  }
}

uint KZenGenre::getTrackCount()
{
  return fTracks->count();
}

uint KZenGenre::getAlbumCount()
{
  return fAlbums.count();
}

uint KZenGenre::getArtistCount()
{
  return fArtists.count();
}

uint KZenGenre::getLength()
{
  int length = 0;
  KZenTrack * track;
  for (uint tr=0;tr<fTracks->count();tr++) {
    track = fTracks->at(tr);
    length += track->getLength();
  }
  
  return length;
}

KZenGenreItem::KZenGenreItem(QListView * parent, KZenGenre * Genre)
  :KZenItem(parent,KZenItem::ItemTypeGenre)
{
  fGenre = Genre;

  setPixmap(0, UserIcon("genre", 16));
  setText(0, fGenre->getName());

  fSortColumn = KZen::FieldArtist;
  fSortOrder = Qt::Ascending;
}

KZenGenreItem::~KZenGenreItem()
{

}

void KZenGenreItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
  bool sd = KZenConfig::get()->showDetailsGenre();

  QFontMetrics fml(listView()->font());
  int margin = listView()->itemMargin()*2;
  int h = fml.lineSpacing();
  if (sd) h += fml.lineSpacing()*2;
  if (pixmap(0)->height()>h) h = pixmap(0)->height();
  if (h % 2 > 0) h++;
  
  setHeight(h + margin + 2);

  KListView *lv = (KListView *)listView();

  static QPixmap buffer;
  buffer.resize(width, height());

  if(buffer.isNull()) {
    KListViewItem::paintCell(p, cg, column, width, align);
    return;
  }
  
  QPainter pBuf(&buffer, true);
  if (isSelected()) {
    pBuf.fillRect(buffer.rect(), cg.highlight());
  } else {
    pBuf.fillRect(buffer.rect(), KZen::backgroundColor(this));
  }

  QFont font(p->font());
  QFontMetrics fm(p->fontMetrics());

  int text_x = pixmap(0)->width() + 10;
  int text_y = 1;
  int textHeight = fm.lineSpacing() + lv->itemMargin() + 1;

  pBuf.drawPixmap(4, (height() - pixmap(0)->height())/2,
		  *pixmap(0));
  
  if (isSelected()) {
    pBuf.setPen(cg.highlightedText());
  } else {
    pBuf.setPen(cg.text());
  }

  if (sd) font.setBold(true);
  pBuf.setFont(font);
  QFontMetrics fmName(font);

  QString name = text(0);

  if(fmName.width(name) + text_x + lv->itemMargin()*2 > width ) {
    int ellWidth = fmName.width(i18n("..."));
    QString text = QString::fromLatin1("");
    int i = 0;
    int len = name.length();
    while ( i < len && fmName.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
      text += name[i];
      i++;
    }
    name = text + i18n("...");
  }

  pBuf.drawText(text_x, text_y, width, textHeight, AlignVCenter, name);

  if (sd) {
    font.setBold(false);
    pBuf.setFont(font);
    QFontMetrics fmInfo(font);

    QString info;
    info += i18n("1 Artist", "%n Artists", fGenre->getArtistCount());
    info += " - ";
    info += i18n("1 Album", "%n Albums", fGenre->getAlbumCount());
        
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }
    pBuf.drawText(text_x, text_y + textHeight, width, fm.lineSpacing(), AlignVCenter, info);

    font.setItalic(true);
    pBuf.setFont(font);
    fmInfo = QFontMetrics(font);
    
    info = "";
    info += i18n("1 Track", "%n Tracks", fGenre->getTrackCount());
    info += QString(i18n(" - [%2]")).arg(KZen::formatTime(fGenre->getLength()));
    
    if(fmInfo.width(info) + text_x + lv->itemMargin()*2 > width ) {
      int ellWidth = fmInfo.width(i18n("..."));
      QString text = QString::fromLatin1("");
      int i = 0;
      int len = info.length();
      while ( i < len && fmInfo.width(text + name[ i ]) + ellWidth < width - text_x - lv->itemMargin()*2) {
	text += info[i];
	i++;
      }
      info = text + i18n("...");
    }
    pBuf.drawText(text_x, text_y + textHeight*2, width, fm.lineSpacing(), AlignVCenter, info);
  }
  
  pBuf.end();
  p->drawPixmap(0, 0, buffer);
}
