/***************************************************************************
 *
 *  $Id: kzentrackview.cpp,v 1.20 2005/07/03 18:27:52 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <qheader.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qvalidator.h>

#include <kcompletion.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klocale.h>
#include <kactioncollection.h>
#include <kdebug.h>

#include "kzenexplorer.h"
#include "kzenexplorerview.h"
#include "kzenitem.h"
#include "kzencategorydialog.h"
#include "kzenartist.h"
#include "kzenalbum.h"
#include "kzengenre.h"

#include "kzentrackview.h"

KZenTrackView::KZenTrackView(KZenExplorer * explorer,
			     KZenExplorerView * explorerview,
			     QWidget *parent, const char *name)
  :QWidget(parent, name)
{
  fExplorer = explorer;
  fExplorerView = explorerview;
  
  fBusy = false;

  fCurrentNaviItem = 0;
  fCurrentPlayList = 0;

  QVBoxLayout * layout = new QVBoxLayout(this);

  fSearchAndList = new QVBox(this);

  fSearchToolBar = new KToolBar(fSearchAndList, "search toolbar");
  fSearchToolBar->boxLayout()->setSpacing(KDialog::spacingHint());

  fgTrackSource = this;

  layout->addWidget(fSearchAndList);

  int column;
  int count;

  fTrackView = new KZenTrackListView(this,fSearchAndList,"TrackView");
  fTrackView->setViewMode(ViewModeAll);
  count = 0;
  while (KZen::getTitle(count)) {
    column = fTrackView->addColumn(i18n(KZen::getTitle(count)),
				      KZen::getWidth(count));
    fTrackView->setColumnAlignment(column,KZen::getAlignment(count));
    fTrackView->setColumnWidthMode(column,KZen::getWidthMode(count));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(count), column);
    fTrackView->setRenameable(column,KZen::getRenameable(count));
    count++;
  }
  
  connect(fTrackView,SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
	  fExplorerView,SLOT(contextMenuTracks(QListViewItem *, const QPoint &, int)));
  connect(fTrackView,SIGNAL(filesDropped(const QStringList &, KZenPlayList *)),
	  fExplorer,SLOT(filesDropped(const QStringList &, KZenPlayList *)));
  connect(fTrackView,SIGNAL(updatePlayList(KZenPlayList *, QStringList &)),
	  fExplorer,SLOT(updatePlayList(KZenPlayList *, QStringList &)));
  connect(fTrackView,SIGNAL(doubleClicked(QListViewItem *,const QPoint &,int)),
	  this,SLOT(itemDoubleClicked(QListViewItem *,const QPoint &,int)));
  connect(fTrackView,SIGNAL(selectionChanged()),
	  this,SLOT(trackSelectionChanged()));
  connect(fTrackView,SIGNAL(itemRenamed(QListViewItem *, const QString &, int)),
	  fExplorer,SLOT(trackEdited(QListViewItem *, const QString &, int)));
  connect(fTrackView->header(),SIGNAL(clicked(int)),
	  this,SLOT(sortChanged(int)));

  fSortColumn = KZen::FieldArtist;
}

KZenTrackView::~KZenTrackView()
{
  fTrackView->clear();
  delete fTrackView;

  fgTrackSource = 0;
}

void KZenTrackView::setupActions()
{
  KActionCollection * actionCollection =
    static_cast<KMainWindow *>(kapp->mainWidget())->actionCollection();

  fSyncPlayList = actionCollection->action("sync_playlist");

  KAction * fAdvancedSearch = actionCollection->action("advanced_search");
  fAdvancedSearch->plug(fSearchToolBar);

  KAction * fClearSearch = actionCollection->action("clear_search");
  fClearSearch->plug(fSearchToolBar);

  QLabel * label = new QLabel(i18n("Search:"),fSearchToolBar,"searchlab");

  fSearchCombo = new KComboBox(true, fSearchToolBar, "SearchCombo");
  fSearchCombo->setAutoCompletion(true);
  fSearchCombo->setDuplicatesEnabled(false);
  fSearchCombo->setURLDropsEnabled(false);
  fSearchCombo->insertItem("");
  connect(fSearchCombo,SIGNAL(activated(const QString &)),
	  this,SLOT(searchChanged(const QString &)));
  connect(fSearchCombo,SIGNAL(returnPressed(const QString &)),
	  this,SLOT(searchReturnPressed(const QString &)));

  label->setBuddy(fSearchCombo);

  fSearchToolBar->setStretchableWidget(fSearchCombo);
}

void KZenTrackView::sortChanged(int c)
{
  if (fCurrentPlayList) {
    if (fViewMode==ViewModePlayList) {
      fTrackView->setSorting(c,!fTrackView->sortOrder());
      fTrackView->sort();
      fTrackView->setSorting(-1,fTrackView->sortOrder());
      
      QPtrList<KZenTrack> tempList;
      KZenTrack * track;
      KZenTrackItem * item = (KZenTrackItem*) fTrackView->firstChild();
      bool modified = false;
      uint idx = 0;
      while (item) {
	track = item->getTrack();
	if (fCurrentPlayList->getTrackList()->at(idx)!=track) modified = true;
	tempList.append(track);
	item = (KZenTrackItem*)item->nextSibling();
	idx++;
      }
      if (modified) {
	fCurrentPlayList->getTrackList()->clear();
	for (uint tr=0;tr<tempList.count();tr++) {
	  track = tempList.at(tr);
	  fCurrentPlayList->getTrackList()->append(track);
	}
	fCurrentPlayList->setModified(true);
	fSyncPlayList->setEnabled(true);
      }
    }
  }

  if (fCurrentNaviItem) {
    if (fViewMode==ViewModeAll ||
	fViewMode==ViewModeArtist ||
	fViewMode==ViewModeAlbum) {
      fCurrentNaviItem->setSortColumn(getTagField(fTrackView->sortColumn()));
      fCurrentNaviItem->setSortOrder(fTrackView->sortOrder());
    }
  } else {
    if (fViewMode==ViewModeAll) {
      fSortColumn = getTagField(fTrackView->sortColumn());
    }
  }
}

void KZenTrackView::displayTrackList(QPtrList<KZenTrack> * list, ViewMode vm, KZenItem * navi)
{
  fViewMode = vm;

  fCurrentNaviItem = navi;
  fCurrentPlayList = 0;

  setViewMode(vm);

  for (uint i=0;i<list->count();i++) {
    addTrack(list->at(i));
  }
}

void KZenTrackView::displayTrackListPL(KZenPlayList * PL)
{
  fViewMode = ViewModePlayList;

  fCurrentNaviItem = 0;
  fCurrentPlayList = PL;
  
  setViewMode(ViewModePlayList);
  
  QPtrList<KZenTrack> * list = PL->getTrackList();
  
  for (uint i=0;i<list->count();i++) {
    addTrack(list->at(i), PL);
  }
}

void KZenTrackView::setViewMode(ViewMode mode)
{
  fTrackView->clear();
  fTrackView->setViewMode(mode);

  int column;
  if (mode==ViewModeAll) {
    column = KZen::getColumn(KZen::FieldArtist);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldAlbum);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldTitle);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));

    if (fCurrentNaviItem) {
      fTrackView->setSorting(KZen::getColumn(fCurrentNaviItem->getSortColumn()));
      fTrackView->setSortOrder(fCurrentNaviItem->getSortOrder());
    } else {
      fTrackView->setSorting(getColumn(fSortColumn));
      fTrackView->setSortOrder(Qt::Ascending);
    }
  }
  if (mode==ViewModeSearch) {
    column = KZen::getColumn(KZen::FieldArtist);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldAlbum);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldTitle);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    
    fTrackView->setSorting(KZen::getColumn(KZen::FieldArtist));
    fTrackView->setSortOrder(Qt::Ascending);
  }
  if (mode==ViewModeArtist) {
    column = KZen::getColumn(KZen::FieldArtist);
    fTrackView->setColumnWidth(column, 0);
    fTrackView->setColumnWidthMode(column, QListView::Manual);
    fTrackView->header()->setResizeEnabled(false , column);

    column = KZen::getColumn(KZen::FieldAlbum);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldTitle);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    
    if (fCurrentNaviItem) {
      fTrackView->setSorting(KZen::getColumn(fCurrentNaviItem->getSortColumn()));
      fTrackView->setSortOrder(fCurrentNaviItem->getSortOrder());
    } else {
      fTrackView->setSorting(KZen::getColumn(KZen::FieldAlbum));
      fTrackView->setSortOrder(Qt::Ascending);
    }
  }
  if (mode==ViewModeAlbum) {
    column = KZen::getColumn(KZen::FieldArtist);
    fTrackView->setColumnWidth(column, 0);
    fTrackView->setColumnWidthMode(column, QListView::Manual);
    fTrackView->header()->setResizeEnabled(false , column);

    column = KZen::getColumn(KZen::FieldAlbum);
    fTrackView->setColumnWidth(column, 0);
    fTrackView->setColumnWidthMode(column, QListView::Manual);
    fTrackView->header()->setResizeEnabled(false , column);

    column = KZen::getColumn(KZen::FieldTitle);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    
    if (fCurrentNaviItem) {
      fTrackView->setSorting(KZen::getColumn(fCurrentNaviItem->getSortColumn()));
      fTrackView->setSortOrder(fCurrentNaviItem->getSortOrder());
    } else {
      fTrackView->setSorting(KZen::getColumn(KZen::FieldTrack));
      fTrackView->setSortOrder(Qt::Ascending);
    }
  }
  if (mode==ViewModePlayList) {
    column = KZen::getColumn(KZen::FieldArtist);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldAlbum);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));
    fTrackView->setColumnWidthMode(column, KZen::getWidthMode(column));
    fTrackView->header()->setResizeEnabled(KZen::getResizable(column), column);

    column = KZen::getColumn(KZen::FieldTitle);
    fTrackView->setColumnWidth(column, KZen::getWidth(column));

    fTrackView->setSorting(-1);
  }
}

void KZenTrackView::addTrack(KZenTrack * track)
{
  track->setItem(new KZenTrackItem(fTrackView, track));
}

void KZenTrackView::addTrack(KZenTrack * track, KZenPlayList * PL)
{
  QListViewItem * last = fTrackView->lastItem();
  if (last) {
    last = new KZenTrackPLItem(fTrackView, last, track, PL);
    track->setItem((KZenTrackItem*)last);
  } else {
    last = new KZenTrackPLItem(fTrackView, track, PL);
    track->setItem((KZenTrackItem*)last);
  }
}

void KZenTrackView::setSortColumn(KZen::TagField c)
{
  fTrackView->setSorting(KZen::getColumn(c));
}

QPtrList<KZenTrackItem> & KZenTrackView::selectedTracks()
{
  fSelectedTracks.clear();
  
  KZenTrackItem * item = 0;
  
  item = (KZenTrackItem*) fTrackView->firstChild();

  while (item) {
    if (item->isSelected()) {
      fSelectedTracks.append(item);
    }
    item = (KZenTrackItem*) item->nextSibling();
  }

  return fSelectedTracks;
}

void KZenTrackView::refresh()
{
  KZenTrackItem * item = (KZenTrackItem*) fTrackView->firstChild();
  
  while (item) {
    item->getTrack()->refresh();
    item = (KZenTrackItem*) item->nextSibling();
  }
}

void KZenTrackView::searchChanged(const QString & text)
{
  QString search = text.stripWhiteSpace();

  if (!fSearchCombo->contains(search)) {
    fSearchCombo->completionObject()->addItem(search);
  }
  
  fExplorer->searchChanged(search);
}

void KZenTrackView::searchReturnPressed(const QString & text)
{
  if (text.length()==0) {
    searchChanged(text);
  }
}

void KZenTrackView::setSearchText(const char * text)
{
  fSearchCombo->setCurrentText(text);
}

void KZenTrackView::setBusy(bool busy)
{
  fBusy = busy;

  fSearchCombo->setEnabled(!fBusy);
  fTrackView->setEnabled(!fBusy);
}

void KZenTrackView::selectAll()
{
  fTrackView->selectAll(true);
}

void KZenTrackView::itemDoubleClicked(QListViewItem * item, const QPoint & pnt, int c)
{
  if (!item) return;

  KListView * lv = (KListView*)item->listView();
  if (!lv->isRenameable(c)) return;

  KZenTrackItem * i = (KZenTrackItem *)item;
  KZenTrack * track = i->getTrack();

  KZen::TagField Field = KZen::getTagField(c);
  KZen::TagFieldType FieldType = KZen::getTagFieldType(c);

  if (FieldType==KZen::FieldTypeText) {
    if (Field==KZen::FieldCategory) {
      KZenCategoryDialog dlg;
      dlg.setTrack(i->getTrack());

      if (dlg.exec()==QDialog::Accepted) {
	for (uint idx=0;idx<dlg.count();idx++) {
	  track->setCategory(idx,dlg.isChecked(idx));
	}
	track->refresh();
	fExplorer->refreshSmartPlayLists(track);
      }
    } else {
      KLineEdit * t = lv->renameLineEdit();
      KCompletion * comp = t->completionObject();

      if (Field==KZen::FieldArtist) {
	comp->setItems(KZenArtist::getArtists());
      } else if (Field==KZen::FieldAlbum) {
	comp->setItems(KZenAlbum::getAlbums());
      } else if (Field==KZen::FieldGenre) {
	comp->setItems(KZenGenre::getGenres());
      } else {
	comp->clear();
      }
      t->setValidator(0);
      lv->rename(item,c);
    }
  }
  
  if (FieldType==KZen::FieldTypeInt) {
    KLineEdit * t = lv->renameLineEdit();
    t->setValidator(new QIntValidator(0,2500,t,"validator"));
    lv->rename(item,c);
  }

  if (FieldType==KZen::FieldTypeRating) {
    QPoint point = lv->mapFromGlobal(pnt);
    QRect r(lv->itemRect(item));
    int width = 0;
    for (int col=0;col<c;col++) width += lv->columnWidth(col);
    
    int clickX = point.x() - (width + r.x());
    
    uint newRating = 0;
    if (clickX<8) {
      newRating = 0;
    } else if (clickX>5*15) {
      newRating = 5;
    } else {
      newRating = (clickX+8)/15;
    }
    track->setRating(newRating);
    track->refresh();
    fExplorer->refreshSmartPlayLists(track);
    i->repaint();
  }
}

void KZenTrackView::trackSelectionChanged()
{
  kdDebug() << selectedTracks().count() << " track(s) selected" << endl;

  if (selectedTracks().count()==0) {
    fExplorer->setTrackOpsEnabled(false);
  } else {
    fExplorer->setTrackOpsEnabled(true);
  }
}

#include "kzentrackview.moc"
