//LabPlot : InterpolationListDialog.cc

#include <klocale.h>
#include <kmessagebox.h>
#include "InterpolationListDialog.h"
#include "defs.h"

#ifdef HAVE_GSL
#include <gsl/gsl_spline.h>
#endif

InterpolationListDialog::InterpolationListDialog(MainWin *mw, const char *name)
	: ListDialog(mw, name)
{
	setCaption(i18n("Interpolation Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Interpolation" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	new QLabel(i18n(" Type : "),hb);

	typecb = new KComboBox(hb);
	QStringList typelist;
	typelist<<i18n("linear")<<i18n("polynomial")<<i18n("cspline")<<i18n("cspline (periodic)");
	typelist<<i18n("akima")<<i18n("akima (periodic)");
	typecb->insertStringList(typelist);
	typecb->setCurrentItem(config->readNumEntry("Type",0));

	hb = new QHBox(tab1);
	new QLabel(i18n(" From : "),hb);
	ale = new KLineEdit(config->readEntry("From","0"),hb);
	ale->setValidator(new QDoubleValidator(ale));
	new QLabel(i18n(" To : "),hb);
	double b=config->readDoubleNumEntry("To",0.0);
	if(s)
		b=s->Table()->numRows();
	ble = new KLineEdit(QString::number(b),hb);
	ble->setValidator(new QDoubleValidator(ble));
	hb = new QHBox(tab1);
	new QLabel(i18n(" number of points : "),hb);
	nrni = new KIntNumInput(config->readNumEntry("Number",100),hb);
	nrni->setRange(1,INF,1,false);

	hb = new QHBox(tab1);
	sorted = new QCheckBox(i18n("points unsorted or not uniq"),hb);
	sorted->setChecked(config->readBoolEntry("Sorted",true));

	if(p) {
		updateRange();
		QObject::connect(lv,SIGNAL(selectionChanged()),SLOT(updateRange()));
	}
	
	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p && p->getPlot(p->API())->Type() == PSURFACE)
		styletab = surfaceStyle(tw,true);
	else
		styletab = simpleStyle(tw, style, symbol);

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(styletab,i18n("Style"));

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
        QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void InterpolationListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Interpolation" );

	config->writeEntry("Type",typecb->currentItem());
	config->writeEntry("From",ale->text());
	config->writeEntry("To",ble->text());
	config->writeEntry("Number",nrni->value());
	config->writeEntry("Sorted",sorted->isChecked());
}

int InterpolationListDialog::apply_clicked() {
#ifdef HAVE_GSL
	double a=(ale->text()).toDouble();
	double b=(ble->text()).toDouble();
	int number = nrni->value();

	gsl_interp_accel *acc = gsl_interp_accel_alloc();

	const gsl_interp_type *inter_type = gsl_interp_linear;
	switch (typecb->currentItem()) {
	case 0: inter_type = gsl_interp_linear;break;
	case 1: inter_type = gsl_interp_polynomial;break;
	case 2: inter_type = gsl_interp_cspline;break;
	case 3: inter_type = gsl_interp_cspline_periodic;break;
	case 4: inter_type = gsl_interp_akima;break;
	case 5: inter_type = gsl_interp_akima_periodic;break;
	}

	if(s) {
		QTable *table = s->Table();
		int col = table->currentColumn();
		int n = table->numRows();
		
		// add column if needed
		bool empty=true;
		for(int i=0;i<n;i++) {
			if(!table->text(i,table->numCols()-1).isEmpty())
				empty =false;
		}
		if (!empty)
			s->addColumn();

		double *ax = new double[n];
		double *ay = new double[n];
		for(int i=0;i<n;i++){
			ax[i]=i;
			ay[i]=table->text(i,col).toDouble();
		}

		gsl_spline *spline = gsl_spline_alloc(inter_type, n);
		gsl_spline_init(spline, ax, ay, n);
		
		int newcol = table->numCols()-1;
		for (int i = 0;i<number;i++) {
			
			// insert into last column
			table->setText((int)(i*n/number),newcol,QString::number((double) gsl_spline_eval(spline, i*n/number, acc)));
		}

		gsl_spline_free (spline);
		free(ax);free(ay);

		return 0;
	}

	GraphList *gl = p->getPlot(p->API())->getGraphList();
	if(gl->Number()==0) {
		KMessageBox::error(this,i18n("No graph found!"));
		return -2;
	}
	int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());

	GRAPHType st = gl->getStruct(item);

	Style *style=0;
	Symbol *symbol=0;
	if(st != GRAPHM) {
		style = new Style(cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	double xmin=0, xmax=1, ymin=0, ymax=1, zmin=0, zmax=1, tmin=0, tmax=1;
	if (st == GRAPH2D) {
		Graph2D *g = gl->getGraph2D(item);
		int nx = g->Number();
		Point *data = g->Data();

		Point *ptr = new Point[number];

		double *ax = new double[nx];
		double *ay = new double[nx];
		for(int i=0;i<nx;i++){
			ax[i]=data[i].X();
			ay[i]=data[i].Y();
		}
		if (sorted->isChecked()) {
			for(int i=0;i<nx-1;i++) {	// sort data
				for(int j=0;j<nx-1;j++) {
					if(ax[j+1]<ax[j]) {
						double tmpx=ax[j], tmpy=ay[j];
						ax[j]=ax[j+1];
						ay[j]=ay[j+1];
						ax[j+1]=tmpx;
						ay[j+1]=tmpy;
					}
				}
			}
			// remove points with same x
			for(int i=0;i<nx-1;i++) {
				if(ax[i]==ax[i+1]) {
					for(int j=i;j<nx-1;j++) {
						ax[j]=ax[j+1];
						ay[j]=ay[j+1];
					}
					nx--;
					i--;
				}
			}
			kdDebug()<<"	new nx="<<nx<<endl;
		}

		gsl_spline *spline = gsl_spline_alloc(inter_type, nx);
		gsl_spline_init(spline, ax, ay, nx);

		for (int i = 0;i<number;i++) {
			double x = a+i/(double)(number-1)*(b-a);
			double y = gsl_spline_eval(spline, x, acc);

			ptr[i].setPoint(x,y);
		}
		delete[] ax;
		delete[] ay;
		
		mw->calculateRanges2D(ptr,number,&xmin,&xmax,&ymin,&ymax);

		gsl_spline_free (spline);
		
		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		QString fun = QString(i18n("interpolation of ")+g->getLabel()->simpleTitle());

		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,number);
		mw->addGraph2D(ng,sheetcb->currentItem());
	}
	else if (st == GRAPH3D) {
		Graph3D *g = gl->getGraph3D(item);
		int nx = g->Number();
		Point3D *data = g->Data();

		Point3D *ptr = new Point3D[number];

		double *ax = new double[nx];
		double *ay = new double[nx];
		double *az = new double[nx];
		for(int i=0;i<nx;i++){
			ax[i]=data[i].X();
			ay[i]=data[i].Y();
			az[i]=data[i].Z();
		}
		
		if (sorted->isChecked()) {
			for(int i=0;i<nx-1;i++) {	// sort data
				for(int j=0;j<nx-1;j++) {
					if(ax[j+1]<ax[j]) {
						double tmpx=ax[j], tmpy=ay[j], tmpz=az[j];
						ax[j]=ax[j+1];
						ay[j]=ay[j+1];
						az[j]=az[j+1];
						ax[j+1]=tmpx;
						ay[j+1]=tmpy;
						az[j+1]=tmpz;
					}
				}
			}
			// remove points with same x
			for(int i=0;i<nx-1;i++) {
				if(ax[i]==ax[i+1]) {
					for(int j=i;j<nx-1;j++) {
						ax[j]=ax[j+1];
						ay[j]=ay[j+1];
						az[j]=az[j+1];
					}
					nx--;
					i--;
				}
			}
			kdDebug()<<"	new nx="<<nx<<endl;
		}

		gsl_spline *spline = gsl_spline_alloc(inter_type, nx);
		gsl_spline_init(spline, ax, ay, nx);

		for (int i = 0;i<number;i++) {
			double x = a+i/(double)(number-1)*(b-a);
			double y = gsl_spline_eval(spline, x, acc);
			
			double z=0;
			 if(i<nx)
			 	z = az[i];
			
			ptr[i].setPoint(x,y,z);
		}
		delete[] ax;
		delete[] ay;
		delete[] az;
		
		mw->calculateRanges3D(ptr,number,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax);

		gsl_spline_free (spline);
		
		LRange range[3];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);

		QString fun = QString(i18n("interpolation of ")+g->getLabel()->simpleTitle());

		Graph3D *ng = new Graph3D(fun,fun,range,SSPREADSHEET,type,style,symbol,ptr,number,1);
		ng->setNumberX(number);
		ng->setNumberY(g->NY());
		mw->addGraph3D(ng,sheetcb->currentItem(),type);
	}
	else if (st == GRAPH4D) {
		Graph4D *g = gl->getGraph4D(item);
		int nx = g->Number();
		Point4D *data = g->Data();

		Point4D *ptr = new Point4D[number];

		double *ax = new double[nx];
		double *ay = new double[nx];
		double *az = new double[nx];
		double *at = new double[nx];
		for(int i=0;i<nx;i++){
			ax[i]=data[i].X();
			ay[i]=data[i].Y();
			ay[i]=data[i].Z();
			ay[i]=data[i].T();
		}
		
		if (sorted->isChecked()) {
			for(int i=0;i<nx-1;i++) {	// sort data
				for(int j=0;j<nx-1;j++) {
					if(ax[j+1]<ax[j]) {
						double tmpx=ax[j], tmpy=ay[j], tmpz=az[j], tmpt=at[i];
						ax[j]=ax[j+1];
						ay[j]=ay[j+1];
						az[j]=az[j+1];
						at[j]=at[j+1];
						ax[j+1]=tmpx;
						ay[j+1]=tmpy;
						az[j+1]=tmpz;
						at[j+1]=tmpt;
					}
				}
			}
			// remove points with same x
			for(int i=0;i<nx-1;i++) {
				if(ax[i]==ax[i+1]) {
					for(int j=i;j<nx-1;j++) {
						ax[j]=ax[j+1];
						ay[j]=ay[j+1];
						az[j]=az[j+1];
						at[j]=at[j+1];
					}
					nx--;
					i--;
				}
			}
			kdDebug()<<"	new nx="<<nx<<endl;
		}

		gsl_spline *spline = gsl_spline_alloc(inter_type, nx);
		gsl_spline_init(spline, ax, ay, nx);

		for (int i = 0;i<number;i++) {
			double x = a+i/(double)(number-1)*(b-a);
			double y = gsl_spline_eval(spline, x, acc);
			
			double z=0, t=0;
			 if(i<nx) {
			 	z = data[i].Z();
			 	t = data[i].T();
			}

			ptr[i].setPoint(x,y,z,t);
		}
		delete[] ax;
		delete[] ay;
		delete[] az;
		delete[] at;
		
		mw->calculateRanges4D(ptr,number,&xmin,&xmax,&ymin,&ymax,&zmin,&zmax,&tmin,&tmax);

		gsl_spline_free (spline);
		
		LRange range[4];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);
		range[3] = LRange(tmin,tmax);

		QString fun = QString(i18n("interpolation of ")+g->getLabel()->simpleTitle());

		Graph4D *ng = new Graph4D(fun,fun,range,SSPREADSHEET,type,style,symbol,ptr,number,g->GType());
		mw->addGraph4D(ng,sheetcb->currentItem());
	}

	gsl_interp_accel_free (acc);
	
	updateList();
#else
		KMessageBox::error(this, i18n("Sorry. Your installation doesn't support the GSL!"));
#endif

	return 0;
}
