#ifndef LOGGER_H
#define LOGGER_H

#include <Exception.h>
#include <string>

/// Default logtag to use if a source does not provide one
#define LOGTAG "<unspecified>"

////// The log calls

/// Log debugging messages
#if 1
	#define log_debug(msg) \
		Log::Logger::instance()->debug(LOGTAG, __FUNCTION__, msg)
#else
	#define log_debug(msg) \
		do {} while(0) 
#endif

/// Log an informational message on a correct event
#if 1
	#define log_info(msg) \
		Log::Logger::instance()->info(LOGTAG, __FUNCTION__, msg)
#else
	#define log_info(msg) \
		do {} while(0) 
#endif

/// Log a message for an unusual but correct event
#if 1
	#define log_unusual(msg) \
		Log::Logger::instance()->unusual(LOGTAG, __FUNCTION__, msg)
#else
	#define log_unusual(msg) \
		do {} while(0) 
#endif

/// Log a warning message
#if 1
	#define log_warn(msg) \
		Log::Logger::instance()->warn(LOGTAG, __FUNCTION__, msg)
#else
	#define log_warn(msg) \
		do {} while(0) 
#endif

/// Log an error message
#if 1
	#define log_err(msg) \
		Log::Logger::instance()->err(LOGTAG, __FUNCTION__, msg)
#else
	#define log_err(msg) \
		do {} while(0) 
#endif

/// Log a critical event
#if 1
	#define log_crit(msg) \
		Log::Logger::instance()->crit(LOGTAG, __FUNCTION__, msg)
#else
	#define log_crit(msg) \
		do {} while(0) 
#endif

/// Namespace containing all logging functions
namespace Log
{

/// Urgency of a log message
enum LogLevel
{
	DEBUG = 1,
	INFO = 2,
	UNUSUAL = 3,
	WARN = 4,
	ERR = 5,
	CRIT = 6
};

/// Interface for filters to be applied to log messages
class Filter
{
public:
	/// Filter function: return the id of the previously registered output
	/// method to use to log a message with the given log level and tag.
	/// Return -1 if the message should be ignored.
	virtual int filter(LogLevel level, const char* tag) const throw () = 0;
};

class Method
{
public:
	/// Output the message `msg', with message metadata in `level', `tag' and
	/// `func'
	virtual void output(LogLevel level, const char* tag, const char* func,
			const std::string& msg) throw () = 0;

	/// Called in the child process after a fork to allow log methods to
	/// perform needed setup routines
	virtual void setupForkedChild() throw () {}
};

/// Singleton Logger class implementing the logging system
class Logger
{
protected:
	/// Maximum number of supported methods
	static const int max_method = 9;

	/// Singleton logger instance
	static Logger* _instance;

	/// Filter to be applied to log messages; if 0, all messages are logged
	const Filter* filter;

	/// Logging output objects ready to be used by this logger
	Method* methods[max_method];

	/// Log the message `msg' with urgency `level' and tag `tag', from function
	/// `func'
	void log(LogLevel level, const char* tag, const char* func,
						const std::string& msg) throw ()
	{
		int id = filter ? filter->filter(level, tag) : -1;
		if (id != -1 && methods[id])
			methods[id]->output(level, tag, func, msg);
	}

	Logger() : filter(0)
	{
		for (int i = 0; i < max_method; i++)
			methods[i] = 0;
	}

	virtual ~Logger()
	{
		for (int i = 0; i < max_method; i++)
			if (methods[i])
				delete methods[i];
	}

public:
	/// Register an output method in this logger
	void registerMethod(int id, Method* method = 0) throw ()
	{
		if (id < max_method)
			methods[id] = method;
	}

	/// Set the filter for log messages
	void setFilter(const Filter& f) throw () { filter = &f; }

	/// Called in the child process after a fork to allow log methods to
	/// perform needed setup routines
	void setupForkedChild() throw () 
	{
		for (int i = 0; i < max_method; i++)
			if (methods[i])
				methods[i]->setupForkedChild();
	}

	/// Log a debug message
	void debug(const char* tag, const char* func, const std::string& msg)
		throw () { log(DEBUG, tag, func, msg); }
	/// Log an informational message
	void info(const char* tag, const char* func, const std::string& msg)
		throw () { log(INFO, tag, func, msg); }
	/// Log an informational but unusual message
	void unusual(const char* tag, const char* func, const std::string& msg)
		throw () { log(UNUSUAL, tag, func, msg); }
	/// Log a warning message
	void warn(const char* tag, const char* func, const std::string& msg)
		throw () { log(WARN, tag, func, msg); }
	/// Log an error message
	void err(const char* tag, const char* func, const std::string& msg)
		throw () { log(ERR, tag, func, msg); }
	/// Log a critical message
	void crit(const char* tag, const char* func, const std::string& msg)
		throw () { log(CRIT, tag, func, msg); }

	/// Get the singleton logger instance
	static Logger* instance() throw ();
};

};

/*
enum stderr_state_type {
	Yes,
	No,
	Restricted
};

// Notify if stderr is available (true) or not (false)
void set_stderr(enum stderr_state_type val);

// Enable or disable debug output
// Note: debug output is disabled by default
void set_debug(bool val);
*/

// vim:set ts=4 sw=4:
#endif
