/*
    ldapdiff
    Copyright (C) 2000-2002 Thomas.Reith@rhoen.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <lber.h>
#include <ldap.h>
#include <ldap_schema.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "ldapdiff.h"

void ldifschemacreate(LDAP *ld,struct s_schema **sschema)
{
 LDAPMessage        *lr     = NULL;
 LDAPMessage        *lm     = NULL;
 BerElement         *berp;
 struct berval     **pberval     = NULL; 
 struct s_schema    *psschema;
 char               *attr;
 char                *attrs[2];
 int                 rc;
 int                 cn;
 int                 attrcount;
 int                 i;

 attrcount   = 0;
 attrs[0]    = ldifgetgconf(CONFSCHEMAATTRIBUTE);
 attrs[1]    = NULL;

 if(strcmp(ldifgetgconf(CONFSCHEMACHECK),"yes") != 0){
  ldiflog(LOG2,"schema: ldapschema disabled");
  return;
 }

 if((rc = ldap_search_s(ld,ldifgetgconf(CONFSCHEMABASE),LDAP_SCOPE_BASE,ldifgetgconf(CONFSCHEMAFILTER),attrs,0,&lr)) == -1){
  fprintf(stderr,"ldap_search() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  fprintf(stderr,"ldap_err2string(): %s",ldap_err2string(rc));
  exit(-1);
 } 

 if((cn = ldap_count_entries(ld,lr)) == -1){
  fprintf(stderr,"ldap_count_entries() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  ldap_get_option(ld,LDAP_OPT_ERROR_NUMBER,&rc);
  fprintf(stderr,"ldap_err2string(): %s",ldap_err2string(rc));
  exit(-1);
  }
 
 if((lm = ldap_first_entry(ld,lr)) == NULL){
  fprintf(stderr,"ldap_first_entry() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  ldap_get_option(ld,LDAP_OPT_ERROR_NUMBER,&rc);
  fprintf(stderr,"ldap_err2string(): %s",ldap_err2string(rc));
  exit(-1);
 } 

 if((attr = ldap_first_attribute(ld,lm,&berp)) == NULL){
  fprintf(stderr,"ldap_first_attribute() failed: file: %s, line: %d\n",__FILE__,__LINE__);
  ldap_get_option(ld,LDAP_OPT_ERROR_NUMBER,&rc);
  fprintf(stderr,"ldap_err2string(): %s",ldap_err2string(rc));
  exit(-1);
 } 

 while(attr != NULL){
  if((pberval = ldap_get_values_len(ld,lm,attr)) != NULL){
   attrcount = ldap_count_values_len(pberval);
   for(i=0;i<attrcount;i++){
    LDAPAttributeType *lat;
    int                code;
    int                j;
    const char        *errp;

    lat = ldap_str2attributetype(pberval[i]->bv_val,&code,&errp,LDAP_SCHEMA_ALLOW_NONE);
    psschema                   = LDALLOC(1,sizeof(struct s_schema));
    psschema->var              = LDDUP((char*)ldap_attributetype2name(lat));
   
    j = 0;
    while(lat->at_names[j] != NULL){
     j++;
    }
    psschema->at_names = LDALLOC(j+1,sizeof(char*));
    j = 0;
    while(lat->at_names[j] != NULL){
     psschema->at_names[j] = LDDUP(lat->at_names[j]);
     j++;
    }
    
    if(lat->at_equality_oid == NULL){
/*
 jpegPhoto, facsimileTelephoneNumber etc.
 it seems that attributes, with lat->at_syntax_oid == NULL 
 have an implicit matching rule
 openldap schema inconsistency?
*/
     if(lat->at_syntax_oid == NULL && strcmp(ldifgetgconf(CONFSCHEMAHACK),"yes") == 0){
      psschema->at_equality_oid = LDDUP("implicit matching rule");
     }
     else{
      psschema->at_equality_oid = NULL;
     }
    }
    else{
     psschema->at_equality_oid = LDDUP(lat->at_equality_oid);
    } 

/*
 jpegPhoto, facsimileTelephoneNumber etc.
 it seems that attributes, with lat->at_syntax_oid != NULL 
 && lat->at_equality_oid == NULL cannot be multi_valued
 ietf schema definition bug?
*/

    if(lat->at_syntax_oid != NULL && lat->at_equality_oid == NULL && strcmp(ldifgetgconf(CONFSCHEMAHACK),"yes") == 0){
     psschema->at_single_value  = 1;
    }
    else{
     psschema->at_single_value  = lat->at_single_value;
    }
    psschema->next             = NULL;

    if(*sschema == NULL){
     *sschema = psschema;
    }
    else{
     struct s_schema *tsschema;
     tsschema = *sschema;
     while(tsschema->next != NULL){
      tsschema = tsschema->next;
     }
     tsschema->next = psschema;
    }
    ldap_attributetype_free(lat);
   }
   ldap_value_free_len(pberval);
  }
  ldap_memfree(attr);
  attr = ldap_next_attribute(ld,lm,berp);
 } 

 ber_free(berp,0);
 ldap_msgfree(lr);

 ldiflog(LOG2,"schema: %d attributes/aliases read successful",attrcount);
}

int ldifschemaequal(struct s_schema *sschema,char *var)
{
 struct s_schema *psschema;
 int              i;

 if(strcmp(ldifgetgconf(CONFSCHEMACHECK),"yes") != 0){
  return ldifchecknoequality(var);
 }

 psschema = sschema;
 while(psschema != NULL){
  i = 0;
  while(psschema->at_names[i] != NULL){
   if(strcasecmp(var,psschema->at_names[i]) == 0){
    if(psschema->at_equality_oid != NULL){
     return 1;
    }
    else{
     return 0;
    }
   }
   i++;
  }
  psschema = psschema->next;
 }
 fprintf(stderr,"attribute: [%s] does not exist in ldap schema\n",var);
 fprintf(stderr,"ldifschemaequal() failed: file: %s, line: %d\n",__FILE__,__LINE__);
 exit(-1);
}

int ldifschemamulti(struct s_schema *sschema,char *var)
{
 struct s_schema *psschema;
 int              i;

 if(strcmp(ldifgetgconf(CONFSCHEMACHECK),"yes") != 0){
  return ldifcheckmulti(var);
 }

 psschema = sschema;
 while(psschema != NULL){
  i = 0;
  while(psschema->at_names[i] != NULL){
   if(strcasecmp(var,psschema->at_names[i]) == 0){
    if(psschema->at_single_value == 0){
     return 1;
    }
    else{
     return 0;
    }
   }
   i++;
  }
  psschema = psschema->next;
 }
 fprintf(stderr,"attribute: [%s] does not exist in ldap schema\n",var);
 fprintf(stderr,"ldifschemamulti() failed: file: %s, line: %d\n",__FILE__,__LINE__);
 exit(-1);
}

char *ldifschemaattr(struct s_schema *sschema,char *var)
{
 struct s_schema *psschema;
 int              i;
 
 if(strcmp(ldifgetgconf(CONFSCHEMACHECK),"yes") != 0){
  return ldifcheckalias(var);
 }

 psschema = sschema;
 while(psschema != NULL){
  i = 0;
  while(psschema->at_names[i] != NULL){
   if(strcasecmp(var,psschema->at_names[i]) == 0){
    if(i > 0){
     ldiflog(LOG2,"schema: attribute [%s] mapped to [%s]",var,psschema->var);
    }
    return(psschema->var);
   }
   i++;
  }
  psschema = psschema->next;
 }
 fprintf(stderr,"attribute: [%s] does not exist in ldap schema\n",var);
 fprintf(stderr,"ldifschemarealattr() failed: file: %s, line: %d\n",__FILE__,__LINE__);
 exit(-1);
}

void ldifschemareplacealiases(struct s_ldif *sldif,struct s_schema *sschema)
{
 struct s_ldif      *psldif;
 struct s_ldifentry *psldifentry;

 psldif = sldif;
 while(psldif != NULL){
  psldifentry = psldif->attrlist;
  while(psldifentry != NULL){
   char *newvar;
   newvar = LDDUP(ldifschemaattr(sschema,psldifentry->var));
   free(psldifentry->var);
   psldifentry->var = newvar;
   psldifentry = psldifentry->next;
  }
  psldif = psldif->next;
 }
 return;
}

void ldifschemafree(struct s_schema **sschema)
{
 struct s_schema *psschema;
 struct s_schema *tsschema;
 int              i;

 if(strcmp(ldifgetgconf(CONFSCHEMACHECK),"yes") != 0){
  return;
 }

 psschema = *sschema;
 while(psschema != NULL){
  free(psschema->var);
  if(psschema->at_equality_oid != NULL){
   free(psschema->at_equality_oid);
  }
  i = 0;
  while(psschema->at_names[i] != NULL){
   free(psschema->at_names[i]);
   i++;
  }
  free(psschema->at_names);
  tsschema = psschema; 
  psschema = psschema->next;
  free(tsschema);
 }
 *sschema = NULL;
}
