/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <ncurses.h>
#include <string.h>
#include "curse.h"

/*
 * blank a rectangluar screen region
 */
void
wblankrect(WINDOW *win, int miny, int minx, int maxy, int maxx, int saveattr)
{
	int	x, y;
	chtype	ch;

	for (y = miny; y <= maxy; ++y)
		for (x = minx; x <= maxx; ++x)
		{
			if (saveattr)
			{
				ch = mvwinch(win, y, x);
				ch &= ~0xff;
				ch |= ' ';
				mvwaddch(win, y, x, ch);
			}
			else
				mvwaddch(win, y, x, ' ');
		}
}

void
blankrect(int miny, int minx, int maxy, int maxx, int saveattr)
{
	wblankrect(stdscr, miny, minx, maxy, maxx, saveattr);
}

/*
 * Vertically scroll a rectangular screen region
 *
 * Positive lines means move it up, negative means move it down
 */
void
wvscrollrect(WINDOW *win, int miny, int minx, int maxy, int maxx, int lines)
{
	int	x, y;
	chtype	ch;

	while (lines > 0)
	{
		for (y = miny; y < maxy; ++y)
			for (x = minx; x <= maxx; ++x)
			{
				ch = mvwinch(win, y+1, x);
				mvwaddch(win, y, x, ch);
			}
		wblankrect(win, maxy, minx, maxy, maxx, 1);
		--lines;
	}
	while (lines < 0)
	{
		for (y = maxy; y > miny; --y)
			for (x = minx; x <= maxx; ++x)
			{
				ch = mvwinch(win, y-1, x);
				mvwaddch(win, y, x, ch);
			}
		wblankrect(win, miny, minx, miny, maxx, 1);
		++lines;
	}
}

void
vscrollrect(int miny, int minx, int maxy, int maxx, int lines)
{
	wvscrollrect(stdscr, miny, minx, maxy, maxx, lines);
}

/*
 * Horizontally scroll a rectangular screen region
 *
 * Positive cols means move it left, negative means move it right
 */
void
whscrollrect(WINDOW *win, int miny, int minx, int maxy, int maxx, int cols)
{
	int	x, y;
	chtype	ch;

	while (cols > 0)
	{
		for (y = miny; y <= maxy; ++y)
			for (x = minx; x < maxx; ++x)
			{
				ch = mvwinch(win, y, x+1);
				mvwaddch(win, y, x, ch);
			}
		wblankrect(win, miny, maxx, maxy, maxx, 1);
		--cols;
	}
	while (cols < 0)
	{
		for (y = miny; y <= maxy; ++y)
			for (x = maxx; x > minx; --x)
			{
				ch = mvwinch(win, y, x-1);
				mvwaddch(win, y, x, ch);
			}
		wblankrect(win, miny, minx, maxy, minx, 1);
		++cols;
	}
}

void
hscrollrect(int miny, int minx, int maxy, int maxx, int cols)
{
	whscrollrect(stdscr, miny, minx, maxy, maxx, cols);
}

/*
 * Center strings on a line
 */
void
mvwcenter(WINDOW *win, int y, char *str)
{
	int	len = strlen(str);
	
	mvwaddstr(win, y, (getmaxx(win)-len) / 2, str);
}
void
mvcenter(int y, char *str)
{
	mvwcenter(stdscr, y, str);
}
void
mvcenter80(int y, char *str)
{
	int	cols;
	int	len = strlen(str);

	cols = getmaxx(stdscr);
	if (cols > 80)
		cols = 80;

	mvwaddstr(stdscr, y, (cols-len) / 2, str);
}

/*
 * Whole line cursor, created by turning on/off attr.
 *
 * N.B. remember to use wbkgdset() to insure that space characters
 * have the same color attributes that non-space characters do.
 * Otherwise the cursor will look funny.
 *
 * N.B. we don't use the *chgat() routiens because they were broken
 * at one time in 2001.
 */
void
wlinecursor(WINDOW *win, int y, int sx, int wid, int on, attr_t attr)
{
	int	x;
	int	ex;

	if (wid)
		ex = sx + wid;
	else
		ex = getmaxx(win);

	for (x = sx; x < ex; ++x)
	{
		chtype	ch = mvwinch(win, y, x);

		if (on)
			ch |= attr;
		else
			ch &= ~attr;
		mvwaddch(win, y, x, ch);
	}
}

void
linecursor(int y, int sx, int wid, int on, attr_t attr)
{
	wlinecursor(stdscr, y, sx, wid, on, attr);
}

/*
 * Print a window thru a command
 */
void
print_window(WINDOW *win, int numlines, char *printcmd)
{
	FILE	*fp;
	int	x, y;
	char	buf[BUFSIZ];

	if (numlines <= 0)
		numlines = getmaxy(win);

	sprintf(buf, "%s >/dev/null 2>/dev/null", printcmd);
	fp = popen(buf, "w");
	if (!fp)
	{
		beep();
		return;
	}

	for (y = 0; y < numlines; ++y)
	{
		for (x = 0; x < getmaxx(win); ++x)
		{
			chtype	ch;
			chtype	atr;

			ch = mvwinch(win, y, x);
			atr = ch & ~0xff;
			ch &= 0xff;
			if (atr & A_ALTCHARSET)
			{
				switch (ch)
				{
				case 'x': ch = '|'; break;
				case 'q': ch = '-'; break;
				case 'k': ch = '+'; break;
				case 'l': ch = '+'; break;
				case 'j': ch = '+'; break;
				case 'm': ch = '+'; break;
				default: ch = '?'; break;
				}
			}

			if (atr == (A_REVERSE|COLOR_PAIR(1)) && ch == ' ')
				// Hack for charts
				fputc('#', fp);
			else if (atr == (A_REVERSE|COLOR_PAIR(2)) && ch == ' ')
				// Hack for charts
				fputc('#', fp);
			else if (atr & A_BOLD)
			{
				fputc(ch, fp);
				fputc('\010', fp);
				fputc(ch, fp);
			}
			else if (atr & A_UNDERLINE)
			{
				fputc(ch, fp);
				fputc('\010', fp);
				fputc('_', fp);
			}
			else
				fputc(ch, fp);
		}
		fputc('\n', fp);
	}

	pclose(fp);
}

/*
 * print_rect_troff
 * 	convert a rectangular area of 'curscr' to troff commands which
 * 	do a credible job of replicating the screen image.
 *
 * 	The output should be compatible with nroff, groff 1.17 without
 * 	color support, and groff 1.18+ with color support.
 *
 * 	Typical use:
 *		print_rect_troff(getbegy(Win), getbegx(Win),
 *				 getmaxy(Win), getmaxx(Win),
 *				 NULL, "screen.tr");
 */
void
wprint_rect_troff(WINDOW *win, int sy, int sx, int lines, int cols,
					char *spchar, char *filename)
{
	FILE	*fp;
	int	my = sy + lines;
	int	mx = sx + cols;
	int	x, y;
	char	chbuf[2] = {0,0};
	char	*chp;
	char	*fgstr[8] =
		{ "bla", "red", "gre", "yel", "blu", "mag", "cya", "whi" };
	char	*bgstr[8] =
		{ "Bla", "Red", "Gre", "Yel", "Blu", "Mag", "Cya", "Whi" };

	fp = fopen(filename, "w");
	if (!fp)
	{
		beep();
		return;
	}

	fprintf(fp,
"'\\\"------------------------------------------\n"
"'\\\" curses screendump starts here\n"
"'\\\"\n"
"'\\\" curses to troff screen dump conversion helpers\n"
"'\\\"\n"
"'\\\" Define foreground and background colors\n"
"'\\\"\n"
".if mred .ds bla \\m[black]\n"
"'\\\" Some issue with groff - a pure black bg doesn't cause a color change.\n"
".if mred .ds Bla \\M[grey1]\n"
".if mred .ds red \\m[red]\n"
".if mred .ds Red \\M[red]\n"
".if mred .ds gre \\m[green]\n"
".if mred .ds Gre \\M[green]\n"
"'\\\" in curses on an xterm, yellow is actually orange.  Go figure.\n"
".if mred .ds yel \\m[orange]\n"
".if mred .ds Yel \\M[orange]\n"
".if mred .ds blu \\m[blue]\n"
".if mred .ds Blu \\M[blue]\n"
".if mred .ds mag \\m[magenta]\n"
".if mred .ds Mag \\M[magenta]\n"
".if mred .ds cya \\m[cyan]\n"
".if mred .ds Cya \\M[cyan]\n"
".if mred .ds whi \\m[white]\n"
".if mred .ds Whi \\M[white]\n"
".if mred .ds pre \\mP\n"
".if mred .ds Pre \\MP\n"
"'\\\"\n"
"'\\\" Define 1 character wide filled box\n"
"'\\\"\n"
".ds UBOX \\v'+.35v'\\D'P 0 -1.0v   \\w'0'u 0   0 +1.0v   -\\w'0'u 0'"
								"\\v'-.35v'\n"
".ie mred .ds BOX \\*[UBOX]\n"
".el .ds BOX\n"
"'\\\"\n"
"'\\\" Define line drawing characters\n"
"'\\\"\n"
".ie t .ds VL \\h'\\w'0'u/2u'\\v'-.625v'\\D'l 0 1v'\\v'-.375v'\\h'\\w'0'u/2u'\n"
".el .ds VL |\n"
".ie t .ds HL \\v'-0.375v'\\D'l \\w'0'u 0'\\v'+0.375v'\n"
".el .ds HL -\n"
".ie t .ds UL \\h'\\w'0'u/2u'\\v'-0.375v'\\D'l 0 1v'\\v'-1v'"
					"\\D'l \\w'0'u/2u 0'\\v'+0.375v'\n"
".el .ds UL +\n"
".ie t .ds UR \\v'-0.375v'\\D'l \\w'0'u/2u 0'\\D'l 0 1v'"
					"\\v'-0.625v'\\h'\\w'0'u/2u'\n"
".el .ds UR +\n"
".ie t .ds LL \\h'\\w'0'u/2u'\\v'-0.375v'\\D'l 0 -1v'\\v'1v'"
					"\\D'l \\w'0'u/2u 0'\\v'+0.375v'\n"
".el .ds LL +\n"
".ie t .ds LR \\v'-0.375v'\\D'l \\w'0'u/2u 0'\\D'l 0 -1v'"
					"\\v'1.375v'\\h'\\w'0'u/2u'\n"
".el .ds LR +\n"
"'\\\"\n"
"'\\\" Define bold font\n"
"'\\\"\n"
".ie t \\{\\\n"
".	ie '\\*[.T]'ps' .ds CB \\f(CB\n"
".	el .ds CB \\f(CW\n"
".\\}\n"
".el .ds CB \\fB\n"
"'\\\"\n"
"'\\\" curses screen dump starts here\n"
"'\\\"\n"
	);

	fprintf(fp, ".ie mred .ds SPC \\0\n");
	fprintf(fp, ".el \\{\\\n");
	fprintf(fp, ".	ie n .ds SPC %s\n", spchar ? spchar : "\\0");
	fprintf(fp, ".	el .ds SPC \\*[UBOX]\\0\n");
	fprintf(fp, ".\\}\n");

	fprintf(fp, ".TS\n");
	fprintf(fp, "lf(CW).\n");
	for (y = sy; y < my; ++y)
	{
		short	ofg, obg;
		int	obold;

		ofg = -1;
		obg = -1;
		obold = 0;
		for (x = sx; x < mx; ++x)
		{
			chtype	ch;
			attr_t	attr;
			short	pair;
			short	fg, bg;
			int	tmp;

			ch = mvwinch(win, y, x);
			wattr_get(win, &attr, &pair, NULL);
			attr = ch;
			pair = PAIR_NUMBER(attr);
			pair_content(pair, &fg, &bg);
			ch &= 0xff;

			if (0) fprintf(stderr,
					"ch=%lx attr=%lx pair=%d fg=%d bg=%d\n",
					ch, attr, pair, fg, bg);

			chbuf[0] = ch;
			chp = chbuf;

			if (ch == '\\')
				chp = "\\e";

			bg &= 0x07;
			fg &= 0x07;
			if (pair == 0)
			{
				fg = 0;
				bg = 7;
			}

			if (attr & A_BOLD)
			{
				if (!obold)
					fprintf(fp, "\\*(CB");
				obold = 1;
			}
			else
			{
				if (obold)
					fprintf(fp, "\\fP");
				obold = 0;
			}

			if (attr & A_REVERSE)
				{ tmp = fg; fg = bg; bg = tmp; }

			if (attr & A_ALTCHARSET)
			{
				switch (ch)
				{
				case 'x': chp = "\\*[VL]"; break;
				case 'q': chp = "\\*[HL]"; break;
				case 'k': chp = "\\*[UR]"; break;
				case 'l': chp = "\\*[UL]"; break;
				case 'j': chp = "\\*[LR]"; break;
				case 'm': chp = "\\*[LL]"; break;
				default: chp = "?"; break;
				}
			}

			if (fg != ofg)
				fprintf(fp, "\\*[%s]", fgstr[fg]);
			if (bg != obg)
				fprintf(fp, "\\*[%s]", bgstr[bg]);
			if (bg != 7)
				fprintf(fp, "\\*[BOX]");
			if (bg != 7 && ch == ' ')
				fputs("\\*[SPC]", fp);
			else
				fputs(chp, fp);

			ofg = fg;
			obg = bg;
		}
		if (ofg != 0)
			fprintf(fp, "\\*[bla]");
		if (obg != 7)
			fprintf(fp, "\\*[Whi]");
		fputc('\n', fp);
	}
	fprintf(fp, ".TE\n");
	fprintf(fp, "'\\\"\n");
	fprintf(fp, "'\\\" curses screen dump ends here\n");
	fprintf(fp, "'\\\"------------------------------------------\n");

	fclose(fp);
}

void
print_rect_troff(int sy, int sx, int lines, int cols,
		char *spchar, char *filename)
{
	wprint_rect_troff(curscr, sy, sx, lines, cols, spchar, filename);
}
