/*
 * dlock.c - Linux /dev/kmem-based lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 *
 * Chris Eleveld <chris@sector7.com>
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.7 99/11/10 15:02:02 abe Exp $";
#endif


#include "lslk.h"


/*
 * Local static variables
 */

static int Coff = 0;			/* kernel loader format is COFF */
static char *MapNm[] = {		/* possible system map file paths */
	"/System.map",
	"/boot/System.map",
	"/zSystem.map",
	"/boot/zSystem.map",
	"/usr/src/linux/System.map",
	"/usr/src/linux/zSystem.map"
};
#define	MAPNMLEN	(sizeof(MapNm) / sizeof(char *))


/*
 * Local definitions
 */

#define	KMEM	"/dev/kmem"
#define	KSHASHSZ	256		/* kernel symbol hash table size */


/*
 * Local function prototypes
 */

_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static int hash_ksym,(char *nm));
_PROTOTYPE(static char *hasparmhash,(char *nm));
_PROTOTYPE(static void savelock,(struct file_lock *fp, struct inode *ip, struct task_struct *tp));
_PROTOTYPE(static int nlist_private,(char *fnm, struct nlist *nl));


/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	struct file f;
	struct file_lock fl;

#if	LINUXV<1300
	KA_T fa, fta;
#endif	/* LINUXV<1300 */

	KA_T fla, lt;
	struct inode i;
	struct task_struct t;

#if	LINUXV>=1300
/*
 * Linux 1.3 and above kernels store locks in a linked file_lock structure
 * list, starting at the kernel symbol file_lock_table.  The file_lock
 * structures point to their associated inode and task structures.
 */

/*
 * Read lock list pointer.
 */
	if ( ! Nl[X_LCKTBL].n_value
	||  kread((KA_T)Nl[X_LCKTBL].n_value, (char *)&lt, sizeof(lt))) {
	    (void) fprintf(stderr, "%s: can't read lock table pointer: %#x\n",
		Pn, Nl[X_LCKTBL].n_value);
	    Exit(1);
	}
	if (!(fla = lt))
	    return;
	do {

	/*
	 * Read and process the chain of file_lock structures.
	 */
	    if (kread(fla, (char *)&fl, sizeof(fl)))
		return;
	/*
	 * Read the file structure and its inode structure.
	 */
	    if (kread((KA_T)fl.fl_file, (char *)&f, sizeof(f)))
		return;
	    if (kread((KA_T)f.f_inode, (char *)&i, sizeof(i)))
		return;
	/*
	 * Ignore inodes whose reference count is zero.
	 */
	    if (i.i_count) {

	    /*
	     * Read the task structure for the process that owns the lock.
	     */
		if (kread((KA_T)fl.fl_owner, (char *)&t, sizeof(t)))
		    return;
	    /*
	     * Save lock information.
	     */
		(void) savelock(&fl, &i, &t);
		if (is_lock_sel()) {
		    NLockU++;
		    Lp = (struct llock_info *)NULL;
		}
	    }
	    fla = (KA_T)fl.fl_nextlink;
	} while (fla && fla != lt);
#else	/* LINUXV<1300 */
/*
 * Linux 1.2 and below kernels have a linked list of open file structures,
 * starting at the kernel symbol first_file.  The file structures point to
 * inodes.  Inodes with locks have a pointer to file_lock structures, which
 * point in turn to task structures.
 */

/*
 * Read the file table pointer
 */
	if ( ! Nl[X_FIRSTFILE].n_value
	||  kread((KA_T)Nl[X_FIRSTFILE].n_value, (char *)&fta, sizeof(fta))) {
	    (void) fprintf(stderr, "%s: can't read file table pointer: %#x\n",
		Pn, Nl[FIRSTFILE].n_value);
	    exit(1);
	}
	if (!(fa = fta))
	    return;
	do {

	/*
	 * Read the file structure.
	 */
	    if (kread((KA_T)fa, (char *)&f, sizeof(f)))
		return;
	/*
	 * Read the inode.
	 */
	    if (kread((KA_T)f.f_inode, (char *)&i, sizeof(i)))
		return;
	    lt = (KA_T)i.i_flock;
	    if (fla = lt) {
		do {

		/*
		 * Read and process the chain of file_lock structures.
		 */
		    if (kread(fla, (char *)&fl, sizeof(fl)))
			return;
		/*
		 * Read the task structure for the process that owns the lock.
		 */
		    if (kread((KA_T)fl.fl_owner, (char *)&t, sizeof(t)))
			return;

		/*
		 * Save lock information.
		 */
		    (void) savelock(&fl, &i, &t);
		    if (is_lock_sel()) {
			NLockU++;
			Lp = (struct llock_info *)NULL;
		    }
		    fla = (KA_T)fl.fl_next;
		} while (fla && fla != lt);
	    }
	    fa = (KA_T)f.f_next;
	} while (fa && fa != fta);
#endif	/* LINUXV>=1300 */

}


/*
 * get_cmdnm() -- get command name
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	if (lp->cmd)
	    return(lp->cmd);
	return("(unknown)");
}


/*
 * get_kernel_access() -- get access to kernel information
 */

void
get_kernel_access()
{

/*
 * Open kernel memory access.
 */
	if ((Kd = open(KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n",
		Pn, KMEM, strerror(errno));
	    Exit(1);
	}
/*
 * Drop setgid permission.
 */

#if	defined(WILLDROPGID)
	(void) dropgid();
#else	/* !defined(WILLDROPGID) */
	if (Nmlst && !is_readable(Nmlst, 1))
	    Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * If not already specified, get the kernel name list path.
 */
	if (!Nmlst) {
	    if (!(Nmlst = get_nlist_path(1))) {
		(void) fprintf(stderr,
		    "%s: can't determine system map file path\n", Pn);
		Exit(1);
	    }
	}
/*
 * Access kernel symbols.
 */
	if (nlist_private(Nmlst, Nl) < 0) {
	    (void) fprintf(stderr, "%s: can't read kernel name list from %s\n",
		Pn, Nmlst);
	    Exit(1);
	}
}


/*
 * get_nlist_path() - get kernel name list path
 */

char *
get_nlist_path(ap)
	int ap;				/* on success, return an allocated path
					 * string pointer if 1; return a
					 * constant character pointer if 0;
					 * return NULL if failure */
{
	int i;
	struct stat sb;

	for (i = 0; i < MAPNMLEN; i++) {
	    if (stat(MapNm[i], &sb) == 0) {
		if (!ap)
		    return("");
		return(MapNm[i]);
	    }
	}
	return((char *)NULL);
}


/*
 * hash_ksym() - hash kernel symbol by name
 */

static int
hash_ksym(nm)
	char *nm;
{
	int i, j;

	for (i = j = 0; *nm; nm++) {
		i ^= (int)*nm << j;
		j ^= 1;
	}
	return(i % KSHASHSZ);
}


/*
 * hasparmhash() - does symbol name have parameter hash
 */

static char *
hasparmhash(nm)
	char *nm;			/* pointer to symbol name */
{
	char *cp, *rv;
	int n;

	if ((cp = strrchr(nm, '_')) == NULL || *(cp + 1) != 'R')
		return(NULL);
	rv = cp;
	for (cp += 2, n = 0; n < 8; cp++, n++) {
		if ((*cp >= '0' && *cp <= '9')
		||  (*cp >= 'a' && *cp <= 'f')
		||  (*cp >= 'A' && *cp <= 'F'))
			continue;
		return(NULL);
	}
	return((*cp == '\0') ? rv : NULL);
}


/*
 * initialize() -- initialize
 */

void
initialize()
{
	(void) get_kernel_access();
}


/*
 * kread() -- read kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel address */
	char *buf;			/* local receiving buffer address */
	int len	;			/* length to read */
{
	int br;

	if (lseek(Kd, addr, SEEK_SET) == (off_t)-1L)
		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * nlist_private() - get name list
 */

static int
nlist_private(fnm, nl)
	char *fnm;			/* name of file to search */
	struct nlist *nl;		/* names to locate */
{
	unsigned long addr;
	char buf[128];
	char *cp, *eol, *nm;
	int elf = 0;
	int i, j, mm, nf, nk, nn;
	struct kernel_sym *ksa = NULL;
	struct kernel_sym *ks;
	FILE *nfs;
	struct ksym {
		char *name;
		unsigned long addr;
		struct ksym *next;
	} **kh, *kp, *kpn;
	struct nlist *np;

	if ((nfs = fopen(fnm, "r")) == NULL)
		return(-1);
/*
 * Read the kernel symbols via get_kernel_syms().
 */
	if ((nk = get_kernel_syms(NULL)) < 0) {
	    if (Owarn) {
		if (errno == ENOSYS) {
		    (void) fprintf(stderr,
			"%s: WARNING: get_kernel_syms() unimplemented\n", Pn);
		    (void) fprintf(stderr,
		        "      CONFIG_MODULES not defined in autoconf.h?\n");
		} else {
		    (void) fprintf(stderr,
			"%s: get_kernel_syms() error: %s\n",
			Pn, strerror(errno));
		}
		(void) fprintf(stderr,
		    "%s: WARNING: unable to verify symbols in %s\n", Pn, fnm);
	    }
	} else {
	    i = nk * sizeof(struct kernel_sym);
	    if ((ksa = (struct kernel_sym *)malloc((MALLOC_S)i)) == NULL) {
		(void) fprintf(stderr, "%s: no space for kernel symbols\n", Pn);
		Exit(1);
	    }
	}
	if (nk > 0) {
	    if (get_kernel_syms(ksa) < 0) {
		(void) fprintf(stderr, "%s: get_kernel_syms: %s\n",
		    Pn, strerror(errno));
		Exit(1);
	    }
	/*
	 * Allocate hash table space.
	 */
	    if ((kh = (struct ksym **)calloc((MALLOC_S)sizeof(struct ksym),
		      KSHASHSZ))
	    == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte kernel symbol hash table\n",
		    Pn, (KSHASHSZ * sizeof(struct ksym)));
		Exit(1);
	    }
	/*
	 * Scan the kernel symbol table and store their addresses, hashed
	 * by their names for later comparison to the same values in
	 * /[z]System.map.
	 *
	 * Look for the symbol "_system_utsname" or "system_utsname" to
	 * determine the loader format of the kernel: it's * COFF if the
	 * symbol has a leading `_'; ELF if it does not.
	 */
	    for (i = 0, ks = ksa; i < nk; i++, ks++) {
		if (ks->name[0] == '#')
		    continue;
		if ((kp = (struct ksym *)malloc((MALLOC_S)sizeof(struct ksym)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol structure: %s\n",
			Pn, nm);
		    Exit(1);
		}
		if ((nm = hasparmhash(ks->name)) != NULL)
		    *nm = '\0';
		if ((kp->name = (char *)malloc((MALLOC_S)(strlen(ks->name)+1)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol name: %s\n", Pn, nm);
		    Exit(1);
		}
		j = hash_ksym(ks->name);
		(void) strcpy(kp->name, ks->name);
		if (strcmp(ks->name, "_system_utsname") == 0)
		    Coff = 1;
		else if (strcmp(ks->name, "system_utsname") == 0)
		    elf = 1;
		kp->addr = ks->value;
		kp->next = kh[j];
		kh[j] = kp;
	    }
	    (void) free((MALLOC_P *)ksa);
	} else
	    kh = (struct ksym **)NULL;
/*
 * Complain if we don't know the kernel binary's format, COFF or ELF, or if
 * we have conflicting evidence.  In either case, set the default format.
 */
	if ((!Coff && !elf) || (Coff && elf)) {

#if	defined(KERN_LD_ELF)
		Coff = 0;
		elf = 1;
#else	/* !defined(KERN_LD_ELF) */
		Coff = 1;
		elf = 0;
#endif	/* defined(KERN_LD_ELF) */

	    if (Owarn) {
		(void) fprintf(stderr, "%s: WARNING: uncertain kernel", Pn);
		(void) fprintf(stderr, " loader format; assuming %s.\n",

#if	defined(KERN_LD_ELF)
			"ELF"
#else	/* !defined(KERN_LD_ELF) */
			"COFF"
#endif	/* defined(KERN_LD_ELF) */

			);

	    }
	}
/*
 * Read the lines of the name list file.  Look for the symbols defined
 * in Nl[].  Look for any symbols also defined from the get_kernel_syms()
 * call and complain if their addresses don't match.  Count the symbols
 * in Nl[] and quit when addresses have been located for all of them.
 */
	for (nn = 0, np = nl; np->n_name; nn++, np++)
		;
	mm = nf = 0;
	while (nf < nn && fgets(buf, sizeof(buf), nfs) != NULL) {
	    if ((eol = strchr(buf, '\n')) == NULL)
		continue;
	    if ((cp = strchr(buf, ' ')) == NULL)
		continue;
	    nm = cp + 3;
	    *eol = '\0';
	    addr = strtol(buf, &cp, 16);
	    if (kh) {

	    /*
	     * Look for name in the kernel symbol hash table; if it's there,
	     * check the address; complain in detail about the first mismatch;
	     * count mismatches.
	     */
		for (kp = kh[hash_ksym(nm)]; kp; kp = kp->next) {
		    if (strcmp(kp->name, nm) == 0)
			break;
		}
		if (kp && addr != kp->addr) {
		    if (++mm == 1 && Owarn) {
			(void) fprintf(stderr,
			    "%s: kernel symbol address mismatch: %s\n",
			    Pn, nm);
			(void) fprintf(stderr,
			    "      get_kernel_syms() value is %#x;", kp->addr);
			(void) fprintf(stderr, " %s value is %#x.\n",
			    fnm, addr);
		    }
		}
	    }
	/*
	 * Search for symbol in Nl[].
	 */
	    for (np = nl; np->n_name; np++) {
		cp = Coff ? np->n_name : (np->n_name + 1);
		if (strcmp(nm, cp) != 0)
		    continue;
		if (np->n_type) {

		/*
		 * Warn about a duplicate.
		 */
		    if (Owarn)
			(void) fprintf(stderr,
			    "%s: WARNING: %s: ambiguous symbol: %s\n",
			    Pn, fnm, cp);
		    continue;
		}
	    /*
	     * Save address; set type to 1, signifying an address has been
	     * located.  Count the number of addresses located.
	     */
		np->n_value = addr;
		np->n_type = 1;
		nf++;
		break;
	    }
	}
	(void) fclose(nfs);
/*
 * Complete the kernel symbol mismatch message if there were additional
 * mismatches.  Quit if there were any mismatches.
 */
	if (mm) {
	    if (Owarn) {
		if (mm > 1) {
		    (void) fprintf(stderr,
			"      There %s %d additional mismatch%s.\n",
			(mm == 2) ? "was" : "were",
			mm - 1,
			(mm == 2) ? "" : "es");
		 }
		 (void) fprintf(stderr,
			"      %s and the booted kernel may not be a",
			fnm);
		 (void) fprintf(stderr, " matched set.\n");
	    }
	    Exit(1);
	}
/*
 * Free the hashed kernel name space.
 */
	if (kh) {
	    for (i = 0; i < KSHASHSZ; i++) {
		if ((kp = kh[i]) == NULL)
		    continue;
		while (kp) {
		    kpn = kp->next;
		    if (kp->name)
			(void) free((MALLOC_P *)kp->name);
		    (void) free((MALLOC_S *)kp);
		    kp = kpn;
		}
	    }
	    (void) free((MALLOC_P *)kh);
	}
	return(1);
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", major(lp->dev), minor(lp->dev));
	return(buf);
}


/*
 * savelock() -- save lock information
 */

static void
savelock(fp, ip, tp)
	struct file_lock *fp;		/* file_lock structure pointer */
	struct inode *ip;		/* inode structure pointer */
	struct task_struct *tp;		/* task structure pointer */
{
	MALLOC_S len;

/*
 * Allocate a local lock structure.
 */
	(void) alloc_llock();
/*
 * Save: inode number; device; process ID; and size.
 */
	Lp->inum = ip->i_ino;
	Lp->dev = ip->i_dev;
	Lp->pid = (unsigned long)tp->pid;
	Lp->sz = (unsigned long)ip->i_size;
	Lp->szs = 1;
/*
 * Save the lock description: mandatory status; type; start; and
 * length or end. (Linux doesn't support mandatory locking.)
 */
	Lp->mand = Lp->type = 0;
	if (fp->fl_type == F_RDLCK)
	     Lp->type = 1;
	else if (fp->fl_type == F_WRLCK)
	    Lp->type = 2;
	Lp->ss = Lp->es = 1;
	Lp->start = (unsigned long)fp->fl_start;
	Lp->end = (unsigned long)fp->fl_end;
/*
 * There doesn't appear to be remote lock information in the Linux
 * file_lock structure.  Therefore all locks are marked local.
 */
	Lp->src = 0;
/*
 * Save command name from task structure.
 */
	Lp->cmd = (char *)NULL;
	if (tp->comm && (len = (MALLOC_S)strlen(tp->comm)) > 0) {
	    if ((Lp->cmd = (char *)malloc(len + 1)) == (char *)NULL) {
		(void) fprintf(stderr,
		    "%s: no space for PID %ld command: %s\n",
		    Pn, Lp->pid, tp->comm);
		Exit(1);
	    }
	    (void) strcpy(Lp->cmd, tp->comm);
	}
}
