/*
 * safe.c - safe (i.e., non-blocking) versions of kernel functions for lslk
 *
 * V. Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: safe.c,v 1.3 99/11/10 14:57:08 abe Exp $";
#endif


#include "lslk.h"

_PROTOTYPE(static void closePipes,(void));
_PROTOTYPE(static int dolstat,(char *path, char *buf, int len));
_PROTOTYPE(static int doreadlink,(char *path, char *buf, int len));
_PROTOTYPE(static int doinchild,(int (*fn)(), char *fp, char *rbuf, int rbln));

#if	defined(HASINTSIGNAL)
_PROTOTYPE(static int handleint,(int sig));
#else
_PROTOTYPE(static void handleint,(int sig));
#endif


/*
 * Local variables
 */

static pid_t Cpid = 0;			/* child PID */
static jmp_buf Jmp_buf;			/* jump buffer */
static int Pipes[] =			/* pipes for child process */
	{ -1, -1, -1, -1 };


/*
 * childx() - make child process exit
 */

void
childx()
{
	if (Cpid > 1) {
	    if (setjmp(Jmp_buf)) {
		(void) alarm(0);
		(void) signal(SIGALRM, SIG_DFL);
		if (Owarn)
		    (void) fprintf(stderr,
			"%s: WARNING -- child process %d may be hung.\n",
			Pn, Cpid);
	    } else {
		(void) kill(Cpid, SIGINT);
		(void) kill(Cpid, SIGKILL);
		(void) signal(SIGALRM, handleint);
		(void) alarm(TmLimit);
		(void) wait(NULL);
		(void) alarm(0);
		(void) signal(SIGALRM, SIG_DFL);
	    }
	    Cpid = 0;
	    (void) closePipes();
	}
}


/*
 * closePipes() - close open pipe file descriptors
 */

static void
closePipes()
{
	int i;

	for (i = 0; i < 4; i++) {
	    if (Pipes[i] >= 0) {
		(void) close(Pipes[i]);
		Pipes[i] = -1;
	    }
	}
}


/*
 * doinchild() -- do a function in a child process
 */

static int
doinchild(fn, fp, rbuf, rbln)
	int (*fn)();			/* function to perform */
	char *fp;			/* function parameter */
	char *rbuf;			/* response buffer */
	int rbln;			/* response buffer length */
{
	int en, rv;
/*
 * Check reply buffer size.
 */
	if (Oovhd && rbln > MAXPATHLEN) {
	    (void) fprintf(stderr,
		"%s: doinchild error; response buffer too large: %d\n",
		Pn, rbln);
	    Exit(1);
	}
/*
 * Set up to handle an alarm signal; handle an alarm signal; build
 * pipes for exchanging information with a child process; start the
 * child process; and perform functions in the child process.
 */
	if (Oovhd) {
	    if (setjmp(Jmp_buf)) {

	    /*
	     * An alarm signal has been received.
	     */
		(void) alarm(0);
		(void) signal(SIGALRM, SIG_DFL);
		if (setjmp(Jmp_buf)) {
		    if (Owarn && Cpid)
			(void) fprintf(stderr,
			    "%s: WARNING -- child process %d may be hung.\n",
			    Pn, Cpid);
		} else if (Cpid) {
		    (void) kill(Cpid, SIGINT);
		    (void) kill(Cpid, SIGKILL);
		    (void) signal(SIGALRM, handleint);
		    (void) alarm(TmLimit);
		    (void) wait(NULL);
		    (void) alarm(0);
		    (void) signal(SIGALRM, SIG_DFL);
		}
		Cpid = 0;
		(void) closePipes();
		errno = ETIMEDOUT;
		return(1);
	    } else if (!Cpid) {

	    /*
	     * Create pipes to exchange function information with a child
	     * process.
	     */
		if (pipe(Pipes) < 0 || pipe(&Pipes[2]) < 0) {
		    (void) fprintf(stderr, "%s: can't open pipes: %s\n",
			Pn, strerror(errno));
		    Exit(1);
		}
	    /*
	     * Fork a child to execute functions.
	     */
		if ((Cpid = fork()) == 0) {

		/*
		 * Begin the child process.
		 */

		    char r_arg[MAXPATHLEN], r_rbuf[MAXPATHLEN];
		    int r_al, r_rbln;
		    int (*r_fn)();

		    (void) close(Pipes[1]);
		    (void) close(Pipes[2]);
		    Pipes[1] = Pipes[2] = -1;
		/*
		 * Read function requests, process them, and return replies.
		 */
		    for (;;) {
			if (read(Pipes[0], (char *)&r_fn, sizeof(r_fn))
			    != sizeof(r_fn)
			||  read(Pipes[0], (char *)&r_al, sizeof(int))
			    != sizeof(int)
			||  r_al < 1
			||  r_al > sizeof(r_arg)
			||  read(Pipes[0], r_arg, r_al) != r_al
			||  read(Pipes[0], (char *)&r_rbln, sizeof(r_rbln))
			    != sizeof(r_rbln)
			||  r_rbln < 1 || r_rbln > sizeof(r_rbuf))
			    break;
			rv = r_fn(r_arg, r_rbuf, r_rbln);
			en = errno;
			if (write(Pipes[3], (char *)&rv, sizeof(rv))
			    != sizeof(rv)
			||  write(Pipes[3], (char *)&en, sizeof(en))
			    != sizeof(en)
			||  write(Pipes[3], r_rbuf, r_rbln) != r_rbln)
			    break;
		    }
		    _exit(0);
		}
	    /*
	     * Continue in the parent process to finish the setup.
	     */
		if (Cpid < 0) {
		    (void) fprintf(stderr, "%s: can't fork: %s\n",
			Pn, strerror(errno));
		    Exit(1);
		}
		(void) close(Pipes[0]);
		(void) close(Pipes[3]);
		Pipes[0] = Pipes[3] = -1;
	    }
	}
	if (Oovhd) {
	    int len;

	/*
	 * Send a function to the child and wait for the response.
	 */
	    len  = strlen(fp) + 1;
	    (void) signal(SIGALRM, handleint);
	    (void) alarm(TmLimit);
	    if (write(Pipes[1], (char *)&fn, sizeof(fn)) != sizeof(fn)
	    ||  write(Pipes[1], (char *)&len, sizeof(len)) != sizeof(len)
	    ||  write(Pipes[1], fp, len) != len
	    ||  write(Pipes[1], (char *)&rbln, sizeof(rbln)) != sizeof(rbln)
	    ||  read(Pipes[2], (char *)&rv, sizeof(rv)) != sizeof(rv)
	    ||  read(Pipes[2], (char *)&en, sizeof(en)) != sizeof(en)
	    ||  read(Pipes[2], rbuf, rbln) != rbln) {
		(void) alarm(0);
		(void) signal(SIGALRM, SIG_DFL);
		(void) childx();
		errno = ECHILD;
		return(-1);
	    }
	} else {

	/*
	 * Do the operation directly -- not in a child.
	 */
	    (void) signal(SIGALRM, handleint);
	    (void) alarm(TmLimit);
	    rv = fn(fp, rbuf, rbln);
	    en = errno;
	}
/*
 * Function completed, response collected -- complete the operation.
 */
	(void) alarm(0);
	(void) signal(SIGALRM, SIG_DFL);
	errno = en;
	return(rv);
}


/*
 * dolstat() - do a stat or lstat() function
 */

static int
dolstat(path, rbuf, rbln)
	char *path;			/* path */
	char *rbuf;			/* response buffer */
	int rbln;			/* response buffer length */

/* ARGSUSED */

{
	return(STATORLSTAT(path, (struct stat *)rbuf));
}


/*
 * doreadlink() -- do a readlink() function
 */

static int
doreadlink(path, rbuf, rbln)
	char *path;			/* path */
	char *rbuf;			/* response buffer */
	int rbln;			/* response buffer length */
{
	return(readlink(path, rbuf, rbln));
}


/*
 * Exit() - do a clean exit()
 */

void
Exit(xv)
	int xv;				/* exit() value */
{
	(void) childx();
	exit(xv);
}


/*
 * handleint() - handle an interrupt
 */

#if	defined(HASINTSIGNAL)
static int
#else
static void
#endif

/* ARGSUSED */

handleint(sig)
	int sig;
{
	longjmp(Jmp_buf, 1);
}


/*
 * Readlink() - read and interpret file system symbolic links
 */

char *
Readlink(arg)
	char *arg;			/* argument to be interpreted */
{
	char abuf[MAXPATHLEN];
	int alen;
	char *ap;
	char *argp1, *argp2;
	int i, len, llen, slen;
	char lbuf[MAXPATHLEN];
	static int ss = 0;
	char *s1;
	static char **stk = NULL;
	static int sx = 0;
	char tbuf[MAXPATHLEN];
/*
 * See if avoiding kernel blocks.
 */
	if (Oblock) {
		if (Owarn)
		    (void) fprintf(stderr,
			"%s: avoiding readlink(%s): -b was specified.\n",
			Pn, arg);
		return(arg);
	}
/*
 * Evaluate each component of the argument for a symbolic link.
 */
	for (alen = 0, ap = abuf, argp1 = argp2 = arg; *argp2; argp1 = argp2 ) {
		for (argp2 = argp1 + 1; *argp2 && *argp2 != '/'; argp2++)
			;
		if ((len = argp2 - arg) >= MAXPATHLEN) {

path_too_long:
			if (Owarn) {
				(void) fprintf(stderr,
					"%s: path too long: %s\n", Pn, arg);
			}
			return(NULL);
		}
		(void) strncpy(tbuf, arg, len);
		tbuf[len] = '\0';
	/*
	 * Dereference a symbolic link.
	 */
		if ((llen = doinchild(doreadlink, tbuf, lbuf, sizeof(lbuf) - 1))
		>= 0) {
		/*
		 * If the link is a new absolute path, replace
		 * the previous assembly with it.
		 */
			if (lbuf[0] == '/') {
				(void) strncpy(abuf, lbuf, llen);
				ap = &abuf[llen];
				*ap = '\0';
				alen = llen;
				continue;
			}
			lbuf[llen] = '\0';
			s1 = lbuf;
		} else {
			llen = argp2 - argp1;
			s1 = argp1;
		}
	/*
	 * Make sure two components are separated by a `/'.
	 *
	 * If the first component is not a link, don't force
	 * a leading '/'.
	 *
	 * If the first component is a link and the source of
	 * the link has a leading '/', force a leading '/'.
	 */
		if (*s1 == '/') {
			slen = 1;
		} else {
			if (alen > 0) {

			/*
			 * This is not the first component.
			 */
				if (abuf[alen - 1] == '/')
					slen = 1;
				else
					slen = 2;
			} else {

			/*
			 * This is the first component.
			 */
				if (s1 == lbuf && tbuf[0] == '/')
					slen = 2;
				else
					slen = 1;
			}
		}
	/*
	 * Add to the path assembly.
	 */
		if ((alen + llen + slen) >= sizeof(abuf))
			goto path_too_long;
		if (slen == 2)
			*ap++ = '/';
		(void) strncpy(ap, s1, llen);
		ap += llen;
		*ap = '\0';
		alen += (llen + slen - 1);
	}
/*
 * If the assembled path and argument are the same, free all but the
 * last string in the stack, and return the argument.
 */
	if (strcmp(arg, abuf) == 0) {
		for (i = 0; i < sx; i++) {
			if (i < (sx - 1))
				(void) free((MALLOC_P *)stk[i]);
			stk[i] = NULL;
		}
		sx = 0;
		return(arg);
	}
/*
 * If the assembled path and argument are different, add it to the
 * string stack, then Readlink() it.
 */
	if ((s1 = (char *)malloc((MALLOC_S)(alen + 1))) == NULL) {

no_readlink_space:

		(void) fprintf(stderr,
			"%s: no Readlink string space for %s\n", Pn, abuf);
		Exit(1);
	}
	(void) strcpy(s1, abuf);
	if (++sx > ss) {
		if (stk == NULL)
			stk = (char **)malloc(sizeof(char *) * sx);
		else
			stk = (char **)realloc(stk, sizeof(char *) * sx);
		if (stk == NULL)
			goto no_readlink_space;
		ss = sx;
	}
	stk[sx - 1] = s1;
	return(Readlink(s1));
}


/*
 * statsafely() - stat path safely (i. e., with timeout)
 */

int
statsafely(path, buf)
	char *path;			/* file path */
	struct stat *buf;		/* stat buffer address */
{
	if (Oblock) {
	    if (Owarn) 
		(void) fprintf(stderr,
		    "%s: avoiding stat(%s): -b was specified.\n",
		    Pn, path);
	    errno = EWOULDBLOCK;
	    return(1);
	}
	return(doinchild(dolstat, path, (char *)buf, sizeof(struct stat)));
}
