/*
   Copyright (C) 2003 Commonwealth Scientific and Industrial Research
   Organisation (CSIRO) Australia

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

   - Neither the name of CSIRO Australia nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
   PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE ORGANISATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <regex.h>

#include <annodex/annodex.h>

#include "common.h"

#define CMML_PREAMBLE \
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
"<!DOCTYPE cmml SYSTEM \"cmml.dtd\">\n"

/*
 * usage ()
 *
 * print usage information.
 */
static void
usage (void)
{
  printf ("\nUsage: anxgrep [OPTION] ... PATTERN [FILE] ...\n");
  printf ("Search for PATTERN in each annodex FILE or standard input.\n");
  printf ("Example: anxgrep -i 'hello' index.anx\n");
  printf ("\nRexexp selection and interpretation\n");
  printf ("  -E, --extended-regexp     PATTERN is an extended regular expression\n");
  printf ("  -G, --basic-regexp        PATTERN is a basic regular expression\n");
  printf ("  -i, --ignore-case         Ignore case distinctions\n");
  printf ("  -v, --invert-match        Select non-matching elements\n");

  printf ("\nCMML options\n");
  printf ("      --head                Search only the head of each file (ignore clips)\n");
  printf ("      --meta                Search only meta elements (ignore clip descriptions)\n"); 

  printf ("\nOutput options\n");
  printf ("  -o filename, --output filename\n");
  printf ("                            Specify output filename (default is stdout)\n");
  printf ("      --label LABEL         Print LABEL as filename for standard input\n");
  printf ("  -q, --quiet, --silent     Suppress all normal output\n");
  printf ("  -L, --files-without-match Only print FILE names containing no match\n");
  printf ("  -l, --files-with-matches  Only print FILE names containing matches\n");
  printf ("  -c, --count               Only print a count of matching lines per FILE\n");
  printf ("\nMiscellaneous options\n");
  printf ("  -h, --help                Display this help and exit\n");
  printf ("  --version                 Output version information and exit\n\n");
  printf ("\nExit status is 0 if match, 1 if no match\n");
}

static void
usage_err (void)
{
  usage ();
  exit (0);
}

static int found = 0;
static int print_filenames = 1, print_matches = 1;
static int print_nomatch_filenames = 0;
static int print_count = 0;
static int invert = 0;
static int head_only = 0;
static int meta_only = 0;

static FILE * outfile;
static char buf[102400];

static regex_t preg;
static char * regex = NULL;
static int cflags = REG_NOSUB;

static char * label = "(standard input)";

typedef struct {
  const char * filename;
  int matches;
} anxgrep_fileinfo_t;

static int
anxgrep_match (const char * s, anxgrep_fileinfo_t * fileinfo)
{
  int match = 0;

  if (s == NULL) return 0;

  match = (regexec (&preg, s, 0, NULL, 0) != REG_NOMATCH);
  if (invert) match = !match;
  if (match) {
    /* If this is the first match for this file, then print the filename */
    if (print_filenames && fileinfo->matches == 0) {
      fprintf (outfile, "%s", fileinfo->filename);
      /* Include a ':' character if the matches are going to be printed */
      if (print_matches) fputc (':', outfile);
      fputc ('\n', outfile);
    }
    fileinfo->matches++;
    found = 1;
  }

  return match;
}

static int
anxgrep_match_meta (AnxList * l, anxgrep_fileinfo_t * fileinfo)
{
  AnxMetaElement * meta;
  int match = 0;

  for (; l; l = l->next) {
    meta = (AnxMetaElement *)l->data;

    if (anxgrep_match (meta->name, fileinfo) ||
	anxgrep_match (meta->content, fileinfo)) {
      match = 1;
    }
  }

  return match;
}

static int
read_head (ANNODEX * anx, const AnxHead * head, void * user_data)
{
  anxgrep_fileinfo_t * fileinfo = (anxgrep_fileinfo_t *)user_data;
  int match = 0;

  if (anxgrep_match (head->title, fileinfo)) match = 1;
  if (anxgrep_match_meta (head->meta, fileinfo)) match = 1;
      
  if (print_matches && match) {
    anx_head_snprint (buf, 102400, (AnxHead *)head);
    fputs (buf, outfile);
    fputc ('\n', outfile);
  }

  return 0;
}

static int
read_clip (ANNODEX * anx, const AnxClip * clip, void * user_data)
{
  anxgrep_fileinfo_t * fileinfo = (anxgrep_fileinfo_t *)user_data;
  double t;
  int match = 0;

  t = anx_tell_time (anx);
  /*  fprintf (outfile, "%.3f (%ld)\t", t, anx_tell (anx));*/

  if (anxgrep_match_meta (clip->meta, fileinfo)) match = 1;
  if (!meta_only && anxgrep_match (clip->desc_text, fileinfo)) match = 1;

  if (print_matches && match) {
    anx_clip_snprint (buf, 102400, (AnxClip *)clip, t, -1.0);
    fputs (buf, outfile);
    fputc ('\n', outfile);
  }

  return 0;
}

static int
anxgrep_file (const char * infilename)
{
  anxgrep_fileinfo_t fileinfo;
  ANNODEX * anx = NULL;
  long n;

  fileinfo.matches = 0;

  if (infilename == NULL) {
    usage_err ();
  } else {
    if (strcmp (infilename, "-") == 0) {
      fileinfo.filename = label;
      anx = anx_open_stdio (stdin, ANX_READ);
    } else {
      fileinfo.filename = infilename;
      anx = anx_open ((char *)infilename, ANX_READ);
    }
  }

  if (anx == NULL) {
    exit_err ("Error opening annodex %s", infilename);
  }

  anx_set_read_head_callback (anx, read_head, &fileinfo);

  if (!head_only)
    anx_set_read_clip_callback (anx, read_clip, &fileinfo);

  while ((n = anx_read (anx, 1024)) > 0);

  if (n == -1) {
    exit_err ("Failed anx_read: %s\n", anx_strerror (anx));
  }

  if (anx_close (anx) != NULL) {
    exit_err ("Failed close of annodex");
  }

  if (print_nomatch_filenames && fileinfo.matches == 0) {
    fputs (infilename, outfile);
    fputc ('\n', outfile);
  }

  if (print_count) {
    fprintf (outfile, "%s: %d\n", infilename, fileinfo.matches);
  }
  
  return 0;
}

int
main (int argc, char *argv[])
{
  int show_version = 0;
  int show_help = 0;
  int i;

  char * infilename = NULL;
  char * outfilename = NULL;

  set_progname (argv[0]);

#define OPT(s) (strcmp (argv[i], (s)) == 0)
    
  /* Parse options; modify behaviour according to user-specified options */
  for (i=1; i < argc; i++) {
    if (OPT("--help") || OPT("-h")) {
      show_help = 1;
    } else if (OPT("--version")) {
      show_version = 1;
    } else if (OPT("--head")) {
      head_only = 1;
    } else if (OPT("--meta")) {
      meta_only = 1;
    } else if (OPT("--extended-regexp") || OPT("-E")) {
      cflags |= REG_EXTENDED;
    } else if (OPT("--basic-regexp") || OPT("-G")) {
      cflags ^= REG_EXTENDED;
    } else if (OPT("--ignore-case") || OPT("-i")) {
      cflags |= REG_ICASE;
    } else if (OPT("--output") || OPT("-o")) {
      i++; if (i >= argc) usage_err();
      outfilename = argv[i];
    } else if (OPT("--label")) {
      i++; if (i >= argc) usage_err();
      label = argv[i];
    } else if (OPT("--quiet") || OPT("--silent") || OPT("-q")) {
      print_filenames = 0;
      print_matches = 0;
      print_nomatch_filenames = 0;
      print_count = 0;
    } else if (OPT("--files-without-match") || OPT("-L")) {
      print_filenames = 0;
      print_matches = 0;
      print_nomatch_filenames = 1;
      print_count = 0;
    } else if (OPT("--files-with-matches") || OPT("-l")) {
      print_filenames = 1;
      print_matches = 0;
      print_nomatch_filenames = 0;
      print_count = 0;
    } else if (OPT("-c") || OPT("--count")) {
      print_filenames = 0;
      print_matches = 0;
      print_nomatch_filenames = 0;
      print_count = 1;
    } else if (OPT("-v") || OPT("--invert-match")) {
      invert = 1;
    } else {
      if (regex == NULL) {
	regex = argv[i];
      } else {
	break;
      }
    }
  }

  if (show_version) {
    printf ("anxgrep version " VERSION "\n");
  }

  if (show_version || show_help) {
    printf ("# anxgrep, Copyright (C) 2003 CSIRO Australia www.csiro.au ; www.annodex.net\n");
  }

  if (show_help) {
    usage ();
  }

  if (show_version || show_help) {
    exit (0);
  }

  if (outfilename == NULL) {
    outfile = stdout;
  } else {
    if ((outfile = fopen (outfilename, "w")) == NULL) {
      exit_err ("error opening %s for writing", outfilename);
    }
  }

  regcomp (&preg, regex, cflags);

  for (; i < argc; i++) {
    infilename = argv[i];
    anxgrep_file (infilename);
  }

  regfree (&preg);

  if (!found) exit (1);

  exit (0);

}
