#include <buffy/config/Config.h>
#include <wibble/exception.h>

#include <libxml++/libxml++.h>

#include <sys/types.h>  // getpwuid, stat, mkdir, getuid
#include <sys/stat.h>   // stat, mkdir
#include <pwd.h>        // getpwuid
#include <unistd.h>     // stat, getuid

#include <errno.h>

#include<iostream>

//#define TRACE

using namespace std;
using namespace wibble;

namespace buffy {
namespace config {



bool Node::isSet(const std::string& key) const
{
	return this->m_config->isSet(m_path + "/" + key);
}

std::string Node::get(const std::string& key) const
{
	return this->m_config->get(m_path + "/" + key);
}

bool Node::getBool(const std::string& key) const
{
	return this->m_config->getBool(m_path + "/" + key);
}

int Node::getInt(const std::string& key) const
{
	return this->m_config->getInt(m_path + "/" + key);
}

void Node::addDefault(const std::string& key, const std::string& val)
{
	return this->m_config->addDefault(m_path + "/" + key, val);
}

void Node::unset(const std::string& key)
{
	return this->m_config->unset(m_path + "/" + key);
}

void Node::set(const std::string& key, const std::string& val)
{
	return this->m_config->set(m_path + "/" + key, val);
}

void Node::setBool(const std::string& key, bool val)
{
	return this->m_config->setBool(m_path + "/" + key, val);
}

void Node::setInt(const std::string& key, int val)
{
	return this->m_config->setInt(m_path + "/" + key, val);
}

std::vector<std::string> LocationsNode::get() const
{
	return this->m_config->getVector(m_path + "/location");
}

void LocationsNode::set(const std::vector<std::string>& vals)
{
	m_config->setVector(m_path + "/location", vals);
}

std::string MailProgramNode::name() const
{
	size_t end = m_path.rfind(']');
	if (end == string::npos)
		return string();
	size_t beg = m_path.rfind('[', end);
	if (beg == string::npos)
		return string();
	return string(m_path, beg + 1, end-beg - 1);
}

bool MailProgramNode::selected() const
{
	xmlpp::Element* n = m_config->nodeOrDefault(m_path);
	if (n == 0)
		return false;
	xmlpp::Node::NodeList nl = n->get_children("selected");
	if (nl.empty())
		return false;

	for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
		if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			if (xmlpp::TextNode* tn = e->get_child_text())
				return tn->get_content() == "true";
	return false;
}

void MailProgramNode::setName(const std::string& val)
{
	if (name() == val)
		return;
	size_t end = m_path.rfind(']');
	if (end == string::npos)
		return;
	size_t beg = m_path.rfind('[', end);
	if (beg == string::npos)
		return;
	string newpath(m_path, 0, beg);
	newpath += "[" + val + "]";
	m_config->unset(newpath);
	if (xmlpp::Element* el = m_config->node(m_path))
		el->set_attribute("name", val);
	m_path = newpath;
}

void MailProgramNode::setSelected()
{
	// Compute the father node
	size_t pos = m_path.rfind('/');
	if (pos == string::npos)
		return;

	MailProgramsNode father(*m_config, m_path.substr(0, pos));

#ifdef TRACE
	cerr << "MPN::setSelected father " << m_path.substr(0, pos) << endl;
#endif

	string name = this->name();
	vector<MailProgramInfo> info = father.getInfo();
	for (vector<MailProgramInfo>::iterator i = info.begin();
			i != info.end(); ++i)
		i->selected = (i->name == name);

	father.set(info);
}

void MailProgramNode::run(const MailFolder& folder)
{
	string cmd = command();
    string::size_type p;
	while ((p = cmd.find("%p")) != string::npos)
		cmd.replace(p, 2, folder.path());

	// TODO: use '~' as working directory
	vector<string> argv;
	argv.push_back("/bin/sh");
	//argv.push_back("/bin/sh");
	argv.push_back("-c");
	argv.push_back(cmd);

	// I wonder what this does, as it's undocumented
//	Glib::spawn_async(".", argv, Glib::SpawnFlags(0), sigc::mem_fun(*this, &MailProgramImpl::on_exit));

	
	pid_t child = fork();
	if (child == -1)
		throw wibble::exception::System("trying to fork a child process");
	
	if (child == 0)
	{
		// Child code path
		try {
			if (execl("/bin/sh", "/bin/sh", "-c", cmd.c_str(), 0) == -1)
				throw wibble::exception::System("trying to fork a child process");
			throw wibble::exception::System("trying to fork a child process");
		} catch (wibble::exception::Generic& e) {
			cerr << e.type() << ": " << e.fullInfo() << endl;
		}
		_exit(0);
	}
}

void MailProgramsNode::convertOld(xmlpp::Element* n) const
{
	// If nothing was found, try converting old style entries
	xmlpp::Node::NodeList nl = n->get_children("mail");
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
		if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
		{
			if (!e->get_attribute("name"))
			{
				// Convert the name child element into a name attribute
				xmlpp::Node::NodeList names = e->get_children("name");
				for (xmlpp::Node::NodeList::const_iterator namei = names.begin(); namei != names.end(); ++namei)
					if (xmlpp::Element* nameEl = dynamic_cast<xmlpp::Element*>(*namei))
					{
						if (xmlpp::TextNode* tn = nameEl->get_child_text())
							e->set_attribute("name", tn->get_content());
						e->remove_child(nameEl);
					}
			}
			if (xmlpp::Attribute* selected = e->get_attribute("selected"))
			{
				if (selected->get_value() == "true")
				{
					xmlpp::Element* val = e->add_child("selected");
					val->set_child_text("true");
				}
				e->remove_attribute("selected");
			}
		}
}

std::vector<MailProgramNode> MailProgramsNode::get() const
{
	xmlpp::Element* n = m_config->nodeOrDefault(m_path);
	if (n == 0)
		return vector<MailProgramNode>();
	xmlpp::Node::NodeList nl = n->get_children("mail");

	vector<MailProgramNode> res;
	for (int loops = 0; loops < 2; ++loops)
	{
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
				if (xmlpp::Attribute* name = e->get_attribute("name"))
					res.push_back(MailProgramNode(*this->m_config, this->m_path + "/mail[" + name->get_value() + "]"));

		// If it didn't find anything, try converting from old (insane) XML
		// layout
		if (!res.empty())
			break;
		else
			convertOld(n);
	}

	return res;
}

std::vector<MailProgramInfo> MailProgramsNode::getInfo() const
{
	xmlpp::Element* n = m_config->nodeOrDefault(m_path);
	if (n == 0)
		return vector<MailProgramInfo>();
	xmlpp::Node::NodeList nl = n->get_children("mail");

	vector<MailProgramInfo> res;
	for (int loops = 0; loops < 2; ++loops)
	{
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
				if (xmlpp::Attribute* name = e->get_attribute("name"))
				{
					MailProgramNode node(*this->m_config, this->m_path + "/mail[" + name->get_value() + "]");
					res.push_back(MailProgramInfo(name->get_value(), node.command(), node.selected()));
				}

		// If it didn't find anything, try converting from old (insane) XML
		// layout
		if (!res.empty())
			break;
		else
			convertOld(n);
	}
	return res;
}

void MailProgramsNode::set(const std::vector<MailProgramInfo>& vals)
{
	xmlpp::Element* el = this->m_config->node(this->m_path);
	if (el == 0) return;
	xmlpp::Node::NodeList nl = el->get_children("mail");
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
		el->remove_child(*i);

	for (vector<MailProgramInfo>::const_iterator i = vals.begin();
			i != vals.end(); ++i)
	{
		xmlpp::Element* item = el->add_child("mail");
		item->set_attribute("name", i->name);

		if (i->selected)
		{
			xmlpp::Element* val = item->add_child("selected");
			val->set_child_text("true");
		}

		xmlpp::Element* val = item->add_child("command");
		val->set_child_text(i->command);
	}
}

MailProgramNode MailProgramsNode::selected() const
{
	xmlpp::Element* n = m_config->nodeOrDefault(m_path);
	if (n == 0) throw wibble::exception::Consistency("No mail programs accessible at " + m_path);
	xmlpp::Node::NodeList nl = n->get_children("mail");

	for (int loops = 0; loops < 2; ++loops)
	{
		// First try to get the first selected element
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
				if (xmlpp::Attribute* name = e->get_attribute("name"))
				{
					MailProgramNode node(*this->m_config, this->m_path + "/mail[" + Storage::escape(name->get_value()) + "]");
					if (node.selected())
						return node;
				}

		// Then just try to get the first element
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin(); i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
				if (xmlpp::Attribute* name = e->get_attribute("name"))
					return MailProgramNode(*this->m_config, this->m_path + "/mail[" + Storage::escape(name->get_value()) + "]");

		// If it didn't find anything, try converting from old (insane) XML
		// layout
		convertOld(n);
	}

	throw wibble::exception::Consistency("No mail programs available at " + m_path);
}

#if 0
void MailProgramsNode::select(const std::string& name)
{
	string sel(selected());
	if (!sel.empty())
		Node(*this->m_config, this->m_path + "/mail[" + sel + "]").unset("selected");

	Node(*this->m_config, this->m_path + "/mail[" + name + "]").setBool("selected", true);
}
#endif

void Config::init()
{
	struct passwd* udata = getpwuid(getuid());
	rcfile = udata->pw_dir;
	rcfile += "/.buffy";

	// Fill in defaults
	addDefault("general/interval", "300");
	addDefault("general/view/important", "true");
	addDefault("general/view/empty", "false");
	addDefault("general/view/read", "false");

	vector<string> locations;
	locations.push_back(string("/var/mail/") + udata->pw_name);
	locations.push_back(string(udata->pw_dir) + "/Maildir");
	locations.push_back(string(udata->pw_dir) + "/Mail");
	locations.push_back(string(udata->pw_dir) + "/mail");
	addDefault("general/locations/location", locations);

	addDefault("general/programs/mail[Mutt]/selected", "true");
	addDefault("general/programs/mail[Mutt]/command", "/usr/bin/x-terminal-emulator -e /usr/bin/mutt -f %p");
	addDefault("general/programs/mail[Other]/command", "/usr/bin/sample-mail-editor --folder %p");

#ifdef TRACE
	cerr << "Initial defaults: ";
	doc_defaults->write_to_stream_formatted(cerr);
#endif
}

void Config::load(const std::string& file)
{
	if (access(file.c_str(), F_OK) == -1)
		initEmpty("buffy");
	else
		Storage::load(file);
}

void Config::save()
{
	save(rcfile);
}


Node Config::application(const std::string& name)
{
	return Node(*this, "applications/app[" + Storage::escape(name) + "]");
}

MailProgramsNode Config::mailPrograms()
{
	return MailProgramsNode(*this, "general/programs");
}

MailProgramNode Config::mailProgram(const std::string& name)
{
	return MailProgramNode(*this, "general/programs/mail[" + Storage::escape(name) + "]");
}

FolderNode Config::folder(const std::string& folder)
{
	return FolderNode(*this, "folders/folder[" + Storage::escape(folder) + "]");
}


}
}

//#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>
#include <buffy/MailboxMailFolder.h>

namespace tut {
using namespace buffy::config;
using namespace wibble::tests;

struct buffy_config_shar {
	Config conf;
	buffy_config_shar() : conf("nonexisting-config.txt") {
	}
	~buffy_config_shar() {
		unlink("test-config-saved.txt");
	}
};
TESTGRP(buffy_config);

// Check an application value
template<> template<>
void to::test<1>()
{
	ensure_equals(conf.application("foo").get("bar"), "");
	conf.application("foo").addDefault("bar", "baz");
	ensure_equals(conf.application("foo").get("bar"), "baz");
	conf.application("foo").set("bar", "cippo");
	ensure_equals(conf.application("foo").get("bar"), "cippo");
}

// Check an application value with path
template<> template<>
void to::test<2>()
{
	ensure_equals(conf.application("foo").get("bar/baz"), "");
	conf.application("foo").addDefault("bar/baz", "lippo");
	ensure_equals(conf.application("foo").get("bar/baz"), "lippo");
	conf.application("foo").set("bar/baz", "lippo1");
	ensure_equals(conf.application("foo").get("bar/baz"), "lippo1");
}

// Check mail programs
template<> template<>
void to::test<3>()
{
	vector<MailProgramNode> programs = conf.mailPrograms().get();
	ensure_equals(programs.size(), 2u);

	ensure_equals(conf.mailPrograms().selected().name(), "Mutt");

	conf.mailProgram("Other").setSelected();
	ensure_equals(conf.mailPrograms().selected().name(), "Other");
	ensure_equals(conf.mailProgram("Mutt").selected(), false);
}

// Check that squares, used as special values internally, parse correctly
template<> template<>
void to::test<4>()
{
	ensure_equals(conf.application("foo]").get("bar"), "");
	conf.application("foo]").set("bar", "cippo2");
	ensure_equals(conf.application("foo]").get("bar"), "cippo2");
}

// Check loading an existing configuration
template<> template<>
void to::test<5> ()
{
	Config conf0("test-config.txt");

	ensure(!conf0.view().read());
	ensure(!conf0.view().empty());
	ensure(conf0.view().important());
	ensure_equals(conf0.general().interval(), 600);

	vector<string> locations = conf0.locations().get();
	ensure_equals(locations.size(), 4u);

#if 0
	/*
	   gen_ensure(locations.find("/var/mail/enrico") != locations.end());
	   gen_ensure(locations.find("/home/enrico/Maildir") != locations.end());
	   gen_ensure(locations.find("/home/enrico/Mail") != locations.end());
	   gen_ensure(locations.find("/home/enrico/mail") != locations.end());
	*/
#endif

	vector<MailProgramNode> programs = conf0.mailPrograms().get();
	ensure_equals(programs.size(), 2u);

	ensure_equals(conf0.mailPrograms().selected().name(), "Mutt");
}

// Check persistance when saving and reloading an existing configuration
template<> template<>
void to::test<6> ()
{
	Config conf0;
	stringstream str;
	str << getpid();
	string testString(str.str());
	buffy::MailFolder testFolder(new buffy::MailboxMailFolder("mbox/empty.mbox"));

	// Set a few nonstandard values, then save them
	vector<string> locations;
	locations.clear();
	locations.push_back("foo");
	locations.push_back("bar");
	locations.push_back("baz");
	conf0.locations().set(locations);
	conf0.application("test").set("pid", testString);
	conf0.folder(testFolder).setForceView(true);
	conf0.folder(testFolder).setForceHide(false);
	conf0.save("test-config-saved.txt");

	// Load the config file that we just saved
	Config conf1("test-config-saved.txt");
	ensure_equals(conf1.application("test").get("pid"), testString);
	ensure_equals(conf1.folder(testFolder).forceview(), true);
	ensure_equals(conf1.folder(testFolder).forcehide(), false);
	ensure_equals(conf1.locations().get().size(), 3u);
}

}

//#endif


// vim:set ts=4 sw=4:
