#include <buffy/config/Storage.h>
#include <wibble/exception.h>

#include <libxml++/libxml++.h>

#include <sys/types.h>  // getpwuid, stat, mkdir, getuid
#include <sys/stat.h>   // stat, mkdir
#include <pwd.h>        // getpwuid
#include <unistd.h>     // stat, getuid

#include <errno.h>

#include <sstream>
#include <iostream>

//#define TRACE

using namespace std;
using namespace wibble;

namespace buffy {
namespace config {

string Storage::escape(const std::string& str)
{
	string res;
	for (string::const_iterator s = str.begin();
			s != str.end(); s++)
		switch (*s)
		{
			case '/': res += "%47;"; break;
			case '[': res += "%91;"; break;
			case ']': res += "%93;"; break;
			case '%': res += "%%"; break;
			default: res += *s; break;
		}
	return res;
}

string Storage::unescape(const std::string& str)
{
	string res;
	for (string::const_iterator s = str.begin();
			s != str.end(); ++s)
		if (*s == '%')
		{
			++s;
			if (s == str.end() || *s == '%')
				res += '%';
			else
			{
				string num;
				for ( ; s != str.end() && *s != ';'; s++)
					num += *s;
				res += (char)atoi(num.c_str());
			}
		} else
			res += *s;
	return res;
}


Storage::Storage()
	: m_xml_parser(0), doc_conf(0), doc_conf_needs_freeing(0), doc_defaults(0), m_el_root(0)
{
	doc_defaults = new xmlpp::Document();
	doc_defaults->create_root_node("buffy");
	m_def_root = doc_defaults->get_root_node();
}

Storage::~Storage()
{
	if (doc_conf_needs_freeing)
		delete doc_conf;
	if (m_xml_parser)
		delete m_xml_parser;
	if (doc_defaults)
		delete doc_defaults;
}

void Storage::initEmpty(const std::string& rootName)
{
//	feedback("Creating new configuration file %.*s.\n", PFSTR(rcfile));
		
	// Create the configuration from scratch
	doc_conf = new xmlpp::Document();
	doc_conf->create_root_node("buffy");
	//_root = doc_conf->create_root_node("buffy");
	doc_conf_needs_freeing = true;
	m_el_root = doc_conf->get_root_node();
}

void Storage::load(const std::string& rcfile)
{
	struct stat rcfile_stat;
	doc_conf_needs_freeing = false;
	if (stat(rcfile.c_str(), &rcfile_stat) == -1)
		throw wibble::exception::File(rcfile, "reading stat() information");

	if (!m_xml_parser)
		m_xml_parser = new xmlpp::DomParser();

	if (S_ISDIR(rcfile_stat.st_mode))
		throw wibble::exception::Consistency(rcfile + " already exists and is a directory");

	if (access(rcfile.c_str(), R_OK) == -1)
		throw wibble::exception::Consistency(rcfile + " already exists and is not readable");

	//		feedback("Reading configuration from %.*s.\n", PFSTR(rcfile));

	// Parse the existing config file
	try {
		// Enable when we'll have a DTD
		//_xmlparser.set_validate();
		m_xml_parser->set_substitute_entities(); // Resolve/unescape text automatically
		m_xml_parser->parse_file(rcfile);
		if (*m_xml_parser)
			doc_conf = m_xml_parser->get_document();
		else
			throw wibble::exception::Consistency("Parser did not parse " + rcfile);
	} catch (const std::exception& ex) {
		throw wibble::exception::Consistency(string(ex.what()) + " when parsing configuration file " + rcfile);
	}

	m_el_root = doc_conf->get_root_node();
}

void Storage::save(const std::string& file)
{
	try {
		if (doc_conf)
		{
//			feedback("Saving configuration to %.*s.\n", PFSTR(rcfile));
			doc_conf->write_to_file_formatted(file);
		}// else
//			feedback("No configuration present: ignoring save request.\n");
	} catch (const std::exception& ex) {
		throw wibble::exception::Consistency(string(ex.what()) + " when saving configuration file " + file);
	}
}

void Storage::dump(std::ostream& out)
{
	if (doc_defaults)
	{
		out << "Default tree:" << endl;
		doc_defaults->write_to_stream_formatted(out);
	}
	if (doc_conf)
	{
		out << "Values tree:" << endl;
		doc_conf->write_to_stream_formatted(out);
	}
}


xmlpp::Element* Storage::getElement(xmlpp::Element* father, const std::string& path) const
{
	size_t pos = path.find('/');
	if (pos == string::npos)
	{
		if (path[path.size() - 1] == ']')
		{
			size_t npos = path.find('[');
			if (npos == string::npos)
				return 0;
			string attr = unescape(path.substr(npos+1, path.size() - npos - 2));
			string npath = path.substr(0, npos);

			xmlpp::Node::NodeList nl = father->get_children(npath);
			for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
					i != nl.end(); i++)
				if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
					if (xmlpp::Attribute* at = e->get_attribute("name"))
						if (at->get_value() == attr)
							return e;
			return 0;
		} else {
			xmlpp::Node::NodeList nl = father->get_children(path);
			if (nl.empty())
				return 0;
			return dynamic_cast<xmlpp::Element*>(*nl.begin());
		}
	}
	else
	{
		xmlpp::Element* n = getElement(father, path.substr(0, pos));
		return n != 0 ? getElement(n, path.substr(pos+1)) : 0;
	}
}

xmlpp::Element* Storage::obtainElement(xmlpp::Element* father, const std::string& path)
{
	size_t pos = path.find('/');
	if (pos == string::npos)
	{
		size_t npos;
		if (path[path.size() - 1] == ']' && ((npos = path.find('[')) != string::npos))
		{
			string attr = unescape(path.substr(npos+1, path.size() - npos - 2));
			string npath = path.substr(0, npos);

			xmlpp::Node::NodeList nl = father->get_children(npath);
			for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
					i != nl.end(); i++)
				if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
					if (xmlpp::Attribute* at = e->get_attribute("name"))
						if (at->get_value() == attr)
							return e;

			xmlpp::Element* app = father->add_child(npath);
			app->set_attribute("name", attr);
			return app;
		} else {
			xmlpp::Node::NodeList nl = father->get_children(path);
			if (nl.empty())
				return father->add_child(path);
			return dynamic_cast<xmlpp::Element*>(*nl.begin());
		}
	}
	else
	{
		xmlpp::Element* n = obtainElement(father, path.substr(0, pos));
		return n != 0 ? obtainElement(n, path.substr(pos+1)) : 0;
	}
}

void Storage::addDefault(const std::string& key, const std::string& val)
{
	if (xmlpp::Element* el = obtainElement(m_def_root, key))
		el->set_child_text(val);
}

void Storage::addDefault(const std::string& key, const std::vector<std::string>& val)
{
	size_t pos = key.rfind('/');
	if (pos == string::npos)
		return;
	xmlpp::Element* father = obtainElement(m_def_root, key.substr(0, pos));
	if (father == 0)
		return;
	string name(key, pos + 1);
	for (std::vector<std::string>::const_iterator i = val.begin(); i != val.end(); ++i)
	{
		xmlpp::Element* el = father->add_child(name);
		el->set_child_text(*i);
	}
}

xmlpp::Element* Storage::getDefault(const std::string& key) const
{
	return getElement(m_def_root, key);
}

xmlpp::Element* Storage::nodeIfExists(const std::string& path) const
{
	return getElement(m_el_root, path);
}

xmlpp::Element* Storage::nodeOrDefault(const std::string& path) const
{
	xmlpp::Element* res = getElement(m_el_root, path);
	return res ? res : res = getDefault(path);
}

xmlpp::Element* Storage::node(const string& path)
{
	return obtainElement(m_el_root, path);
}

bool Storage::isSet(const std::string& path) const
{
	return nodeIfExists(path) != 0;
}

std::string Storage::get(const std::string& path) const
{
	if (xmlpp::Element* n = nodeOrDefault(path))
		if (xmlpp::TextNode* tn = n->get_child_text())
			return tn->get_content();
	return string();
}

bool Storage::getBool(const std::string& path) const
{
	return get(path) == "true";
}

int Storage::getInt(const std::string& path) const
{
	return strtoul(get(path).c_str(), 0, 10);
}

std::vector<std::string> Storage::getVector(const std::string& path) const
{
	size_t pos = path.rfind('/');
	if (pos == string::npos) return vector<std::string>();

	xmlpp::Element* n = this->nodeOrDefault(path.substr(0, pos));
	if (n == 0) return vector<std::string>();

	xmlpp::Node::NodeList nl = n->get_children(path.substr(pos + 1));
	if (nl.empty()) return vector<std::string>();

	vector<string> res;
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			res.push_back(e->get_child_text()->get_content());
	return res;
}

std::vector<std::string> Storage::children(const std::string& path) const
{
	xmlpp::Element* n = nodeOrDefault(path);
	if (n == 0)
		return vector<std::string>();

	xmlpp::Node::NodeList nl = n->get_children();
	if (nl.empty())
		return vector<std::string>();

	vector<string> res;
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			res.push_back(e->get_name());
	return res;
}

std::vector<std::string> Storage::childNames(const std::string& path) const
{
	xmlpp::Element* n = nodeOrDefault(path);
	if (n == 0)
		return vector<std::string>();

	xmlpp::Node::NodeList nl = n->get_children();
	if (nl.empty())
		return vector<std::string>();

	vector<string> res;
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			if (xmlpp::Attribute* a = e->get_attribute("name"))
				res.push_back(a->get_value());
	return res;
}

void Storage::unset(const std::string& path)
{
	size_t pos = path.rfind('/');
	xmlpp::Element* father;
	if (pos == string::npos)
		father = m_el_root;
	else
		father = nodeIfExists(path.substr(0, pos));
	if (!father) return;
	xmlpp::Element* el = nodeIfExists(path);
	if (!el) return;
	father->remove_child(el);
}

void Storage::set(const std::string& path, const std::string& val)
{
	if (xmlpp::Element* el = node(path))
		el->set_child_text(val);
}

void Storage::setBool(const std::string& path, bool val)
{
	set(path, val ? "true" : "false");
}

void Storage::setInt(const std::string& path, int val)
{
	stringstream str;
	str << val;
	set(path, str.str());
}

void Storage::setVector(const std::string& path, const std::vector<std::string>& val)
{
	size_t pos = path.rfind('/');
	if (pos == string::npos) return;
	string name(path, pos+1);

	xmlpp::Element* el = node(path.substr(0, pos));
	xmlpp::Node::NodeList nl = el->get_children(name);

	// First remove all location nodes
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		el->remove_child(*i);

	// Then add the new ones
	for (vector<string>::const_iterator i = val.begin();
			i != val.end(); i++)
		el->add_child(name)->set_child_text(*i);
}

}
}

//#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace buffy::config;
using namespace wibble::tests;

struct buffy_config_storage_shar {
	Storage st;
	buffy_config_storage_shar() {
		st.initEmpty("buffy");
		st.addDefault("nick", "enrico");
		st.addDefault("view/empty", "true");
		st.addDefault("view/time", "10");
	}
	~buffy_config_storage_shar() {
		unlink("test-config-saved.txt");
	}
};
TESTGRP(buffy_config_storage);

// Test escaping functions
template<> template<>
void to::test<1>()
{
	ensure_equals(Storage::unescape(Storage::escape("foo")), "foo");
	ensure_equals(Storage::unescape(Storage::escape("f%oo")), "f%oo");
	ensure_equals(Storage::unescape(Storage::escape("f/oo")), "f/oo");
	ensure_equals(Storage::unescape(Storage::escape("f[o]o")), "f[o]o");
	ensure_equals(Storage::unescape(Storage::escape("f[o]o%")), "f[o]o%");
	ensure_equals(Storage::unescape(Storage::escape("f[o]o/")), "f[o]o/");
}

// Check string values
template<> template<>
void to::test<2>()
{
	// No initial default
	ensure_equals(st.get("name"), "");
	st.set("name", "Enrico");
	ensure_equals(st.get("name"), "Enrico");
	st.unset("name");
	ensure_equals(st.get("name"), "");

	// Setting an initial default
	st.addDefault("name", "Enrico");
	ensure_equals(st.get("name"), "Enrico");
	st.set("name", "Maurizio");
	ensure_equals(st.get("name"), "Maurizio");
	st.unset("name");
	ensure_equals(st.get("name"), "Enrico");

	// With initial default
	ensure_equals(st.get("nick"), "enrico");
	st.set("nick", "tanno");
	ensure_equals(st.get("nick"), "tanno");
	st.unset("nick");
	ensure_equals(st.get("nick"), "enrico");
}

// Check bool values
template<> template<>
void to::test<3>()
{
	// No initial default
	ensure_equals(st.getBool("view/read"), false);
	st.setBool("view/read", true);
	ensure_equals(st.getBool("view/read"), true);
	st.unset("view/read");
	ensure_equals(st.getBool("view/read"), false);

	// Setting an initial default
	st.addDefault("view/read", "true");
	ensure_equals(st.getBool("view/read"), true);
	st.setBool("view/read", false);
	ensure_equals(st.getBool("view/read"), false);
	st.unset("view/read");
	ensure_equals(st.getBool("view/read"), true);

	// With initial default
	ensure_equals(st.getBool("view/empty"), true);
	st.setBool("view/empty", false);
	ensure_equals(st.getBool("view/empty"), false);
	st.unset("view/empty");
	ensure_equals(st.getBool("view/empty"), true);
}

// Check int values
template<> template<>
void to::test<4>()
{
	// No initial default
	ensure_equals(st.getInt("view/pause"), 0);
	st.setInt("view/pause", 10);
	ensure_equals(st.getInt("view/pause"), 10);
	st.unset("view/pause");
	ensure_equals(st.getInt("view/pause"), 0);

	// Setting an initial default
	st.addDefault("view/pause", "10");
	ensure_equals(st.getInt("view/pause"), 10);
	st.setInt("view/pause", 5);
	ensure_equals(st.getInt("view/pause"), 5);
	st.unset("view/pause");
	ensure_equals(st.getInt("view/pause"), 10);

	// With initial default
	ensure_equals(st.getInt("view/time"), 10);
	st.setInt("view/time", 5);
	ensure_equals(st.getInt("view/time"), 5);
	st.unset("view/time");
	ensure_equals(st.getInt("view/time"), 10);
}

// Check that no autovivification is happening
template<> template<>
void to::test<7> ()
{
	// Start with just defaults
	Storage st0;
	st0.initEmpty("moo");

	st0.addDefault("prova", "300");

	// Access the default value
	ensure_equals(st0.getInt("prova"), 300);
	ensure(! st0.isSet("prova"));

	// Save and reload
	st0.save("test-config-saved.txt");
	Storage st1;
	st1.load("test-config-saved.txt");

	// Check that the default value didn't get saved
	ensure_equals(st1.isSet("prova"), false);
}

}

//#endif


// vim:set ts=4 sw=4:
