/**
 * @file libgalago-gtk/galago-gtk-service-list.c
 *       Service list widget
 *
 * @Copyright (C) 2005-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-service-list.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-stock.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>

struct _GalagoGtkServiceListPriv
{
	gboolean show_people;
	gboolean show_all_contacts;

	GalagoService *service;
	GalagoGtkServiceListItemType type;

	GList *valid_services;

	gboolean populate_lock;
};

enum
{
	COLUMN_DATA,
	COLUMN_TYPE,
	COLUMN_ICON,
	COLUMN_TEXT,
	NUM_COLUMNS
};

enum
{
	SELECTION_CHANGED,
	SERVICE_ACTIVATED,
	LAST_SIGNAL
};

static void galago_gtk_service_list_class_init(GalagoGtkServiceListClass *klass);
static void galago_gtk_service_list_init(GalagoGtkServiceList *service_list);
static void galago_gtk_service_list_finalize(GObject *obj);
static void galago_gtk_service_list_destroy(GtkObject *obj);
static void galago_gtk_service_list_row_activated(GtkTreeView *treeview,
												  GtkTreePath *path,
												  GtkTreeViewColumn *column);

static GtkTreeViewClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

GType
galago_gtk_service_list_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkServiceListClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_service_list_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkServiceList),
			0,
			(GInstanceInitFunc)galago_gtk_service_list_init
		};

		type = g_type_register_static(GTK_TYPE_TREE_VIEW,
									  "GalagoGtkServiceList", &info, 0);
	}

	return type;
}

static void
galago_gtk_service_list_class_init(GalagoGtkServiceListClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkTreeViewClass *treeview_class;

	galago_gtk_stock_init();

	parent_class = g_type_class_peek_parent(klass);

	gobject_class  = G_OBJECT_CLASS(klass);
	object_class   = GTK_OBJECT_CLASS(klass);
	treeview_class = GTK_TREE_VIEW_CLASS(klass);

	gobject_class->finalize = galago_gtk_service_list_finalize;

	object_class->destroy = galago_gtk_service_list_destroy;

	treeview_class->row_activated = galago_gtk_service_list_row_activated;

	signals[SELECTION_CHANGED] =
		g_signal_new("selection-changed",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkServiceListClass,
									 selection_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	signals[SERVICE_ACTIVATED] =
		g_signal_new("service-activated",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkServiceListClass,
									 service_activated),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__UINT_POINTER,
					 G_TYPE_NONE, 2,
					 G_TYPE_UINT,
					 G_TYPE_POINTER);
}

static void
selection_changed_cb(GtkTreeSelection *sel, GalagoGtkServiceList *service_list)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GalagoService *service;
	GalagoGtkServiceListItemType type;

	if (!gtk_tree_selection_get_selected(sel, &model, &iter))
		return;

	gtk_tree_model_get(model, &iter,
					   COLUMN_DATA, &service,
					   COLUMN_TYPE, &type,
					   -1);

	if (type == GALAGO_GTK_SERVICE_LIST_ITEM_SERVICE)
		galago_gtk_service_list_set_service(service_list, service);
	else
		galago_gtk_service_list_set_item_type(service_list, type);

	g_signal_emit(service_list, signals[SELECTION_CHANGED], 0);
}

static void
append_service(GalagoGtkServiceList *service_list, GtkListStore *model,
			   GalagoService *service)
{
	GdkPixbuf *pixbuf;
	GtkTreeIter iter;

	if (galago_service_get_accounts(service, TRUE) == NULL)
		return;

	pixbuf = galago_gdk_pixbuf_new_from_service_with_size(service,
		GTK_ICON_SIZE_MENU);

	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter,
					   COLUMN_DATA, service,
					   COLUMN_TYPE, GALAGO_GTK_SERVICE_LIST_ITEM_SERVICE,
					   COLUMN_ICON, pixbuf,
					   COLUMN_TEXT, galago_service_get_name(service),
					   -1);

	if (pixbuf != NULL)
		g_object_unref(G_OBJECT(pixbuf));
}

static void
populate_services_list(GalagoGtkServiceList *service_list)
{
	GtkListStore *model;
	GdkPixbuf *pixbuf;
	GtkTreeIter iter;

	if (service_list->priv->populate_lock)
		return;

	service_list->priv->populate_lock = TRUE;

	model =
		GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(service_list)));

	gtk_list_store_clear(model);

	if (service_list->priv->show_people)
	{
		/* People */
		pixbuf = gtk_widget_render_icon(GTK_WIDGET(service_list),
										GALAGO_GTK_STOCK_PERSON,
										GTK_ICON_SIZE_MENU, NULL);
		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter,
						   COLUMN_TYPE, GALAGO_GTK_SERVICE_LIST_ITEM_PEOPLE,
						   COLUMN_ICON, pixbuf,
						   COLUMN_TEXT, _("People"),
						   -1);
		g_object_unref(G_OBJECT(pixbuf));
	}

	if (service_list->priv->show_all_contacts)
	{
		/* All Contacts */
		/*
		 * TODO: Don't hardcode. This should use category tagging, which
		 *       needs to be added to Galago at some point as an account
		 *       property.
		 */
		pixbuf = galago_gdk_pixbuf_new_generic(16, 16);
		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter,
						   COLUMN_TYPE,
						   GALAGO_GTK_SERVICE_LIST_ITEM_ALL_CONTACTS,
						   COLUMN_ICON, pixbuf,
						   COLUMN_TEXT, _("All Contacts"),
						   -1);
		g_object_unref(G_OBJECT(pixbuf));
	}

#if GTK_CHECK_VERSION(2,6,0)
	if (service_list->priv->show_people ||
		service_list->priv->show_all_contacts)
	{
		/* Separator */
		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter, COLUMN_TEXT, NULL, -1);
	}
#endif

	if (service_list->priv->valid_services != NULL)
	{
		GList *l;

		for (l = service_list->priv->valid_services; l != NULL; l = l->next)
		{
			append_service(service_list, model, (GalagoService *)l->data);
		}
	}
	else
	{
		GList *l;

		for (l = galago_get_services(GALAGO_REMOTE, TRUE);
			 l != NULL;
			 l = l->next)
		{
			append_service(service_list, model, (GalagoService *)l->data);
		}
	}

	service_list->priv->populate_lock = FALSE;
}

#if GTK_CHECK_VERSION(2,6,0)
static gboolean
services_row_separator_func(GtkTreeModel *model, GtkTreeIter *iter,
							gpointer user_data)
{
	char *text;

	gtk_tree_model_get(model, iter, COLUMN_TEXT, &text, -1);

	if (text == NULL)
		return TRUE;

	g_free(text);

	return FALSE;
}
#endif /* GTK_CHECK_VERSION(2,6,0) */

static gint
sort_services_func(GtkTreeModel *model, GtkTreeIter *iter_a,
				   GtkTreeIter *iter_b, gpointer user_data)
{
	GalagoGtkServiceListItemType type_a;
	char *text_a, *text_b;
	gint result;

	gtk_tree_model_get(model, iter_a,
					   COLUMN_TYPE, &type_a,
					   COLUMN_TEXT, &text_a,
					   -1);
	gtk_tree_model_get(model, iter_b,
					   COLUMN_TEXT, &text_b,
					   -1);

	if (type_a == GALAGO_GTK_SERVICE_LIST_ITEM_PEOPLE ||
		type_a == GALAGO_GTK_SERVICE_LIST_ITEM_ALL_CONTACTS ||
		text_a == NULL)
	{
		result = -1;
	}
	else
	{
		result = g_ascii_strcasecmp(text_a, text_b);
	}

	if (text_a != NULL)
		g_free(text_a);

	if (text_b != NULL)
		g_free(text_b);

	return result;
}

static void
galago_gtk_service_list_init(GalagoGtkServiceList *service_list)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *model;
	GtkTreeSelection *sel;

	service_list->priv = g_new0(GalagoGtkServiceListPriv, 1);

	model = gtk_list_store_new(NUM_COLUMNS, G_TYPE_POINTER, G_TYPE_INT,
							   GDK_TYPE_PIXBUF, G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(service_list), GTK_TREE_MODEL(model));
	gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(model),
									COLUMN_TEXT,
									sort_services_func, service_list, NULL);
	g_object_unref(G_OBJECT(model));

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(service_list), FALSE);
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(service_list), TRUE);
	gtk_tree_view_set_search_column(GTK_TREE_VIEW(service_list),
									COLUMN_TEXT);

#if GTK_CHECK_VERSION(2,6,0)
	gtk_tree_view_set_row_separator_func(GTK_TREE_VIEW(service_list),
										 services_row_separator_func,
										 NULL, NULL);
#endif

	column = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(service_list), column);

	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer,
									   "pixbuf", COLUMN_ICON);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_add_attribute(column, renderer,
									   "text", COLUMN_TEXT);

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(service_list));
	g_signal_connect(G_OBJECT(sel), "changed",
					 G_CALLBACK(selection_changed_cb), service_list);

	populate_services_list(service_list);
}

static void
galago_gtk_service_list_finalize(GObject *obj)
{
	GalagoGtkServiceList *service_list;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(obj));

	service_list = GALAGO_GTK_SERVICE_LIST(obj);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_service_list_destroy(GtkObject *obj)
{
	GalagoGtkServiceList *service_list;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(obj));

	service_list = GALAGO_GTK_SERVICE_LIST(obj);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
galago_gtk_service_list_row_activated(GtkTreeView *treeview, GtkTreePath *path,
									  GtkTreeViewColumn *column)
{
	GalagoGtkServiceList *service_list;

	service_list = GALAGO_GTK_SERVICE_LIST(treeview);

	g_signal_emit(service_list, signals[SERVICE_ACTIVATED], 0,
				  galago_gtk_service_list_get_item_type(service_list),
				  galago_gtk_service_list_get_service(service_list));

	if (parent_class->row_activated != NULL)
		parent_class->row_activated(treeview, path, column);
}

GtkWidget *
galago_gtk_service_list_new(void)
{
	return GTK_WIDGET(g_object_new(GALAGO_GTK_TYPE_SERVICE_LIST, NULL));
}

void
galago_gtk_service_list_set_show_people(GalagoGtkServiceList *service_list,
										gboolean show_people)
{
	g_return_if_fail(service_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list));

	if (service_list->priv->show_people == show_people)
		return;

	service_list->priv->show_people = show_people;

	populate_services_list(service_list);
}

void
galago_gtk_service_list_set_show_all_contacts(
	GalagoGtkServiceList *service_list, gboolean show_all_contacts)
{
	g_return_if_fail(service_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list));

	if (service_list->priv->show_all_contacts == show_all_contacts)
		return;

	service_list->priv->show_all_contacts = show_all_contacts;

	populate_services_list(service_list);
}

void
galago_gtk_service_list_set_service(GalagoGtkServiceList *service_list,
									GalagoService *service)
{
	g_return_if_fail(service_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list));

	if (service == service_list->priv->service)
		return;

	service_list->priv->service = service;

	if (service == NULL)
	{
		galago_gtk_service_list_set_item_type(service_list,
			GALAGO_GTK_SERVICE_LIST_ITEM_ALL_CONTACTS);
	}
	else
	{
		GtkTreeIter iter;
		GtkTreeModel *model;
		GalagoService *temp_service;
		gboolean valid;

		service_list->priv->type = GALAGO_GTK_SERVICE_LIST_ITEM_SERVICE;

		model = gtk_tree_view_get_model(GTK_TREE_VIEW(service_list));

		for (valid = gtk_tree_model_get_iter_first(model, &iter);
			 valid;
			 valid = gtk_tree_model_iter_next(model, &iter))
		{
			gtk_tree_model_get(model, &iter,
							   COLUMN_DATA, &temp_service,
							   -1);

			if (service == temp_service)
			{
				GtkTreeSelection *sel =
					gtk_tree_view_get_selection(GTK_TREE_VIEW(service_list));

				gtk_tree_selection_select_iter(sel, &iter);
				return;
			}
		}

		g_assert_not_reached();
	}
}

void
galago_gtk_service_list_set_item_type(GalagoGtkServiceList *service_list,
									  GalagoGtkServiceListItemType type)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	gboolean valid;
	GalagoGtkServiceListItemType item_type;

	g_return_if_fail(service_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list));
	g_return_if_fail(type != GALAGO_GTK_SERVICE_LIST_ITEM_SERVICE);
	g_return_if_fail(type != GALAGO_GTK_SERVICE_LIST_ITEM_UNSET);
	g_return_if_fail(service_list->priv->show_people ||
					 type != GALAGO_GTK_SERVICE_LIST_ITEM_PEOPLE);
	g_return_if_fail(service_list->priv->show_all_contacts ||
					 type != GALAGO_GTK_SERVICE_LIST_ITEM_ALL_CONTACTS);

	if (service_list->priv->type == type)
		return;

	service_list->priv->type = type;
	service_list->priv->service = NULL;

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(service_list));

	for (valid = gtk_tree_model_get_iter_first(model, &iter);
		 valid;
		 valid = gtk_tree_model_iter_next(model, &iter))
	{
		gtk_tree_model_get(model, &iter,
						   COLUMN_TYPE, &item_type,
						   -1);

		if (item_type == type)
		{
			GtkTreeSelection *sel =
				gtk_tree_view_get_selection(GTK_TREE_VIEW(service_list));

			gtk_tree_selection_select_iter(sel, &iter);
			return;
		}
	}

	g_assert_not_reached();
}

void
galago_gtk_service_list_set_valid_services(GalagoGtkServiceList *service_list,
										   GList *services)
{
	g_return_if_fail(service_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list));

	if (service_list->priv->valid_services == services)
		return;

	g_list_free(service_list->priv->valid_services);

	service_list->priv->valid_services = services;
}

GalagoGtkServiceListItemType
galago_gtk_service_list_get_item_type(const GalagoGtkServiceList *service_list)
{
	g_return_val_if_fail(service_list != NULL,
						 GALAGO_GTK_SERVICE_LIST_ITEM_UNSET);
	g_return_val_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list),
						 GALAGO_GTK_SERVICE_LIST_ITEM_UNSET);

	return service_list->priv->type;
}

GalagoService *
galago_gtk_service_list_get_service(const GalagoGtkServiceList *service_list)
{
	g_return_val_if_fail(service_list != NULL,                     NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_SERVICE_LIST(service_list), NULL);

	return service_list->priv->service;
}
