/******************************************************************************************************************************************
 cscrollview.c
******************************************************************************************************************************************/

#include "cscrollview.h"
#include "ctreeview.h"
#include "ctextview.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CScrollViewListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScrollViewListener::CScrollViewListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScrollViewListener::~CScrollViewListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CScrollView);

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnScroll
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::OnScroll (GtkScrolledWindow *inWidget, GtkScrollType inType, gboolean, gpointer inData)
{
	// get the scroll view gtkol instance
	CWidget *theWidget = reinterpret_cast <CWidget *> (inData);

	// send the notification
	if (theWidget != NULL && theWidget -> GetListener() != NULL)
		static_cast <CScrollViewListener *> (theWidget -> GetListener()) -> OnScroll (theWidget, inType);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScrollView::CScrollView (CContainer *inOwner, const CWidgetListener *inListener)
	    :CContainer	 (inOwner, inListener)
{
	// widget instanciation process request
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CScrollView::~CScrollView ()
{
	// widget deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CScrollView::PerformWidgetInstanciate ()
{
	return ::gtk_scrolled_window_new (NULL, NULL);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::PerformWidgetInitialize ()
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the gtk widget pointer
	if (inGtkWidget == NULL) return;

	// signal connection
	::g_signal_connect (G_OBJECT(inGtkWidget), "scroll-child", G_CALLBACK(CScrollView::OnScroll), this);

	// set scroll bars...
	::gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(inGtkWidget), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	// show the gtk widget
	Show ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// child addon
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::PerformContainerAdd (CWidget *inChild)
{
	// get the child gtk widget
	GtkWidget *inChildGtkWidget = inChild != NULL ? inChild -> GetGtkWidget() : NULL;

	// get this gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL || inChildGtkWidget == NULL) return;

	// some widgets have native scrolling attributes
	if (inChild -> ClassIs (__metaclass(CTreeView)) || inChild -> ClassIs (__metaclass(CTextView)))
	{
		::gtk_container_add (GTK_CONTAINER(inGtkWidget), inChildGtkWidget);
	}
	// no native slots...
	else
	{
		// add the child widget, because the scrollview is a container there is no need to handle multiple children insertion
		// requests, just try to avoid gtk errors on the automatic allocated gtk view append request
		if (::gtk_bin_get_child(GTK_BIN(inGtkWidget)) == NULL || 
		    ::gtk_bin_get_child(GTK_BIN(::gtk_bin_get_child(GTK_BIN(inGtkWidget)))) == NULL) 
			::gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(inGtkWidget), inChildGtkWidget);
		else
			::gtk_container_add (GTK_CONTAINER(::gtk_bin_get_child(GTK_BIN(inGtkWidget))), inChildGtkWidget);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// child removal
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::PerformContainerRemove (CWidget *inChild)
{
	// get the child gtk widget
	GtkWidget *inChildGtkWidget = inChild != NULL ? inChild -> GetGtkWidget() : NULL;

	// get this gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// pointer check
	if (inGtkWidget == NULL || inChildGtkWidget == NULL) return;

	// some widgets have native scrolling attributes
	if (inChild -> ClassIs (__metaclass(CTreeView)) || inChild -> ClassIs (__metaclass(CTextView)))
	{
		::gtk_container_remove (GTK_CONTAINER(inGtkWidget), inChildGtkWidget);
	}
	// no native slots...
	else
	{
		::gtk_container_remove (GTK_CONTAINER(::gtk_bin_get_child(GTK_BIN(inGtkWidget))), inChildGtkWidget);
	}

	// generic call
	CContainer::PerformContainerRemove (inChild);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// policy writer
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::SetPolicies (const GtkPolicyType inHPolicy, const GtkPolicyType inVPolicy)
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the gtk widget pointer
	if (inGtkWidget == NULL) return;

	// policies affectation
	::gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(inGtkWidget), inHPolicy, inVPolicy);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// horizontal policy access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkPolicyType CScrollView::GetHPolicy () const
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the gtk widget pointer
	if (inGtkWidget == NULL) return GTK_POLICY_NEVER;

	// out policy
	GtkPolicyType outHPolicy=GTK_POLICY_NEVER, outVPolicy=GTK_POLICY_NEVER;

	// get the policies
	::gtk_scrolled_window_get_policy (GTK_SCROLLED_WINDOW(inGtkWidget), &outHPolicy, &outVPolicy);

	// ok
	return outHPolicy;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// vertical policy access
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkPolicyType CScrollView::GetVPolicy () const
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the gtk widget pointer
	if (inGtkWidget == NULL) return GTK_POLICY_NEVER;

	// out policy
	GtkPolicyType outHPolicy=GTK_POLICY_NEVER, outVPolicy=GTK_POLICY_NEVER;

	// get the policies
	::gtk_scrolled_window_get_policy (GTK_SCROLLED_WINDOW(inGtkWidget), &outHPolicy, &outVPolicy);

	// ok
	return outVPolicy;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// shadow type
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::SetShadowType (const GtkShadowType inGtkShadowType)
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the gtk widget pointer
	if (inGtkWidget == NULL) return

	// ok
	::gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(inGtkWidget), inGtkShadowType);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// shadow type
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkShadowType CScrollView::GetShadowType () const
{
	// get the gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the gtk widget pointer
	if (inGtkWidget == NULL) return GTK_SHADOW_NONE;

	// ok
	::gtk_scrolled_window_get_shadow_type (GTK_SCROLLED_WINDOW(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CScrollView::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CContainer::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// allocate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_SCROLLVIEW_ELEMENT);

			// modify the io xml node so that the potential overloaded serialization definition continues under this one
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// get the widget policies
			GtkPolicyType inHPolicy = GetHPolicy(); GtkPolicyType inVPolicy = GetVPolicy();

			// set the corresponding strings
			CString outHPolicy (inHPolicy==GTK_POLICY_NEVER ? "never" : inHPolicy==GTK_POLICY_ALWAYS ? "always" : "automatic");
			CString outVPolicy (inVPolicy==GTK_POLICY_NEVER ? "never" : inVPolicy==GTK_POLICY_ALWAYS ? "always" : "automatic");

			// set the element attributes
			newXMLElement -> AddAttribute (XML_SCROLLVIEW_ATTR_HPOLICY, outHPolicy);
			newXMLElement -> AddAttribute (XML_SCROLLVIEW_ATTR_VPOLICY, outVPolicy);
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_SCROLLVIEW_ELEMENT);

			// check we got an expected cscrollview node
			if (inXMLNode == NULL)
				throw new CException (CString("CScrollView::Serialize, specified xml node is not a \"") + 
							      XML_SCROLLVIEW_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cwidget node
			ioXMLElementNode = inXMLNode;

			// get the scroll view policies
			CString inHPolicy (::xml_node_get_attribute (inXMLNode, XML_SCROLLVIEW_ATTR_HPOLICY).GetValue());
			CString inVPolicy (::xml_node_get_attribute (inXMLNode, XML_SCROLLVIEW_ATTR_VPOLICY).GetValue());

			// set the scroll view policies
			SetPolicies (inHPolicy == CString("never")  ? GTK_POLICY_NEVER  : 
				     inHPolicy == CString("always") ? GTK_POLICY_ALWAYS : GTK_POLICY_AUTOMATIC,
				     inVPolicy == CString("never")  ? GTK_POLICY_NEVER  : 
				     inVPolicy == CString("always") ? GTK_POLICY_ALWAYS : GTK_POLICY_AUTOMATIC);
		}
		break;
	}
}




