/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: LazyCommand.java,v 1.4 2005/08/11 03:05:57 pietschy Exp $
 */

package org.pietschy.command;

/**
 * The LazyCommand defers the construction of the command until the first execution.
 * Subclasses must implement the {@link #build} and {@link #lazyExecute} methods.
 */
public abstract class
LazyCommand
extends ActionCommand
{
   private boolean built = false;

   public LazyCommand()
   {
   }

   public LazyCommand(String commandId)
   {
      super(commandId);
   }

   public LazyCommand(CommandManager commandManager, String commandId)
   {
      super(commandManager, commandId);
   }

   /**
    * This implementation will call {@link #build} on the first execution and delegate
    * the handling of the request to {@link #lazyExecute}.
    */
   public void
   handleExecute()
   {
      beforeExecute();

      if (!built)
      {
         build();
         built = true;
      }

      lazyExecute();

      afterExecute();
   }

   /**
    * Invoked before the command is executed.
    *
    * @deprecated Use {@link ActionCommandInterceptor} instead.
    */
   protected void
   beforeExecute()
   {
   }

   /**
    * Invoked after the command is executed.
    *
    * @deprecated Use {@link ActionCommandInterceptor} instead.
    */
   protected void
   afterExecute()
   {
   }

   /**
    * Invoked on the first time the command is executed.
    */
   public abstract void build();

   /**
    * Invoked after build and then again whenever the command is executed. 
    */
   public abstract void lazyExecute();

}
