/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <zlib.h>

#include <libinstrudeo/isddatafile.h>
#include <libinstrudeo/isdtextballoon.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdrectangle.h>
#include <libinstrudeo/isdvideoproperties.h>
#include <libinstrudeo/isdvideocanvas.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while accessing video canvas: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----CONSTRUCTORS-----
ISDVideoCanvas::ISDVideoCanvas(ISDVideoProperties* videoProperties)
    : ISDGLWidget(),
      videoProperties(videoProperties),
      vertMirrorCanvas(false)
{
    int bufferSize = videoProperties->getWidth()*videoProperties->getHeight()*videoProperties->getBytesPerPixel();
    imageBuffer = (char*)malloc(bufferSize);
    decodeBuffer = (char*)malloc(bufferSize);
}

//-----DESTRUCTORS-----
ISDVideoCanvas::~ISDVideoCanvas()
{
    //buffers needs to be freed, not deleted
    if (imageBuffer){
	free(imageBuffer);
	imageBuffer = NULL;
    }
    if (decodeBuffer){
	free(decodeBuffer);
	decodeBuffer = NULL;
    }

    //Note: the video properties are deleted in the recording class.
}

//-----PUBLIC METHODS-----
ISDObject::ISDErrorCode ISDVideoCanvas::initGL()
{
    glRasterPos2f(0.0, 0.0);
    
    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDVideoCanvas::reInitGL()
{
    initGL();

    reInitChildren();

    RETURN_SUCCESS;   
}
void ISDVideoCanvas::display()
{   
    /*
     * TODO:
     * ?? check for endianness and convert if necessary ??
     * We assume the recording was made in the same endianness as this routine,
     * this isn't always true: check the pixelFormat.bigEndian flag
     */
    
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();
    
    glDrawPixels(videoProperties->getWidth(), videoProperties->getHeight(),
		 ISD_GL_PIXEL_FORMAT, ISD_GL_PIXEL_TYPE, imageBuffer);

    //draw the children on top of the video "background"
    drawChildren();

    glFinish();
}
void ISDVideoCanvas::updateScales(float newScale)
{
    //zoom the raster-drawing
    glPixelZoom(newScale, newScale);

    /*
     * Zoom all the commentboxes
     * This is potentially dangerous since the API doesn't force that
     * all children are commentboxes, so be careful.
     */
    set<ISDGLWidget*>::const_iterator iter;
    for (iter=children.begin();iter!=children.end();iter++) {
	ISDCommentbox* c = dynamic_cast<ISDCommentbox*>(*iter);

	//Note: don't treat this as a failure, just continue with a warning.
	if (!c) {
	    LOG_WARNING("Encountered a child of the canvas that wasn't a commentbox.");
	    continue;
	}
	
	c->recalcScales();
    }
}
ISDObject::ISDErrorCode ISDVideoCanvas::rectangleUpdate(const ISDRectangle* rect, char* data)
{   
    //check the bounds
    if (rect->x + rect->width > videoProperties->getWidth() || 
	rect->y + rect->height > videoProperties->getHeight())
    {
	//this is pretty serious, could be a buffer-overflow attack
	LOG_CRITICAL("Tried to update the image with a rectangle that was larger then the image.");
	RETURN_ERROR(ISD_FORMAT_ERROR);
    }

    //position the bufferpointer and calc some useful values
    int imgWidthInBytes = videoProperties->getWidth() * videoProperties->getBytesPerPixel();
    int rectWidthInBytes = rect->width * videoProperties->getBytesPerPixel();
    
    //we need to Y-mirror the data because of the OpenGL coordinates vs. (recorded) screen coordinates
    //int imageBufferOffset = (rect->y * imgWidthInBytes) + (rect->x * videoProperties->getBytesPerPixel());
    int imageBufferOffset = (((videoProperties->getHeight()-rect->y-1) * imgWidthInBytes) + 
			     (rect->x * videoProperties->getBytesPerPixel()));
    
    char* imgPtr = imageBuffer + imageBufferOffset;

    char* bufPtr;
    int h, error;
    uLongf outputLength;
    switch (rect->encType) {
	case (ISDDataFile::ISD_RECT_ENCODING_RAW):
	    //just write scanlines to the image, no decompression
	    bufPtr = data;
	    for (h=0; h<rect->height; h++) {
		memcpy(imgPtr, bufPtr, rectWidthInBytes);
		bufPtr += rectWidthInBytes;
		imgPtr += imgWidthInBytes;
	    }

	    break;
	case (ISDDataFile::ISD_RECT_ENCODING_RLE):
    	    /*
	     * Note: we can't decode the data directly to the image buffer,
	     * because the rect isn't aligned to the image width, the image gets corrupted
	     *
	     * This was updated (in version 0.1.3) to use the zlib's (utility) uncompress function instead of 
	     * the intermediate functions in the rle.c file (which was deleted in this release).
	     * Upon entry, outputLength must contain the size of the decode buffer.
	     */
	    outputLength = rectWidthInBytes*rect->height;
	    error = uncompress((Bytef*)decodeBuffer, (uLongf*)&outputLength,
			       (Bytef*)data, rect->dataLen);
	    
	    if (error != Z_OK || outputLength!=rectWidthInBytes*rect->height) {
		LOG_CRITICAL("Error while uncompressing RLE rectangle.");
		RETURN_ERROR(ISD_ENCODING_ERROR);
	    }
	    
	    //copy the decoded pixels to the image
	    bufPtr = decodeBuffer;
	    for (h=0; h<rect->height; h++) {
		memcpy(imgPtr, bufPtr, rectWidthInBytes);
		bufPtr += rectWidthInBytes;
		imgPtr -= imgWidthInBytes;
	    }
	    	    	    
	    break;
	default:
	    LOG_WARNING("Trying to encode a rectangle with an unknown encoding type, skipping.");
	    RETURN_ERROR(ISD_ENCODING_ERROR);
	    break;
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDVideoCanvas::getCurrentFrame(char* pixelData, ISDPixelOrder order)
{
    if (pixelData==NULL) {
	LOG_WARNING("Requested a frame-read with uninitialized arguments.");
	RETURN_ERROR(ISD_ENCODING_ERROR);
    }

    //force a frame-update before reading it
    display();
    
    /*
     * The image must be flipped so the origin is back at the upper left corner.
     * To achieve this, we read in the scanlines in reversed order (bottom-up)
     */
    char* bufPtr = pixelData;
    for (int row=videoProperties->getHeight()-1;row>=0;row--) {
	switch (order) {
	case ISD_PIXEL_ARGB:
	    glPixelStorei(GL_UNPACK_SWAP_BYTES, 1);
	    glReadPixels(0, row, videoProperties->getWidth(), 1, 
			 GL_BGRA, GL_UNSIGNED_BYTE, bufPtr);
	    glPixelStorei(GL_UNPACK_SWAP_BYTES, 0);
	    break;
	case ISD_PIXEL_RGBA:
	    glReadPixels(0, row, videoProperties->getWidth(), 1, 
			 GL_RGBA, GL_UNSIGNED_BYTE, bufPtr);
	    break;
	}

	bufPtr += videoProperties->getWidth()*videoProperties->getBytesPerPixel();
    }
    
    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDVideoCanvas::getPixelData(ISDRectangle* rect, char* pixelData,
						     char* offScreenBuffer, ISDPixelOrder order)
{
    if (rect==NULL || pixelData==NULL) {
	LOG_WARNING("Requested a rectangle-read with uninitialized arguments.");
	RETURN_ERROR(ISD_ENCODING_ERROR);
    }

    //force a frame-update before reading it
    display();
    
    /*
     * The image must be flipped so the origin is back at the upper left corner.
     * To achieve this, we read in the scanlines in reversed order (bottom-up)
     */
    char* bufPtr = pixelData;
    int rectWidthInBytes = rect->width * videoProperties->getBytesPerPixel();

    //if we are using a off screen rendering, just copy the right pixels
    if (offScreenBuffer!=NULL) {
	int imgWidthInBytes = videoProperties->getWidth() * videoProperties->getBytesPerPixel();
	int imageBufferOffset = (((videoProperties->getHeight()-rect->y-1) * imgWidthInBytes) + 
				 (rect->x * videoProperties->getBytesPerPixel()));
	char* imgPtr = offScreenBuffer + imageBufferOffset;
	
	//copy the pixels from the offscreen to the target-buffer
	for (int h=0; h<rect->height; h++) {
	    memcpy(imgPtr, bufPtr, rectWidthInBytes);
	    bufPtr += rectWidthInBytes;
	    imgPtr -= imgWidthInBytes;
	}
    }
    else {
	for (int row=1;row<=rect->height;row++) {
	    switch (order) {
	    case ISD_PIXEL_ARGB:
		glPixelStorei(GL_UNPACK_SWAP_BYTES, 1);
		glReadPixels(rect->x, getHeight()-(rect->y+row), rect->width, 1, 
			     GL_BGRA, GL_UNSIGNED_BYTE, bufPtr);
		glPixelStorei(GL_UNPACK_SWAP_BYTES, 0);
		break;
	    case ISD_PIXEL_RGBA:
		glReadPixels(rect->x, getHeight()-(rect->y+row), rect->width, 1, 
			     GL_RGBA, GL_UNSIGNED_BYTE, bufPtr);
		break;
	    }
	    
	    bufPtr += rectWidthInBytes;
	}
    }
}
int ISDVideoCanvas::getWidth()
{
    return videoProperties->getWidth();
}
int ISDVideoCanvas::getHeight()
{
    return videoProperties->getHeight();
}

//-----PROTECTED METHODS-----

