/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <libinstrudeo/isdrecording.h>
#include <libinstrudeo/isdprogresscallback.h>
#include <libinstrudeo/isdwsftptransmitter.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdwsplatform.h>
#include <libinstrudeo/isdwscategory.h>
#include <libinstrudeo/isdwscommunicator.h>

/*
 * Remember to include this file in the implementation file (here, .cpp),
 * otherwise it is included more than once and variable-name clashes occur.
 */
#include <libinstrudeo/webservice/InstrudeoServiceBinding.nsmap>

#undef LOG_HEADER
#define LOG_HEADER "Error inside Instrudeo WebService communicator: \n"
#include <libinstrudeo/isdloggermacros.h>

ISDProgressCallback* ISDWSCommunicator::callbackClass = NULL;

//-----CONSTRUCTORS-----
ISDWSCommunicator::ISDWSCommunicator(string& serviceEndpoint)
    : ISDObject(),
      transmitFileSize(0)
{
    if (serviceEndpoint=="") {
	serviceEndpoint = DEFAULT_SERVICE_ENDPOINT;
    }
    wsStub = new InstrudeoServiceBinding(serviceEndpoint);
}

//-----DESTRUCTOR-----
ISDWSCommunicator::~ISDWSCommunicator()
{
    if (wsStub!=NULL) {
	delete wsStub;
	wsStub = NULL;
    }
}

//-----PUBLIC METHODS-----
ISDObject::ISDErrorCode ISDWSCommunicator::getSubPlatforms(ISDWSPlatform* cat, list<ISDWSPlatform*>& returnList, string& errorString)
{
    if (cat==NULL) {
	//LOG_WARNING("Trying to fetch the subplatforms of a NULL platform");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    //create webservice stubs
    _isdxsd__availablePlatformsRequest request;
    _isdxsd__availablePlatformsResponse response;
    
    //empty the list if necessary
    returnList.clear();
    
    //init request
    isdxsd__Platform requestPlatform;
    requestPlatform.id = cat->getId();
    requestPlatform.title = cat->getTitle();
    request.majorVersion = ISDRecording::MAJOR_VERSION_NUMBER;
    request.minorVersion = ISDRecording::MINOR_VERSION_NUMBER;
    request.platform = &requestPlatform;
    
    //retrieve array from the webservice
    if (wsStub->__isdws__getAvailablePlatforms(&request, &response) == SOAP_OK) {

	/*
	 * Check if the webservice returned an error.
	 * A value different from 0 means: error occurred
	 */
	if (response.errorCode) {
	    errorString = response.errorString;
	    RETURN_ERROR(ISD_WS_RETURNED_ERROR);
	}
	//all ok
	else{
	    std::vector<isdxsd__Platform*> vector = response.platformArray->platform;
	    std::vector<isdxsd__Platform*>::const_iterator iter;
	
	    //copy all returned platforms to the return-list
	    for (iter=vector.begin(); iter!=vector.end(); ++iter){
		isdxsd__Platform* fetchedCat = *iter;
		Glib::ustring title(fetchedCat->title);
		returnList.push_back(new ISDWSPlatform(fetchedCat->id,
						       title));
	    }
	}
    }
    else{
	//LOG_WARNING("Error while contacting the Instrudeo WebService.");
	//debug	soap_print_fault(wsStub->soap, stderr);
	RETURN_ERROR(ISD_WS_CONTACT_ERROR);
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDWSCommunicator::getSubCategories(ISDWSCategory* cat, list<ISDWSCategory*>& returnList, string& errorString)
{
    if (cat==NULL) {
	//LOG_WARNING("Trying to fetch the subcategories of a NULL category");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    //create webservice stubs
    _isdxsd__availableCategoriesRequest request;
    _isdxsd__availableCategoriesResponse response;
    
    //empty the list if necessary
    returnList.clear();
    
    //init request
    isdxsd__Category requestCategory;
    requestCategory.id = cat->getId();
    requestCategory.title = cat->getTitle();
    request.majorVersion = ISDRecording::MAJOR_VERSION_NUMBER;
    request.minorVersion = ISDRecording::MINOR_VERSION_NUMBER;
    request.category = &requestCategory;
    
    //retrieve array from the webservice
    if (wsStub->__isdws__getAvailableCategories(&request, &response) == SOAP_OK) {

	/*
	 * Check if the webservice returned an error.
	 * A value different from 0 means: error occurred
	 */
	if (response.errorCode) {
	    errorString = response.errorString;
	    RETURN_ERROR(ISD_WS_RETURNED_ERROR);
	}
	//all ok
	else{
	    std::vector<isdxsd__Category*> vector = response.categoryArray->category;
	    std::vector<isdxsd__Category*>::const_iterator iter;
	
	    //copy all returned categories to the return-list
	    for (iter=vector.begin(); iter!=vector.end(); ++iter){
		isdxsd__Category* fetchedCat = *iter;
		Glib::ustring title(fetchedCat->title);
		returnList.push_back(new ISDWSCategory(fetchedCat->id,
						       title));
	    }
	}
    }
    else{
	//LOG_WARNING("Error while contacting the Instrudeo WebService.");
	//debug	soap_print_fault(wsStub->soap, stderr);
	RETURN_ERROR(ISD_WS_CONTACT_ERROR);
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDWSCommunicator::getFtpPath(string& username, string& password,
						      string& ftpPath, int& sessionId,
						      string& errorString)
{
    //create webservice stubs
    _isdxsd__newFtpPathRequest request;
    _isdxsd__newFtpPathResponse response;

    //init request
    request.majorVersion = ISDRecording::MAJOR_VERSION_NUMBER;
    request.minorVersion = ISDRecording::MINOR_VERSION_NUMBER;
    request.username = username;
    request.password = password;

    //retrieve array from the webservice
    if (wsStub->__isdws__getNewFtpPath(&request, &response) == SOAP_OK) {
	/*
	 * Check if the webservice returned an error.
	 * A value different from 0 means: error occurred
	 */
	if (response.errorCode) {
	    errorString = response.errorString;
	    RETURN_ERROR(ISD_WS_RETURNED_ERROR);
	}
	//all ok
	else{
	    sessionId = response.id;
	    ftpPath = response.ftpPath;
	}
    }
    else{
	//LOG_WARNING("Error while contacting the Instrudeo WebService.");
	//debug	
	soap_print_fault(wsStub->soap, stderr);
	RETURN_ERROR(ISD_WS_CONTACT_ERROR);
    }
    
    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDWSCommunicator::transmitRecording(string& recordingFileName, string& ftpPath,
							     string& username, string& password,
							     ISDProgressCallback* callbackClass)
{
    ISDObject::ISDErrorCode retVal;
    //split the ftpPath
    string host, dir, file;
    int port;
    if (splitFtpPathString(ftpPath, host, port, dir, file) != ISD_SUCCESS) {
	LOG_WARNING("Error while parsing the FTP path while transmitting a recording.");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    ISDWSFtpTransmitter transmitter;

    //set the callback class
    if (callbackClass!=NULL) {
	ISDWSCommunicator::callbackClass = callbackClass;
	transmitter.setSentBytesCallback(this, ISDWSCommunicator::ftpCallback);
	//get the file size, needed to calc procent
	transmitFileSize = ISDUtils::getInstance()->getFileSizeInBytes(recordingFileName);
    }
    
    retVal = transmitter.sendRecording(recordingFileName, host, port,
				       dir, file,
				       username, password);

    if (ISDWSCommunicator::callbackClass!=NULL && 
	ISDWSCommunicator::callbackClass->wasCancelled()) 
	{
	    //this is not really an error, just set the return value
	    retVal = ISD_FTP_CANCELLED_ERROR;
	}

    //upload exited ok
    if (retVal==ISD_SUCCESS) {
	RETURN_SUCCESS;
    }
    //upload exited with error or cancelled
    else if (retVal==ISD_FTP_CANCELLED_ERROR) {
	return ISD_FTP_CANCELLED_ERROR;
    }
    else {
	RETURN_ERROR(retVal);
    }
}
ISDObject::ISDErrorCode ISDWSCommunicator::setUploadDone(int sessionId, bool aborted, string& errorString)
{
    //create webservice stubs
    _isdxsd__ftpUploadDoneRequest request;
    _isdxsd__ftpUploadDoneResponse response;

    //init request
    request.majorVersion = ISDRecording::MAJOR_VERSION_NUMBER;
    request.minorVersion = ISDRecording::MINOR_VERSION_NUMBER;
    request.id = sessionId;
    request.error = aborted;

    //retrieve array from the webservice
    if (wsStub->__isdws__setFtpUploadDone(&request, &response) == SOAP_OK) {
	/*
	 * Check if the webservice returned an error.
	 * A value different from 0 means: error occurred
	 */
	if (response.errorCode) {
	    errorString = response.errorString;
	    RETURN_ERROR(ISD_WS_RETURNED_ERROR);
	}
	//all ok
	else{
	    RETURN_SUCCESS;
	}
    }
    else{
	//LOG_WARNING("Error while contacting the Instrudeo WebService.");
	//debug	soap_print_fault(wsStub->soap, stderr);
	RETURN_ERROR(ISD_WS_CONTACT_ERROR);
    }

    RETURN_SUCCESS;
}

//-----PROTECTED STATIC METHODS-----
int ISDWSCommunicator::ftpCallback(void* callbackClass, double xfered, double total)
{
    /* 
     * This code comes from the libftp++ documentation.
     * It casts the pointers to the correct type and calls the class method
     */
    ((ISDWSCommunicator*)callbackClass)->handleCallback(xfered);

    //if this method returns 1, the transmission is aborted.
    if (ISDWSCommunicator::callbackClass &&
	ISDWSCommunicator::callbackClass->wasCancelled()) 
	{
	    return 1;
	}
    else {
	return 0;
    }
}

//-----PROTECTED METHODS-----
void ISDWSCommunicator::handleCallback(double xfered)
{
    //notify the callback class if it is set
    if (ISDWSCommunicator::callbackClass!=NULL) {
	ISDWSCommunicator::callbackClass->procentDone((float)xfered/(float)transmitFileSize*100.0);
    }
}
ISDObject::ISDErrorCode ISDWSCommunicator::splitFtpPathString(string& ftpPath, string& host, int& port, 
							      string& dir, string& file)
{
    //work on a local copy
    string path = ftpPath;

    //first, check if the path starts with "ftp://" and delete it if necessary
    if (path.substr(0, 6)=="ftp://") {
	path = path.substr(6);
    }

    //extract the server part, possibly with the port, separated with a colon
    int firstSlash = path.find_first_of('/');
    string hostAndPort = path.substr(0, firstSlash);

    //if a port number is present, extract it, otherwise, use default 21
    int colonPos = hostAndPort.find_first_of(':');
    if (colonPos!=hostAndPort.npos) {
	host = hostAndPort.substr(0, colonPos);
	string portStr = hostAndPort.substr(colonPos+1);
	port = ISDUtils::getInstance()->stringToInt(portStr);
    }
    else {
	host = hostAndPort;
	port = DEFAULT_FTP_PORT;
    }

    //extract the directory and filename, omitting the first slash of the directory and including the last
    string dirAndFile = path.substr(firstSlash+1);
    int lastSlash = dirAndFile.find_last_of('/');
    dir = dirAndFile.substr(0, lastSlash+1);
    file = dirAndFile.substr(lastSlash+1);
    
    
    RETURN_SUCCESS;
}
