/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <list>

#include <libinstrudeo/isdwsplatform.h>
#include <libinstrudeo/isdwscategory.h>
#include <libinstrudeo/isdrecording.h>
#include <libinstrudeo/isdvideocanvas.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdcommentbox.h>
#include <libinstrudeo/isdxmlfile.h>

//-----CONSTRUCTORS-----
ISDXmlFile::ISDXmlFile()
    : ISDObject()
{
    lastError = initNewDocument();
}
ISDXmlFile::ISDXmlFile(string fileName, ISDVideoCanvas* videoCanvas)
    : ISDObject()
{
    try {
	if (videoCanvas==NULL){
	    LOG_CRITICAL("Can't initialize xml file with a NULL videoCanvas pointer.");
	    lastError = ISD_ARGS_ERROR;
	    return;
	}

	xmlpp::DomParser* parser = new xmlpp::DomParser();
	parser->set_substitute_entities(); //We just want the text to be resolved/unescaped automatically.
	parser->parse_file(fileName);
	xmlpp::Document* xmlDocument = parser->get_document();
	if (!xmlDocument){
	    LOG_CRITICAL("Couldn't get the DOM document from the XML parser");
	    lastError = ISD_FILE_ERROR;
	    return;
	}
	
	xmlpp::Element* rootNode = xmlDocument->get_root_node();
	if (!rootNode){
	    LOG_CRITICAL("Couldn't get the root node from the XML parser.");
	    lastError = ISD_FILE_ERROR;
	    return;
	}

	/*
	 * initialise the meta info object
	 */
	Glib::ustring tempStr;
	tempStr = rootNode->get_attribute(ROOT_VERSION_NAME)->get_value();
	metaInfo.setVersionNumberString(tempStr);

	tempStr = getStringValue(xmlDocument, TITLE_NAME);
	metaInfo.setTitle(tempStr);

	tempStr = getStringValue(xmlDocument, QUESTION_NAME);
	metaInfo.setQuestion(tempStr);

	tempStr = getStringValue(xmlDocument, DESCRIPTION_NAME);
	metaInfo.setDescription(tempStr);

	/*
	 * initialize the platforms
	 */
	if (readPlatformList(xmlDocument, metaInfo) != ISD_SUCCESS) {
	    LOG_WARNING("Error while reading the platforms from the XML file.");
	}

	/*
	 * initialize the categories
	 */
	if (readCategoryList(xmlDocument, metaInfo) != ISD_SUCCESS) {
	    LOG_WARNING("Error while reading the categories from the XML file.");
	}
	
	/*
	 * initialize the comment-set
	 */
	xmlpp::NodeSet results = getAllComments(xmlDocument);
	xmlpp::NodeSet::const_iterator iter;
	for (iter=results.begin(); iter!=results.end(); iter++) {
	    xmlpp::Element* elem = dynamic_cast<xmlpp::Element*>(*iter);
	    ISDCommentbox* commentbox;

	    //find the language
	    xmlpp::Element* langElem = getCommentLanguage(xmlDocument, elem);
	    if (langElem==NULL) {
		LOG_CRITICAL("Error while fetching the language of a comment-object in xmlFile constructor;");
		lastError = ISD_INIT_ERROR;
		break;
	    }
	    Glib::ustring lang = langElem->get_attribute(COMMENTLIST_LANGUAGE_NAME)->get_value();

	    if (fillCommentFromElement(videoCanvas, elem, commentbox, lang) != ISD_SUCCESS) {
		LOG_CRITICAL("Error while initialising the comment-object-set in xmlFile constructor.");
		lastError = ISD_INIT_ERROR;
		break;
	    }
	    else {
		allComments.push_back(commentbox);
	    }
	}
    }
    catch(const std::exception& ex) {
	LOG_CRITICAL("Couldn't get the root node from the XML parser." + fileName + "\n" + ex.what());
	lastError = ISD_FILE_ERROR;
    }
}

//-----DESTRUCTOR-----
ISDXmlFile::~ISDXmlFile()
{
}

//-----PUBLIC STATIC METHODS-----
Glib::ustring ISDXmlFile::getVersionNumberString()
{
    Glib::ustring majorStr = ISDUtils::getInstance()->intToString(ISDRecording::MAJOR_VERSION_NUMBER);
    Glib::ustring minorStr = ISDUtils::getInstance()->intToString(ISDRecording::MINOR_VERSION_NUMBER);
    Glib::ustring microStr = ISDUtils::getInstance()->intToString(ISDRecording::MICRO_VERSION_NUMBER);
    Glib::ustring nanoStr = ISDUtils::getInstance()->intToString(ISDRecording::NANO_VERSION_NUMBER);
    Glib::ustring retVal = majorStr+"."+minorStr+"."+microStr+"."+nanoStr;
    return retVal;
}

//-----PUBLIC METHODS-----
ISDObject::ISDErrorCode ISDXmlFile::writeToFile(string& outputFileName, bool createNew)
{
    ISDObject::ISDErrorCode retVal;
    
    if (createNew){
	if ((retVal = ISDUtils::getInstance()->createTempFile(outputFileName)) != ISD_SUCCESS) {
	    RETURN_ERROR(ISD_FILE_ERROR);
	}
    }

    //create a standard, version 1.0 DOM document, with default root node and version number
    xmlpp::Document* xmlDocument = new xmlpp::Document();
    if (!xmlDocument){
	LOG_CRITICAL("Couldn't create a DOM document while saving the XML file.");
	RETURN_ERROR(ISD_FILE_ERROR);
    }

    //init root node and version number
    xmlpp::Element* rootNode = xmlDocument->create_root_node(ROOT_NAME);
    if (!rootNode){
	delete xmlDocument;
	LOG_CRITICAL("Couldn't create root node in new XML meta file.");
	RETURN_ERROR(ISD_FILE_ERROR);
    }
    rootNode->set_attribute(ROOT_VERSION_NAME, metaInfo.getVersionNumberString());

    //save meta info
    rootNode->add_child(TITLE_NAME)->set_child_text(metaInfo.getTitle());
    rootNode->add_child(QUESTION_NAME)->set_child_text(metaInfo.getQuestion());
    rootNode->add_child(DESCRIPTION_NAME)->set_child_text(metaInfo.getDescription());

    //save the platforms
    list<ISDWSPlatform*> allPlats = metaInfo.getAllPlatforms();
    if (allPlats.size()>0) {
	//create the commentlist
	list<ISDWSPlatform*>::const_iterator iter = allPlats.begin();
	xmlpp::Element* platNode = rootNode->add_child(PLATFORM_NAME);
	platNode->set_attribute(PLATFORM_ID_NAME, ISDUtils::getInstance()->intToString((*iter)->getId()));
	platNode->set_attribute(PLATFORM_TITLE_NAME, (*iter)->getTitle());
	iter++;
	for (; iter!=allPlats.end(); iter++) {
	    platNode = platNode->add_child(PLATFORM_NAME);
	    platNode->set_attribute(PLATFORM_ID_NAME, ISDUtils::getInstance()->intToString((*iter)->getId()));
	    platNode->set_attribute(PLATFORM_TITLE_NAME, (*iter)->getTitle());
	}
    }

    //save the categories
    list<ISDWSCategory*> allCats = metaInfo.getAllCategories();
    if (allCats.size()>0) {
	//create the commentlist
	list<ISDWSCategory*>::const_iterator iter = allCats.begin();
	xmlpp::Element* catNode = rootNode->add_child(CATEGORY_NAME);
	catNode->set_attribute(CATEGORY_ID_NAME, ISDUtils::getInstance()->intToString((*iter)->getId()));
	catNode->set_attribute(CATEGORY_TITLE_NAME, (*iter)->getTitle());
	iter++;
	for (; iter!=allCats.end(); iter++) {
	    catNode = catNode->add_child(CATEGORY_NAME);
	    catNode->set_attribute(CATEGORY_ID_NAME, ISDUtils::getInstance()->intToString((*iter)->getId()));
	    catNode->set_attribute(CATEGORY_TITLE_NAME, (*iter)->getTitle());
	}
    }

    //save the comments
    list<ISDCommentbox*>::const_iterator iter;
    for (iter=allComments.begin();iter!=allComments.end();iter++) {
	saveComment(xmlDocument, NULL, *iter);
    }

    //here, we can assume the filename has been set correctly
    xmlDocument->write_to_file(outputFileName);

    delete xmlDocument;

    RETURN_SUCCESS;
}

//-----XML-FILE GETTER/SETTER METHODS-----
ISDRecordingMetaInfo* ISDXmlFile::getMetaInfo()
{
    return &metaInfo;
}
ISDObject::ISDErrorCode ISDXmlFile::addComment(ISDCommentbox* commentbox)
{
    //insert the commentbox in the commentbox-set
    allComments.push_back(commentbox);
	
    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDXmlFile::removeComment(ISDCommentbox*& commentbox)
{
    //remove the commentbox in from commentbox-set
    allComments.remove(commentbox);

    delete commentbox;
    commentbox = NULL;

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDXmlFile::updateCommentsForPosition(ISDVideoCanvas* videoCanvas, int position, Glib::ustring* lang, bool& noChange)
{
    if (videoCanvas==NULL) {
	LOG_WARNING("Tried to update comments at a position with a NULL videoCanvas");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    //optimistic initialisation
    noChange = true;

    //these lists keep track of changes per update
    addedInLastUpdate.clear();
    removedInLastUpdate.clear();

    /*
     * Iterate over all comments, and if the requested position lies inside
     * the lifetime of the comment, if the comment has the specified language,
     * and it isn't already a child of the videoCanvas,
     * add it as a child of the videoCanvas.
     *
     * If we encounter a commentbox that is already the child of the videoCanvas,
     * but the position doesn't lie inside the lifetime of the commentbox anymore,
     * remove it from the child-list of the videoCanvas.
     */
    list<ISDCommentbox*>::const_iterator iter;
    for (iter=allComments.begin();iter!=allComments.end();iter++) {
	ISDCommentbox* comment = (*iter);

	//language check
	//Note: if NULL is specified for the language, catchall-comments will be disabled too,
	//      if you want to only enable catchall's, pass ISD_COMMENTBOX_LANGUAGE_CATCHALL as the lang.
	if (lang!=NULL) {
	    //Note: include a check for the special catchall-language.
	    if ((*lang)==comment->getLanguage() || comment->getLanguage()==ISD_COMMENTBOX_LANGUAGE_CATCHALL) {
		int startTime = comment->getStartTime();
		int endTime = startTime+comment->getDuration();
		//child check
		if (videoCanvas->hasChild(comment)) {
		    //outside lifetime check
		    if (position < startTime || position > endTime) {
			videoCanvas->removeChild(comment);
			removedInLastUpdate.push_back(comment);
			noChange = false;
		    } 
		}
		//no child, inside lifetime check
		else if(position >= startTime && position <= endTime) {
		    videoCanvas->addChild(comment);
		    addedInLastUpdate.push_back(comment);
		    noChange = false;
		}
	    }
	    //the language is different, remove the comment if showing
	    else {
		if (videoCanvas->hasChild(comment)) {
		    videoCanvas->removeChild(comment);
		    removedInLastUpdate.push_back(comment);
		    noChange = false;
		}
	    }
	}
	//comments are disabled, remove the comment if showing
	else {
	    if (videoCanvas->hasChild(comment)) {
		videoCanvas->removeChild(comment);
		removedInLastUpdate.push_back(comment);
		noChange = false;
	    }
	}
	
    }
    
    // if some change happened, force a redraw of the (OpenGL)canvas
    if (!noChange) {
	videoCanvas->display();
    }

    RETURN_SUCCESS;
}
list<ISDCommentbox*>& ISDXmlFile::getAllCommentObjects()
{
    return allComments;
}
ISDObject::ISDErrorCode ISDXmlFile::getCommentLanguages(set<Glib::ustring>& langList)
{
    list<ISDCommentbox*>::const_iterator iter;
    //iterate over all commentboxes
    for (iter=allComments.begin();iter!=allComments.end();iter++) {
	ISDCommentbox* comment = (*iter);
	
	//if the language isn't in the list, add it as a distinct language
	//Note: don't add ISD_COMMENTBOX_LANGUAGE_CATCHALL
	if (langList.find(comment->getLanguage()) == langList.end() && 
	    comment->getLanguage() != ISD_COMMENTBOX_LANGUAGE_CATCHALL)
	    {
		langList.insert(comment->getLanguage());
	    }
    }

    RETURN_SUCCESS;
}
int ISDXmlFile::getUniqueCommentId()
{
    int uniqueId = -1;

    /*
     * we use the size of the comment-set as the start of the id-search,
     * and increment it for every comment we find with that id.
     * This isn't really efficient, but we assume the id's of the list are pretty
     * closely sorted so that little clashes will occur.
     */
    
    for (uniqueId = allComments.size();getComment(uniqueId)!=NULL;uniqueId++);
    
    //don't start at 0, but at 1
    if (uniqueId==0) {
	uniqueId = 1;
    }

    return uniqueId;
}
list<ISDCommentbox*>& ISDXmlFile::getAddedInLastUpdate()
{
    return addedInLastUpdate;
}
list<ISDCommentbox*>& ISDXmlFile::getRemovedInLastUpdate()
{
    return removedInLastUpdate;
}

//-----PRIVATE/PROTECTED METHODS-----
ISDObject::ISDErrorCode ISDXmlFile::initNewDocument()
{
    //init the meta info object
    Glib::ustring version = ISDXmlFile::getVersionNumberString();
    metaInfo.setVersionNumberString(version);

    RETURN_SUCCESS;
}
ISDCommentbox* ISDXmlFile::getComment(int id)
{
    ISDCommentbox* retVal = NULL;

    //TODO: this can be optimized
    list<ISDCommentbox*>::const_iterator iter;
    for (iter=allComments.begin();iter!=allComments.end();iter++) {
	if ((*iter)->getId()==id) {
	    retVal = *iter;
	    break;
	}
    }

    return retVal;
}

//-----PROTECTED XML ACCESS METHODS-----
ISDObject::ISDErrorCode ISDXmlFile::addNewLanguageList(xmlpp::Document* xmlDocument, Glib::ustring language)
{
    //do nothing if the commentlist already exists.
    if (searchLanguageList(xmlDocument, language) != NULL) {
	RETURN_SUCCESS;
    }
    
    xmlpp::Element* elem = addEmptyElement(xmlDocument, COMMENTLIST_NAME);
    //if null is returned, an error occurred
    if (elem) {
	elem->set_attribute(COMMENTLIST_LANGUAGE_NAME, language);
	RETURN_SUCCESS;
    }
    else {
	RETURN_ERROR(ISD_XML_ELEMENT_ERROR);
    }
    
}
Glib::ustring ISDXmlFile::getStringValue(xmlpp::Document* xmlDocument, const string elementName)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    Glib::ustring xpath;
    xpath = xpath + "/" + ROOT_NAME + "/" + elementName;
    
    xmlpp::NodeSet results = rootNode->find(xpath);
    //check if there are results
    if (results.size()!=0){
	//no duplicates allowed
	if (results.size()!=1){
	    LOG_WARNING("Tried to get value of duplicate nodes.");
	    lastError = ISD_XML_DUPLICATES_ERROR;
	}
	else{
	    xmlpp::Element* elem = dynamic_cast<xmlpp::Element*>(results[0]);
	    if (elem != NULL){
		if (elem->get_child_text()==NULL) {
		    //return the empty string if no value is present
		    return Glib::ustring();
		}
		else {
		    return elem->get_child_text()->get_content();
		}
	    }
	    else {
		LOG_WARNING("Error while getting content of node: "+elementName);
		lastError = ISD_XML_DUPLICATES_ERROR;
	    }
	}
    }
    
    //return empty string by default, if nothing was found
    return Glib::ustring();
}
ISDObject::ISDErrorCode ISDXmlFile::readPlatformList(xmlpp::Document* xmlDocument, ISDRecordingMetaInfo& metaInfo)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    xmlpp::Element* elem = NULL;
    Glib::ustring xpath;
    xpath = xpath + "/" + ROOT_NAME + "/" + PLATFORM_NAME;

    //search the first, top-level platform
    xmlpp::NodeSet results = rootNode->find(xpath);
    if (results.size()!=0) {
	//no duplicates allowed
	if (results.size()!=1){
	    LOG_WARNING("More than one top-level platform found.");
	    lastError = ISD_XML_DUPLICATES_ERROR;
	}
	else {
	    elem = dynamic_cast<xmlpp::Element*>(results[0]);
	    if (elem){
		ISDWSPlatform* plat;
		if (fillPlatformFromElement(elem, plat) != ISD_SUCCESS) {
		    LOG_WARNING("Error while converting XML element to platform-object");
		    RETURN_ERROR(ISD_XML_ELEMENT_ERROR); 
		}
		metaInfo.addPlatform(plat);

		//add all the children
		xmlpp::Node::NodeList childPlats = elem->get_children();
		xmlpp::Node::NodeList::const_iterator iter;
		for (iter=childPlats.begin(); iter!=childPlats.end(); ) {
		    ISDWSPlatform* plat;
		    elem = dynamic_cast<xmlpp::Element*>(*iter);
		    if (fillPlatformFromElement(elem, plat) != ISD_SUCCESS) {
			LOG_WARNING("Error while converting XML element to platform-object");
			RETURN_ERROR(ISD_XML_ELEMENT_ERROR); 
		    }
		    metaInfo.addPlatform(plat);
		    childPlats = elem->get_children();
		    iter=childPlats.begin();
		}
	    }
	    //no else-clause: it's possible there are no platforms
	}
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDXmlFile::readCategoryList(xmlpp::Document* xmlDocument, ISDRecordingMetaInfo& metaInfo)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    xmlpp::Element* elem = NULL;
    Glib::ustring xpath;
    xpath = xpath + "/" + ROOT_NAME + "/" + CATEGORY_NAME;

    //search the first, top-level category
    xmlpp::NodeSet results = rootNode->find(xpath);
    if (results.size()!=0) {
	//no duplicates allowed
	if (results.size()!=1){
	    LOG_WARNING("More than one top-level category found.");
	    lastError = ISD_XML_DUPLICATES_ERROR;
	}
	else {
	    elem = dynamic_cast<xmlpp::Element*>(results[0]);
	    if (elem){
		ISDWSCategory* cat;
		if (fillCategoryFromElement(elem, cat) != ISD_SUCCESS) {
		    LOG_WARNING("Error while converting XML element to category-object");
		    RETURN_ERROR(ISD_XML_ELEMENT_ERROR); 
		}
		metaInfo.addCategory(cat);

		//add all the children
		xmlpp::Node::NodeList childCats = elem->get_children();
		xmlpp::Node::NodeList::const_iterator iter;
		for (iter=childCats.begin(); iter!=childCats.end(); ) {
		    ISDWSCategory* cat;
		    elem = dynamic_cast<xmlpp::Element*>(*iter);
		    if (fillCategoryFromElement(elem, cat) != ISD_SUCCESS) {
			LOG_WARNING("Error while converting XML element to category-object");
			RETURN_ERROR(ISD_XML_ELEMENT_ERROR); 
		    }
		    metaInfo.addCategory(cat);
		    childCats = elem->get_children();
		    iter=childCats.begin();
		}
	    }
	    //no else-clause: it's possible there are no categories
	}
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDXmlFile::fillPlatformFromElement(xmlpp::Element* elem, ISDWSPlatform*& cat)
{
    cat = NULL;
    if (elem==NULL) {
	LOG_WARNING("Tried to fill a platform with a NULL comment-element.");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    ISDUtils* utils = ISDUtils::getInstance();
    string tempStr;

    tempStr = elem->get_attribute(PLATFORM_ID_NAME)->get_value();
    int id = utils->stringToInt(tempStr);
    
    Glib::ustring title = elem->get_attribute(PLATFORM_TITLE_NAME)->get_value();
    cat = new ISDWSPlatform(id, title);

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDXmlFile::fillCategoryFromElement(xmlpp::Element* elem, ISDWSCategory*& cat)
{
    cat = NULL;
    if (elem==NULL) {
	LOG_WARNING("Tried to fill a category with a NULL comment-element.");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }

    ISDUtils* utils = ISDUtils::getInstance();
    string tempStr;

    tempStr = elem->get_attribute(CATEGORY_ID_NAME)->get_value();
    int id = utils->stringToInt(tempStr);
    
    Glib::ustring title = elem->get_attribute(CATEGORY_TITLE_NAME)->get_value();
    cat = new ISDWSCategory(id, title);

    RETURN_SUCCESS;
}
xmlpp::Element* ISDXmlFile::setStringValue(xmlpp::Document* xmlDocument, const string elementName, Glib::ustring value)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    xmlpp::Element* elem = NULL;
    Glib::ustring xpath;
    xpath = xpath + "/" + ROOT_NAME + "/" + elementName;

    //check if the node already exists
    xmlpp::NodeSet results = rootNode->find(xpath);
    if (results.size()!=0) {
	//no duplicates allowed
	if (results.size()!=1){
	    LOG_WARNING("Tried to update value of duplicate nodes.");
	    lastError = ISD_XML_DUPLICATES_ERROR;
	}
	else {
	    elem = dynamic_cast<xmlpp::Element*>(results[0]);
	    if (elem){
		elem->set_child_text(value);
	    }
	}
    }
    //insert a new node
    else{
	elem = rootNode->add_child(elementName);
	if (!elem){
	    LOG_WARNING("Error ocurred while inserting a new node.");
	    lastError = ISD_XML_ELEMENT_ERROR;
	    return NULL;
	}
	elem->set_child_text(value);
    }

    return elem;
}
xmlpp::Element* ISDXmlFile::addEmptyElement(xmlpp::Document* xmlDocument, const string elementName)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    xmlpp::Element* elem = NULL;
    
    //Note: don't check if the node already exists, this is possible!
    elem = rootNode->add_child(elementName);
    if (!elem){
	LOG_WARNING("Error ocurred while inserting a new node.");
	lastError = ISD_XML_ELEMENT_ERROR;
	elem = NULL;
    }
    
    return elem;
}
xmlpp::Element* ISDXmlFile::searchLanguageList(xmlpp::Document* xmlDocument, Glib::ustring lang)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    xmlpp::Element* elem = NULL;
    Glib::ustring xpath;
    //format: ROOT_NAME/COMMENTLIST_NAME[@COMMENTLIST_LANGUAGE_NAME='lang']
    xpath = xpath + "/" + ROOT_NAME + "/"
	+ COMMENTLIST_NAME
	+ "[@" + COMMENTLIST_LANGUAGE_NAME + "='" + lang + "']";

    xmlpp::NodeSet results = rootNode->find(xpath);
    //check if the comment already exists
    if (results.size()!=0){
	//no duplicates allowed
	if (results.size()!=1){
	    LOG_WARNING("Search for a commentlist by language resulted in multiple elements.");
	    lastError = ISD_XML_DUPLICATES_ERROR;
	}
	else{
	    elem = dynamic_cast<xmlpp::Element*>(results[0]);
	}
    }

    return elem;
}
xmlpp::Element* ISDXmlFile::searchComment(xmlpp::Document* xmlDocument, int id, Glib::ustring lang)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    xmlpp::Element* elem = NULL;
    Glib::ustring idStr(ISDUtils::getInstance()->intToString(id));
    Glib::ustring xpath;
    //format: ROOT_NAME/COMMENTLIST_NAME[@COMMENTLIST_LANGUAGE_NAME='lang']/COMMENT_NAME[@COMMENT_ID_NAME='id']
    xpath = xpath + "/" + ROOT_NAME + "/"
	+ COMMENTLIST_NAME
	+ "[@" + COMMENTLIST_LANGUAGE_NAME + "='" + lang + "']"
	+ "/" + COMMENT_NAME
	+ "[@" + COMMENT_ID_NAME + "='" + idStr + "']";

    xmlpp::NodeSet results = rootNode->find(xpath);
    //check if the comment exists
    if (results.size()!=0){
	//no duplicates allowed
	if (results.size()!=1){
	    LOG_WARNING("Search for comment by id resulted in multiple elements.");
	    lastError = ISD_XML_DUPLICATES_ERROR;
	}
	else{
	    elem = dynamic_cast<xmlpp::Element*>(results[0]);
	}
    }

    return elem;
}
xmlpp::Element* ISDXmlFile::saveComment(xmlpp::Document* xmlDocument, xmlpp::Element* elem, ISDCommentbox* commentbox)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    
    /*
     * Add the language-list to the XML file, if it is already present, nothing happens.
     * TODO: this can be optimized, since this method can query the XML document two times
     *       and isn't needed very much.
     */
    addNewLanguageList(xmlDocument, commentbox->getLanguage());

    //update elem with the new values
    if (elem){
	fillElementFromComment(elem, commentbox);
	return elem;
    }
    //insert a new comment
    else {
	xmlpp::Element* commentList = NULL;
	xmlpp::Element* newComment = NULL;
	Glib::ustring xpath;
	xpath = xpath + "/" + ROOT_NAME + "/" + COMMENTLIST_NAME + "[@" + COMMENTLIST_LANGUAGE_NAME + "='" + commentbox->getLanguage() + "']";
	
	xmlpp::NodeSet results = rootNode->find(xpath);
	//fetch the commentlist-node with the specified language
	if (results.size()!=0){
	    //no duplicate commentlists allowed
	    if (results.size()!=1){
		LOG_WARNING("Encountered a duplicate commentlist while inserting a new comment.");
		lastError = ISD_XML_DUPLICATES_ERROR;
	    }
	    else{
		commentList = dynamic_cast<xmlpp::Element*>(results[0]);
		if (commentList) {
		    newComment = commentList->add_child(COMMENT_NAME);
		    if (newComment) {
			fillElementFromComment(newComment, commentbox);
		    }
		}
	    }
	}
	//commentlist was not found
	else {
	    LOG_WARNING("Tried to insert a comment in a nonexisting commentlist; this shouldn't happen.");
	    lastError = ISD_XML_MISSING_ERROR;
	}

	return newComment;
    }
}
ISDObject::ISDErrorCode ISDXmlFile::fillElementFromComment(xmlpp::Element* elem, ISDCommentbox* commentbox)
{
    //remove all attributes
    xmlpp::Element::AttributeList attrbs = elem->get_attributes();
    xmlpp::Element::AttributeList::const_iterator itera;
    for (itera=attrbs.begin(); itera!=attrbs.end(); ++itera){
	elem->remove_attribute((*itera)->get_name());
    }

    //remove the children
    xmlpp::Node::NodeList children = elem->get_children();
    xmlpp::Node::NodeList::const_iterator iterc;
    for (iterc=children.begin(); iterc!=children.end(); ++iterc){
	elem->remove_child(*iterc);
    }

    //---add the new values---
    int intVals[4];
    float floatVals[4];

    //id
    elem->set_attribute(COMMENT_ID_NAME, ISDUtils::getInstance()->intToString(commentbox->getId()));

    //type
    elem->set_attribute(COMMENT_TYPE_NAME, ISDUtils::getInstance()->intToString(commentbox->getType()));
    
    //Note: we don't need to save the language, it is saved by the language-list parent-element
    
    //position
    commentbox->getPosition(intVals);
    elem->set_attribute(COMMENT_POSITION_X_NAME, ISDUtils::getInstance()->intToString(intVals[0]));
    elem->set_attribute(COMMENT_POSITION_Y_NAME, ISDUtils::getInstance()->intToString(intVals[1]));

    //size
    commentbox->getSize(intVals);
    elem->set_attribute(COMMENT_WIDTH_NAME, ISDUtils::getInstance()->intToString(intVals[0]));
    elem->set_attribute(COMMENT_HEIGHT_NAME, ISDUtils::getInstance()->intToString(intVals[1]));

    //mirrormode
    elem->set_attribute(COMMENT_MIRRORMODE_NAME, ISDUtils::getInstance()->intToString(commentbox->getMirrorMode()));
    
    //timing
    elem->set_attribute(COMMENT_STARTTIME_NAME, ISDUtils::getInstance()->intToString(commentbox->getStartTime()));
    elem->set_attribute(COMMENT_DURATION_NAME, ISDUtils::getInstance()->intToString(commentbox->getDuration()));

    //color
    commentbox->getColor(floatVals);
    elem->set_attribute(COMMENT_COLOR_R_NAME, ISDUtils::getInstance()->floatToString(floatVals[0]));
    elem->set_attribute(COMMENT_COLOR_G_NAME, ISDUtils::getInstance()->floatToString(floatVals[1]));
    elem->set_attribute(COMMENT_COLOR_B_NAME, ISDUtils::getInstance()->floatToString(floatVals[2]));
    elem->set_attribute(COMMENT_COLOR_A_NAME, ISDUtils::getInstance()->floatToString(floatVals[3]));

    //textcolor
    commentbox->getTextColor(floatVals);
    elem->set_attribute(COMMENT_TEXTCOLOR_R_NAME, ISDUtils::getInstance()->floatToString(floatVals[0]));
    elem->set_attribute(COMMENT_TEXTCOLOR_G_NAME, ISDUtils::getInstance()->floatToString(floatVals[1]));
    elem->set_attribute(COMMENT_TEXTCOLOR_B_NAME, ISDUtils::getInstance()->floatToString(floatVals[2]));
    elem->set_attribute(COMMENT_TEXTCOLOR_A_NAME, ISDUtils::getInstance()->floatToString(floatVals[3]));

    //text
    elem->add_child_text(commentbox->getText());

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDXmlFile::fillCommentFromElement(ISDVideoCanvas* parent, xmlpp::Element* elem, ISDCommentbox*& commentbox, Glib::ustring& lang)
{
    if (elem==NULL) {
	LOG_WARNING("Tried to fill a commentbox with a NULL comment-element.");
	RETURN_ERROR(ISD_ARGS_ERROR);
    }
    
    string attr1Str, attr2Str, attr3Str, attr4Str;
    ISDUtils* utils = ISDUtils::getInstance();

    //get ID, type and create commentbox
    attr1Str = elem->get_attribute(COMMENT_ID_NAME)->get_value();
    attr2Str = elem->get_attribute(COMMENT_TYPE_NAME)->get_value();
    commentbox = ISDCommentbox::createCommentbox((ISDCommentbox::commentboxType)utils->stringToInt(attr2Str),
						 parent,
						 utils->stringToInt(attr1Str));

    //position
    attr1Str = elem->get_attribute(COMMENT_POSITION_X_NAME)->get_value();
    attr2Str = elem->get_attribute(COMMENT_POSITION_Y_NAME)->get_value();
    commentbox->setPosition(utils->stringToInt(attr1Str),
			    utils->stringToInt(attr2Str));

    //position
    attr1Str = elem->get_attribute(COMMENT_WIDTH_NAME)->get_value();
    attr2Str = elem->get_attribute(COMMENT_HEIGHT_NAME)->get_value();
    commentbox->setSize(utils->stringToInt(attr1Str),
			utils->stringToInt(attr2Str));

    //mirrormode
    attr1Str = elem->get_attribute(COMMENT_MIRRORMODE_NAME)->get_value();
    commentbox->setMirrorMode((ISDCommentbox::commentboxMirrorMode)utils->stringToInt(attr1Str));
    
    //timing
    attr1Str = elem->get_attribute(COMMENT_STARTTIME_NAME)->get_value();
    attr2Str = elem->get_attribute(COMMENT_DURATION_NAME)->get_value();
    commentbox->setStartTime(utils->stringToInt(attr1Str));
    commentbox->setDuration(utils->stringToInt(attr2Str));

    //color
    attr1Str = elem->get_attribute(COMMENT_COLOR_R_NAME)->get_value();
    attr2Str = elem->get_attribute(COMMENT_COLOR_G_NAME)->get_value();
    attr3Str = elem->get_attribute(COMMENT_COLOR_B_NAME)->get_value();
    attr4Str = elem->get_attribute(COMMENT_COLOR_A_NAME)->get_value();
    commentbox->setColor(utils->stringToFloat(attr1Str),
			 utils->stringToFloat(attr2Str),
			 utils->stringToFloat(attr3Str),
			 utils->stringToFloat(attr4Str));

    //textcolor
    attr1Str = elem->get_attribute(COMMENT_TEXTCOLOR_R_NAME)->get_value();
    attr2Str = elem->get_attribute(COMMENT_TEXTCOLOR_G_NAME)->get_value();
    attr3Str = elem->get_attribute(COMMENT_TEXTCOLOR_B_NAME)->get_value();
    attr4Str = elem->get_attribute(COMMENT_TEXTCOLOR_A_NAME)->get_value();
    commentbox->setTextColor(utils->stringToFloat(attr1Str),
			     utils->stringToFloat(attr2Str),
			     utils->stringToFloat(attr3Str),
			     utils->stringToFloat(attr4Str));

    //text
    xmlpp::TextNode* textNode = elem->get_child_text();
    Glib::ustring text = textNode==NULL?"":textNode->get_content();
    commentbox->setText(text);

    commentbox->setLanguage(lang);

    RETURN_SUCCESS;
}
xmlpp::NodeSet ISDXmlFile::getAllComments(xmlpp::Document* xmlDocument)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    Glib::ustring xpath;
    //select all the comments
    //format: ROOT_NAME/*/COMMENT_NAME[@COMMENT_ID_NAME]
    xpath = xpath + "/" + ROOT_NAME + "/*"
	+ "/" + COMMENT_NAME
	+ "[@" + COMMENT_ID_NAME + "]";

    return rootNode->find(xpath);
}
bool ISDXmlFile::commentExists(xmlpp::Document* xmlDocument, int id)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    Glib::ustring idStr(ISDUtils::getInstance()->intToString(id));
    Glib::ustring xpath;
    //select all the comments
    //format: ROOT_NAME/*/COMMENT_NAME[@COMMENT_ID_NAME='id']
    xpath = xpath + "/" + ROOT_NAME + "/"
	+ "*"
	+ "/" + COMMENT_NAME
	+ "[@" + COMMENT_ID_NAME + "='" + idStr + "']";

    xmlpp::NodeSet results = rootNode->find(xpath);

    return results.size()!=0;
}
xmlpp::Element* ISDXmlFile::getCommentLanguage(xmlpp::Document* xmlDocument, xmlpp::Element* elem)
{
    xmlpp::Element* rootNode = xmlDocument->get_root_node();
    Glib::ustring xpath;

    //First, find all the languagelist elements

    //format: ROOT_NAME/COMMENTLIST_NAME
    xpath = xpath + "/" + ROOT_NAME + "/"
	+ COMMENTLIST_NAME;

    xmlpp::NodeSet results = rootNode->find(xpath);
    xmlpp::NodeSet::const_iterator iter;
    ISDUtils* utils = ISDUtils::getInstance();

    //for every language, try to find the comment
    for (iter=results.begin();iter!=results.end();iter++) {
	xmlpp::Element* langElem = dynamic_cast<xmlpp::Element*>(*iter);
	
	//extract the comment-ID
	string tempStr = elem->get_attribute(COMMENT_ID_NAME)->get_value();
	int id = utils->stringToInt(tempStr);

	//extract the language-list language
	Glib::ustring lang = langElem->get_attribute(COMMENTLIST_LANGUAGE_NAME)->get_value();
	
	//if there exists a comment with that language and ID, we've found the language
	if (searchComment(xmlDocument, id, lang)!=NULL) {
	    return langElem;
	}
    }

    return NULL;
}
