/**
 * Copyright (c) 2006, www.jempbox.org
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of pdfbox; nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * http://www.jempbox.org
 *
 */
package org.jempbox.xmp;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jempbox.impl.XMLUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.ProcessingInstruction;

/**
 * This class represents the top level XMP data structure and gives access to the various schemas 
 * that are available as part of the XMP specification.
 *
 * @author Ben Litchfield (ben@benlitchfield.com)
 * @version $Revision: 1.1 $
 */
public class XMPMetadata
{
    /**
     * Supported encoding for persisted XML.
     */
    public static final String ENCODING_UTF8 = "UTF-8";
    /**
     * Supported encoding for persisted XML.
     */
    public static final String ENCODING_UTF16BE = "UTF-16BE";
    /**
     * Supported encoding for persisted XML.
     */
    public static final String ENCODING_UTF16LE = "UTF-16LE";
    
    
    /**
     * The DOM representation of the metadata.
     */
    protected Document xmpDocument;
    
    private String encoding = ENCODING_UTF8;
    private Map nsMappings = new HashMap();
    
    /**
     * Default constructor, creates blank XMP doc.
     *
     * @throws IOException If there is an error creating the initial document.
     */
    public XMPMetadata() throws IOException
    {
        xmpDocument = XMLUtil.newDocument();
        ProcessingInstruction beginXPacket = xmpDocument.createProcessingInstruction( "xpacket", 
                "begin=\"\uFEFF\" id=\"W5M0MpCehiHzreSzNTczkc9d\"" );
                
        xmpDocument.appendChild( beginXPacket );
        Element xmpMeta = xmpDocument.createElementNS( "adobe:ns:meta/", "x:xmpmeta" );
        xmpMeta.setAttributeNS( "adobe:ns:meta/", "xmlns:x", "adobe:ns:meta/" );
        
        xmpDocument.appendChild( xmpMeta );
        
        Element rdf = xmpDocument.createElement( "rdf:RDF" );
        rdf.setAttributeNS( "http://www.w3.org/1999/02/22-rdf-syntax-ns#", "xmlns:rdf", 
                            "http://www.w3.org/1999/02/22-rdf-syntax-ns#" );
        
        xmpMeta.appendChild( rdf );
        
        ProcessingInstruction endXPacket = xmpDocument.createProcessingInstruction( "xpacket", "end=\"w\"" );
        xmpDocument.appendChild( endXPacket );
        init();
    }
    
    /**
     * Constructor from an existing XML document.
     * 
     * @param doc The root XMP document.
     */
    public XMPMetadata( Document doc )
    {
        xmpDocument = doc;
        init();
    }
    
    private void init()
    {
        nsMappings.put( XMPSchemaPDF.NAMESPACE, XMPSchemaPDF.class );
        nsMappings.put( XMPSchemaBasic.NAMESPACE, XMPSchemaBasic.class );
        nsMappings.put( XMPSchemaDublinCore.NAMESPACE, XMPSchemaDublinCore.class );
        nsMappings.put( XMPSchemaMediaManagement.NAMESPACE, XMPSchemaMediaManagement.class );
        nsMappings.put( XMPSchemaRightsManagement.NAMESPACE, XMPSchemaRightsManagement.class );
    }
    
    /**
     * Save the XMP document to a file.
     *
     * @param file The file to save the XMP document to.
     * 
     * @throws Exception If there is an error while writing to the stream.
     */
    public void save( String file ) throws Exception
    {
        XMLUtil.save( xmpDocument, file, encoding );
    }
    
    /**
     * Get the XML document from this object.
     * 
     * @return This object as an XML document.
     */
    public Document getXMPDocument()
    {
        return xmpDocument;
    }
    
    /**
     * Create and add a new PDF Schema to this metadata.  Typically a XMP document
     * will only have one PDF schema (but multiple are supported) so it is recommended
     * that you first check the existence of a PDF scheme by using getPDFSchema()
     *
     * @return A new blank PDF schema that is now part of the metadata.
     */
    public XMPSchemaPDF addPDFSchema()
    {
        XMPSchemaPDF pdf = new XMPSchemaPDF(this);
        Element rdf = getRDFElement();
        rdf.appendChild( pdf.getElement() );
        return pdf;
    }
    
    /**
     * Create and add a new Dublin Core Schema to this metadata.  Typically a XMP document
     * will only have one schema for each type (but multiple are supported) so it is recommended
     * that you first check the existence of a this scheme by using getDublinCoreSchema()
     *
     * @return A new blank PDF schema that is now part of the metadata.
     */
    public XMPSchemaDublinCore addDublinCoreSchema()
    {
        XMPSchemaDublinCore dc = new XMPSchemaDublinCore(this);
        Element rdf = getRDFElement();
        rdf.appendChild( dc.getElement() );
        return dc;
    }
    
    /**
     * Create and add a new Basic Schema to this metadata.  Typically a XMP document
     * will only have one schema for each type (but multiple are supported) so it is recommended
     * that you first check the existence of a this scheme by using getDublinCoreSchema()
     *
     * @return A new blank PDF schema that is now part of the metadata.
     */
    public XMPSchemaBasic addBasicSchema()
    {
        XMPSchemaBasic dc = new XMPSchemaBasic(this);
        Element rdf = getRDFElement();
        rdf.appendChild( dc.getElement() );
        return dc;
    }
    
    /**
     * The encoding used to write the XML.  Default value:UTF-8<br/>
     * See the ENCODING_XXX constants
     *
     * @param xmlEncoding The encoding to write the XML as.
     */
    public void setEncoding( String xmlEncoding )
    {
        encoding = xmlEncoding;
    }
    
    /**
     * Get the current encoding that will be used to write the XML.
     *
     * @return The current encoding to write the XML to.
     */
    public String getEncoding()
    {
        return encoding;
    }
    
    /**
     * Get the root RDF element.
     *
     * @return The root RDF element.
     */
    private Element getRDFElement()
    {
        Element rdf = null;
        NodeList nodes = xmpDocument.getElementsByTagName( "rdf:RDF" );
        if( nodes.getLength() > 0 )
        {
            rdf = (Element)nodes.item( 0 );
        }
        return rdf;
    }
    
    /**
     * Load metadata from the filesystem.
     *
     * @param file The file to load the metadata from.
     * 
     * @return The loaded XMP document. 
     *
     * @throws IOException If there is an error reading the data.
     */
    public static XMPMetadata load( String file ) throws IOException
    {
        return new XMPMetadata( XMLUtil.parse( file ) );
    }
    
    /**
     * Load metadata from the filesystem.
     *
     * @param is The stream to load the data from.
     * 
     * @return The loaded XMP document. 
     *
     * @throws IOException If there is an error reading the data.
     */
    public static XMPMetadata load( InputStream is ) throws IOException
    {
        return new XMPMetadata( XMLUtil.parse( is ) );
    }
    
    /**
     * Test main program.
     * @param args The command line arguments.
     * @throws Exception If there is an error.
     */
    public static void main( String[] args ) throws Exception
    {
        XMPMetadata metadata = new XMPMetadata();
        XMPSchemaPDF pdf = metadata.addPDFSchema();
        pdf.setAbout( "uuid:b8659d3a-369e-11d9-b951-000393c97fd8" );
        pdf.setKeywords( "ben,bob,pdf" );
        pdf.setPDFVersion( "1.3");
        pdf.setProducer( "Acrobat Distiller 6.0.1 for Macintosh" );
        
        XMPSchemaDublinCore dc = metadata.addDublinCoreSchema();
        dc.addContributor( "Ben Litchfield" );
        dc.addContributor( "Solar Eclipse" );
        dc.addContributor( "Some Other Guy");
        
        XMPSchemaBasic basic = metadata.addBasicSchema();
        Thumbnail t = new Thumbnail( metadata );
        t.setFormat( Thumbnail.FORMAT_JPEG );
        t.setImage( "IMAGE_DATA" );
        t.setHeight( new Integer( 100 ) );
        t.setWidth( new Integer( 200 ) );
        basic.setThumbnail( t );
        basic.setBaseURL( "http://www.pdfbox.org/");
        
        List schemas = metadata.getSchemas();
        System.out.println( "schemas=" + schemas );
        
        metadata.save( "test.xmp" );
    }
    
    /**
     * This will get a list of XMPSchema(or subclass) objects.
     * 
     * @return A non null read-only list of schemas that are part of this metadata.
     * 
     * @throws IOException If there is an error creating a specific schema.
     */
    public List getSchemas() throws IOException
    {
        NodeList schemaList = xmpDocument.getElementsByTagName( "rdf:Description" );
        List retval = new ArrayList( schemaList.getLength() );
        for( int i=0; i<schemaList.getLength(); i++ )
        {
            Element schema = (Element)schemaList.item( i );
            boolean found = false;
            NamedNodeMap attributes = schema.getAttributes();
            for( int j=0; j<attributes.getLength() && !found; j++ )
            {
                Node attribute = attributes.item( j );
                String name = attribute.getNodeName();
                String value = attribute.getNodeValue();
                if( name.startsWith( "xmlns:" ) && nsMappings.containsKey( value ) )
                {
                    Class schemaClass = (Class)nsMappings.get( value );
                    try
                    {
                        Constructor ctor = schemaClass.getConstructor( new Class[]{Element.class} );
                        retval.add( ctor.newInstance( new Object[] {schema}) );
                        found = true;
                    }
                    catch( NoSuchMethodException e )
                    {
                        throw new IOException( "Error: Class " + schemaClass.getName() + " must have a constructor with the signature of " + schemaClass.getName() + "( org.w3c.dom.Element )");
                    }
                    catch( Exception e )
                    {
                        e.printStackTrace();
                        throw new IOException( e.getMessage() );
                    }
                }
            }
            if( !found )
            {
                retval.add( new XMPSchema( schema ) );
            }
        }
        return retval;
    }
    
    

}
