// $Id: fe_base.C,v 1.36 2006/09/19 02:37:42 roystgnr Exp $

// The libMesh Finite Element Library.
// Copyright (C) 2002-2005  Benjamin S. Kirk, John W. Peterson
  
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
  
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
  
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA



// Local includes
#include "fe.h"
#include "inf_fe.h"
#include "libmesh_logging.h"
// For coarsening code:
#include "dense_matrix.h"
#include "dense_vector.h"
#include "dof_map.h"
#include "elem.h"
#include "fe_interface.h"
#include "numeric_vector.h"
#include "quadrature.h"



// ------------------------------------------------------------
// FEBase class members
AutoPtr<FEBase> FEBase::build (const unsigned int dim,
			       const FEType& fet)
{
  // The stupid AutoPtr<FEBase> ap(); return ap;
  // construct is required to satisfy IBM's xlC

  switch (dim)
    {
      // 1D
    case 1:
      {
	switch (fet.family)
	  {
	  case CLOUGH:
	    {
	      AutoPtr<FEBase> ap(new FE<1,CLOUGH>(fet));
	      return ap;
	    }
	    
	  case HERMITE:
	    {
	      AutoPtr<FEBase> ap(new FE<1,HERMITE>(fet));
	      return ap;
	    }
	    
	  case LAGRANGE:
	    {
	      AutoPtr<FEBase> ap(new FE<1,LAGRANGE>(fet));
	      return ap;
	    }
		   
	  case HIERARCHIC:
	    {
	      AutoPtr<FEBase> ap(new FE<1,HIERARCHIC>(fet));
	      return ap;
	    }
	    
	  case MONOMIAL:
	    {
	      AutoPtr<FEBase> ap(new FE<1,MONOMIAL>(fet));
	      return ap;
	    }
	    
#ifdef ENABLE_HIGHER_ORDER_SHAPES
	  case SZABAB:
	    {
	      AutoPtr<FEBase> ap(new FE<1,SZABAB>(fet));
	      return ap;
	    }

	  case BERNSTEIN:
	    {
	      AutoPtr<FEBase> ap(new FE<1,BERNSTEIN>(fet));
	      return ap;
	    }
#endif

	  case XYZ:
	    {
	      AutoPtr<FEBase> ap(new FEXYZ<1>(fet));
	      return ap;
	    }

	  default:
	    std::cout << "ERROR: Bad FEType.family= " << fet.family << std::endl;
	    error();
	  }
      }

      
      // 2D
    case 2:
      {
	switch (fet.family)
	  {
	  case CLOUGH:
	    {
	      AutoPtr<FEBase> ap(new FE<2,CLOUGH>(fet));
	      return ap;
	    }
	    
	  case HERMITE:
	    {
	      AutoPtr<FEBase> ap(new FE<2,HERMITE>(fet));
	      return ap;
	    }

	  case LAGRANGE:
	    {
	      AutoPtr<FEBase> ap(new FE<2,LAGRANGE>(fet));
	      return ap;
	    }
	    
	  case HIERARCHIC:
	    {
	      AutoPtr<FEBase> ap(new FE<2,HIERARCHIC>(fet));
	      return ap;
	    }
	    
	  case MONOMIAL:
	    {
	      AutoPtr<FEBase> ap(new FE<2,MONOMIAL>(fet));
	      return ap;
	    }
	    
#ifdef ENABLE_HIGHER_ORDER_SHAPES
	  case SZABAB:
	    {
	      AutoPtr<FEBase> ap(new FE<2,SZABAB>(fet));
	      return ap;
	    }

	  case BERNSTEIN:
	    {
	      AutoPtr<FEBase> ap(new FE<2,BERNSTEIN>(fet));
	      return ap;
	    }
#endif

	  case XYZ:
	    {
	      AutoPtr<FEBase> ap(new FEXYZ<2>(fet));
	      return ap;
	    }

	  default:
	    std::cout << "ERROR: Bad FEType.family= " << fet.family << std::endl;
	    error();
	  }
      }

      
      // 3D
    case 3:
      {
	switch (fet.family)
	  {
	  case CLOUGH:
	    {
	      std::cout << "ERROR: Clough-Tocher elements currently only support 1D and 2D" <<
		      std::endl;
	      error();
	    }
	    
	  case HERMITE:
	    {
	      AutoPtr<FEBase> ap(new FE<3,HERMITE>(fet));
	      return ap;
	    }
	    
	  case LAGRANGE:
	    {
	      AutoPtr<FEBase> ap(new FE<3,LAGRANGE>(fet));
	      return ap;
	    }
	    
	  case HIERARCHIC:
	    {
	      AutoPtr<FEBase> ap(new FE<3,HIERARCHIC>(fet));
	      return ap;
	    }
	    
	  case MONOMIAL:
	    {
	      AutoPtr<FEBase> ap(new FE<3,MONOMIAL>(fet));
	      return ap;
	    }
	    
#ifdef ENABLE_HIGHER_ORDER_SHAPES
	  case SZABAB:
	    {
	      AutoPtr<FEBase> ap(new FE<3,SZABAB>(fet));
	      return ap;
	    }

	  case BERNSTEIN:
	    {
	      AutoPtr<FEBase> ap(new FE<3,BERNSTEIN>(fet));
	      return ap;
	    }
#endif

	  case XYZ:
	    {
	      AutoPtr<FEBase> ap(new FEXYZ<3>(fet));
	      return ap;
	    }

	  default:
	    std::cout << "ERROR: Bad FEType.family= " << fet.family << std::endl;
	    error();
	  }
      }

    default:
      error();
    }

  error();
  AutoPtr<FEBase> ap(NULL);
  return ap;
}







#ifdef ENABLE_INFINITE_ELEMENTS


AutoPtr<FEBase> FEBase::build_InfFE (const unsigned int dim,
				     const FEType& fet)
{
  // The stupid AutoPtr<FEBase> ap(); return ap;
  // construct is required to satisfy IBM's xlC

  switch (dim)
    {

      // 1D
    case 1:
      {
	switch (fet.radial_family)
	  {
	  case INFINITE_MAP:
	    {
	      std::cerr << "ERROR: Don't build an infinite element " << std::endl
			<< " with FEFamily = " << fet.radial_family << std::endl;
	      error();
	    }

	  case JACOBI_20_00:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<1,JACOBI_20_00,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case JACOBI_30_00:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<1,JACOBI_30_00,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case LEGENDRE:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<1,LEGENDRE,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case LAGRANGE:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<1,LAGRANGE,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }


	    
	  default:
	    std::cerr << "ERROR: Bad FEType.radial_family= " << fet.radial_family << std::endl;
	    error();
	  }

      }

      


      // 2D
    case 2:
      {
	switch (fet.radial_family)
	  {
	  case INFINITE_MAP:
	    {
	      std::cerr << "ERROR: Don't build an infinite element " << std::endl
			<< " with FEFamily = " << fet.radial_family << std::endl;
	      error();
	    }

	  case JACOBI_20_00:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<2,JACOBI_20_00,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case JACOBI_30_00:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<2,JACOBI_30_00,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case LEGENDRE:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<2,LEGENDRE,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case LAGRANGE:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<2,LAGRANGE,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }


	    
	  default:
	    std::cerr << "ERROR: Bad FEType.radial_family= " << fet.radial_family << std::endl;
	    error();
	  }

      }

      


      // 3D
    case 3:
      {
	switch (fet.radial_family)
	  {
	  case INFINITE_MAP:
	    {
	      std::cerr << "ERROR: Don't build an infinite element " << std::endl
			<< " with FEFamily = " << fet.radial_family << std::endl;
	      error();
	    }

	  case JACOBI_20_00:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<3,JACOBI_20_00,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case JACOBI_30_00:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<3,JACOBI_30_00,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case LEGENDRE:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<3,LEGENDRE,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }

	  case LAGRANGE:
	    {
  	      switch (fet.inf_map)
	        {
		  case CARTESIAN:
		    {
		      AutoPtr<FEBase> ap(new InfFE<3,LAGRANGE,CARTESIAN>(fet));
		      return ap;
		    }
		  default:
		    std::cerr << "ERROR: Don't build an infinite element " << std::endl
			      << " with InfMapType = " << fet.inf_map << std::endl;
		    error();
		}
	    }


	    
	  default:
	    std::cerr << "ERROR: Bad FEType.radial_family= " << fet.radial_family << std::endl;
	    error();
	  }
      }

    default:
      error();
    }

  error();
  AutoPtr<FEBase> ap(NULL);
  return ap;
}




#endif // ifdef ENABLE_INFINITE_ELEMENTS











void FEBase::compute_shape_functions (const Elem*)
{
  //-------------------------------------------------------------------------
  // Compute the shape function values (and derivatives)
  // at the Quadrature points.  Note that the actual values
  // have already been computed via init_shape_functions

  // Start logging the shape function computation
  START_LOG("compute_shape_functions()", "FE");

  calculations_started = true;

  // If the user forgot to request anything, we'll be safe and
  // calculate everything:
#ifdef ENABLE_SECOND_DERIVATIVES
  if (!calculate_phi && !calculate_dphi && !calculate_d2phi)
    calculate_phi = calculate_dphi = calculate_d2phi = true;
#else
  if (!calculate_phi && !calculate_dphi)
    calculate_phi = calculate_dphi = true;
#endif // ENABLE_SECOND_DERIVATIVES

  // Compute the value of the derivative shape function i at quadrature point p
  switch (dim)
    {
      
    case 1:
      {
	if (calculate_dphi)
	  for (unsigned int i=0; i<dphi.size(); i++)
	    for (unsigned int p=0; p<dphi[i].size(); p++)
	      {
	        // dphi/dx    = (dphi/dxi)*(dxi/dx)
	        dphi[i][p](0) =
		  dphidx[i][p] = dphidxi[i][p]*dxidx_map[p];
	      
	        dphi[i][p](1) = dphidy[i][p] = 0.;
	        dphi[i][p](2) = dphidz[i][p] = 0.;
	      }
#ifdef ENABLE_SECOND_DERIVATIVES
	if (calculate_d2phi)
	  for (unsigned int i=0; i<d2phi.size(); i++)
	    for (unsigned int p=0; p<d2phi[i].size(); p++)
	      {
	        d2phi[i][p](0,0) = d2phidx2[i][p] = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidx_map[p];
#if DIM>1
	        d2phi[i][p](0,1) = d2phidxdy[i][p] = 
		  d2phi[i][p](1,0) = 0.;
	        d2phi[i][p](1,1) = d2phidy2[i][p] = 0.;
#if DIM>2
	        d2phi[i][p](0,2) = d2phidxdz[i][p] =
		  d2phi[i][p](2,0) = 0.;
	        d2phi[i][p](1,2) = d2phidydz[i][p] = 
		  d2phi[i][p](2,1) = 0.;
	        d2phi[i][p](2,2) = d2phidz2[i][p] = 0.;
#endif
#endif
	      }
#endif

	// All done
	break;
      }

    case 2:
      {
	if (calculate_dphi)
	  for (unsigned int i=0; i<dphi.size(); i++)
	    for (unsigned int p=0; p<dphi[i].size(); p++)
	      {
	        // dphi/dx    = (dphi/dxi)*(dxi/dx) + (dphi/deta)*(deta/dx)
	        dphi[i][p](0) =
		  dphidx[i][p] = (dphidxi[i][p]*dxidx_map[p] +
				  dphideta[i][p]*detadx_map[p]);
	      
	        // dphi/dy    = (dphi/dxi)*(dxi/dy) + (dphi/deta)*(deta/dy)
	        dphi[i][p](1) =
		  dphidy[i][p] = (dphidxi[i][p]*dxidy_map[p] +
				  dphideta[i][p]*detady_map[p]);
	      
	        // dphi/dz    = (dphi/dxi)*(dxi/dz) + (dphi/deta)*(deta/dz)
#if DIM == 3  
	        dphi[i][p](2) = // can only assign to the Z component if DIM==3
#endif
		dphidz[i][p] = (dphidxi[i][p]*dxidz_map[p] +
				dphideta[i][p]*detadz_map[p]);
	      }

#ifdef ENABLE_SECOND_DERIVATIVES
	if (calculate_d2phi)
	  for (unsigned int i=0; i<d2phi.size(); i++)
	    for (unsigned int p=0; p<d2phi[i].size(); p++)
	      {
	        d2phi[i][p](0,0) = d2phidx2[i][p] = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidx_map[p] +
		  2*d2phidxideta[i][p]*dxidx_map[p]*detadx_map[p] +
		  d2phideta2[i][p]*detadx_map[p]*detadx_map[p];
	        d2phi[i][p](0,1) = d2phidxdy[i][p] =
		  d2phi[i][p](1,0) = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidy_map[p] +
		  d2phidxideta[i][p]*dxidx_map[p]*detady_map[p] +
		  d2phideta2[i][p]*detadx_map[p]*detady_map[p] +
		  d2phidxideta[i][p]*detadx_map[p]*dxidy_map[p];
	        d2phi[i][p](1,1) = d2phidy2[i][p] =
		  d2phidxi2[i][p]*dxidy_map[p]*dxidy_map[p] +
		  2*d2phidxideta[i][p]*dxidy_map[p]*detady_map[p] +
		  d2phideta2[i][p]*detady_map[p]*detady_map[p];
#if DIM == 3  
	        d2phi[i][p](0,2) = d2phidxdz[i][p] = 
		  d2phi[i][p](2,0) = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidz_map[p] +
		  d2phidxideta[i][p]*dxidx_map[p]*detadz_map[p] +
		  d2phideta2[i][p]*detadx_map[p]*detadz_map[p] +
		  d2phidxideta[i][p]*detadx_map[p]*dxidz_map[p];
	        d2phi[i][p](1,2) = d2phidydz[i][p] = 
		  d2phi[i][p](2,1) =
		  d2phidxi2[i][p]*dxidy_map[p]*dxidz_map[p] +
		  d2phidxideta[i][p]*dxidy_map[p]*detadz_map[p] +
		  d2phideta2[i][p]*detady_map[p]*detadz_map[p] +
		  d2phidxideta[i][p]*detady_map[p]*dxidz_map[p];
	        d2phi[i][p](2,2) = d2phidz2[i][p] =
		  d2phidxi2[i][p]*dxidz_map[p]*dxidz_map[p] +
		  2*d2phidxideta[i][p]*dxidz_map[p]*detadz_map[p] +
		  d2phideta2[i][p]*detadz_map[p]*detadz_map[p];
#endif
	      }
#endif

	// All done
	break;
      }
    
    case 3:
      {
	if (calculate_dphi)
	  for (unsigned int i=0; i<dphi.size(); i++)
	    for (unsigned int p=0; p<dphi[i].size(); p++)
	      {
	        // dphi/dx    = (dphi/dxi)*(dxi/dx) + (dphi/deta)*(deta/dx) + (dphi/dzeta)*(dzeta/dx);
	        dphi[i][p](0) =
		  dphidx[i][p] = (dphidxi[i][p]*dxidx_map[p] +
				  dphideta[i][p]*detadx_map[p] +
				  dphidzeta[i][p]*dzetadx_map[p]);
		
	        // dphi/dy    = (dphi/dxi)*(dxi/dy) + (dphi/deta)*(deta/dy) + (dphi/dzeta)*(dzeta/dy);
	        dphi[i][p](1) =
		  dphidy[i][p] = (dphidxi[i][p]*dxidy_map[p] +
				  dphideta[i][p]*detady_map[p] +
				  dphidzeta[i][p]*dzetady_map[p]);
		
	        // dphi/dz    = (dphi/dxi)*(dxi/dz) + (dphi/deta)*(deta/dz) + (dphi/dzeta)*(dzeta/dz);
	        dphi[i][p](2) =
		  dphidz[i][p] = (dphidxi[i][p]*dxidz_map[p] +
				  dphideta[i][p]*detadz_map[p] +
				  dphidzeta[i][p]*dzetadz_map[p]);	      
	      }

#ifdef ENABLE_SECOND_DERIVATIVES
	if (calculate_d2phi)
	  for (unsigned int i=0; i<d2phi.size(); i++)
	    for (unsigned int p=0; p<d2phi[i].size(); p++)
	      {
	        d2phi[i][p](0,0) = d2phidx2[i][p] = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidx_map[p] +
		  2*d2phidxideta[i][p]*dxidx_map[p]*detadx_map[p] +
		  2*d2phidxidzeta[i][p]*dxidx_map[p]*dzetadx_map[p] +
		  2*d2phidetadzeta[i][p]*detadx_map[p]*dzetadx_map[p] +
		  d2phideta2[i][p]*detadx_map[p]*detadx_map[p] +
		  d2phidzeta2[i][p]*dzetadx_map[p]*dzetadx_map[p];
	        d2phi[i][p](0,1) = d2phidxdy[i][p] =
		  d2phi[i][p](1,0) = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidy_map[p] +
		  d2phidxideta[i][p]*dxidx_map[p]*detady_map[p] +
		  d2phidxidzeta[i][p]*dxidx_map[p]*dzetady_map[p] +
		  d2phideta2[i][p]*detadx_map[p]*detady_map[p] +
		  d2phidxideta[i][p]*detadx_map[p]*dxidy_map[p] +
		  d2phidetadzeta[i][p]*detadx_map[p]*dzetady_map[p] +
		  d2phidzeta2[i][p]*dzetadx_map[p]*dzetady_map[p] +
		  d2phidxidzeta[i][p]*dzetadx_map[p]*dxidy_map[p] +
		  d2phidetadzeta[i][p]*dzetadx_map[p]*detady_map[p];
	        d2phi[i][p](0,2) = d2phidxdz[i][p] = 
		  d2phi[i][p](2,0) = 
		  d2phidxi2[i][p]*dxidx_map[p]*dxidz_map[p] +
		  d2phidxideta[i][p]*dxidx_map[p]*detadz_map[p] +
		  d2phidxidzeta[i][p]*dxidx_map[p]*dzetadz_map[p] +
		  d2phideta2[i][p]*detadx_map[p]*detadz_map[p] +
		  d2phidxideta[i][p]*detadx_map[p]*dxidz_map[p] +
		  d2phidetadzeta[i][p]*detadx_map[p]*dzetadz_map[p] +
		  d2phidzeta2[i][p]*dzetadx_map[p]*dzetadz_map[p] +
		  d2phidxidzeta[i][p]*dzetadx_map[p]*dxidz_map[p] +
		  d2phidetadzeta[i][p]*dzetadx_map[p]*detadz_map[p];
	        d2phi[i][p](1,1) = d2phidy2[i][p] =
		  d2phidxi2[i][p]*dxidy_map[p]*dxidy_map[p] +
		  2*d2phidxideta[i][p]*dxidy_map[p]*detady_map[p] +
		  2*d2phidxidzeta[i][p]*dxidy_map[p]*dzetady_map[p] +
		  2*d2phidetadzeta[i][p]*detady_map[p]*dzetady_map[p] +
		  d2phideta2[i][p]*detady_map[p]*detady_map[p] +
		  d2phidzeta2[i][p]*dzetady_map[p]*dzetady_map[p];
	        d2phi[i][p](1,2) = d2phidydz[i][p] = 
		  d2phi[i][p](2,1) =
		  d2phidxi2[i][p]*dxidy_map[p]*dxidz_map[p] +
		  d2phidxideta[i][p]*dxidy_map[p]*detadz_map[p] +
		  d2phidxidzeta[i][p]*dxidy_map[p]*dzetadz_map[p] +
		  d2phideta2[i][p]*detady_map[p]*detadz_map[p] +
		  d2phidxideta[i][p]*detady_map[p]*dxidz_map[p] +
		  d2phidetadzeta[i][p]*detady_map[p]*dzetadz_map[p] +
		  d2phidzeta2[i][p]*dzetady_map[p]*dzetadz_map[p] +
		  d2phidxidzeta[i][p]*dzetady_map[p]*dxidz_map[p] +
		  d2phidetadzeta[i][p]*dzetady_map[p]*detadz_map[p];
	        d2phi[i][p](2,2) = d2phidz2[i][p] =
		  d2phidxi2[i][p]*dxidz_map[p]*dxidz_map[p] +
		  2*d2phidxideta[i][p]*dxidz_map[p]*detadz_map[p] +
		  2*d2phidxidzeta[i][p]*dxidz_map[p]*dzetadz_map[p] +
		  2*d2phidetadzeta[i][p]*detadz_map[p]*dzetadz_map[p] +
		  d2phideta2[i][p]*detadz_map[p]*detadz_map[p] +
		  d2phidzeta2[i][p]*dzetadz_map[p]*dzetadz_map[p];
	      }
#endif
	// All done
	break;
      }

    default:
      {
	error();
      }
    }
  
  // Stop logging the shape function computation
  STOP_LOG("compute_shape_functions()", "FE");
}



bool FEBase::on_reference_element(const Point& p, const ElemType t, const Real eps)
{
  assert (eps >= 0.);
  
  const Real xi   = p(0);
  const Real eta  = p(1);
  const Real zeta = p(2);
  
  switch (t)
    {

    case EDGE2:
    case EDGE3:
    case EDGE4:
      {
	// The reference 1D element is [-1,1].
	if ((xi >= -1.-eps) &&
	    (xi <=  1.+eps))
	  return true;

	return false;
      }

      
    case TRI3:
    case TRI6:
      {
	// The reference triangle is isocoles
	// and is bound by xi=0, eta=0, and xi+eta=1.
	if ((xi  >= 0.-eps) &&
	    (eta >= 0.-eps) &&
	    ((xi + eta) <= 1.+eps))
	  return true;

	return false;
      }

      
    case QUAD4:
    case QUAD8:
    case QUAD9:
      {
	// The reference quadrilateral element is [-1,1]^2.
	if ((xi  >= -1.-eps) &&
	    (xi  <=  1.+eps) &&
	    (eta >= -1.-eps) &&
	    (eta <=  1.+eps))
	  return true;
		
	return false;
      }


    case TET4:
    case TET10:
      {
	// The reference tetrahedral is isocoles
	// and is bound by xi=0, eta=0, zeta=0,
	// and xi+eta+zeta=1.
	if ((xi   >= 0.-eps) &&
	    (eta  >= 0.-eps) &&
	    (zeta >= 0.-eps) &&
	    ((xi + eta + zeta) <= 1.+eps))
	  return true;
		
	return false;
      }

      
    case HEX8:
    case HEX20:
    case HEX27:
      {
	/*
	  if ((xi   >= -1.) &&
	  (xi   <=  1.) &&
	  (eta  >= -1.) &&
	  (eta  <=  1.) &&
	  (zeta >= -1.) &&
	  (zeta <=  1.))
	  return true;
	*/
	
	// The reference hexahedral element is [-1,1]^3.
	if ((xi   >= -1.-eps) &&
	    (xi   <=  1.+eps) &&
	    (eta  >= -1.-eps) &&
	    (eta  <=  1.+eps) &&
	    (zeta >= -1.-eps) &&
	    (zeta <=  1.+eps))
	  {
	    //	    std::cout << "Strange Point:\n";
	    //	    p.print();
	    return true;
	  }

	return false;
      }

    case PRISM6:
    case PRISM15:
    case PRISM18:
      {
	// Figure this one out...
	// inside the reference triange with zeta in [-1,1]
	if ((xi   >=  0.-eps) &&
	    (eta  >=  0.-eps) &&
	    (zeta >= -1.-eps) &&
	    (zeta <=  1.+eps) &&
	    ((xi + eta) <= 1.+eps))
	  return true;

	return false;
      }


    case PYRAMID5:
      {
	std::cerr << "BEN: Implement this you lazy bastard!"
		  << std::endl;
	error();

	return false;
      }

#ifdef ENABLE_INFINITE_ELEMENTS
    case INFHEX8:
      {      
	// The reference infhex8 is a [-1,1]^3.
	if ((xi   >= -1.-eps) &&
	    (xi   <=  1.+eps) &&
	    (eta  >= -1.-eps) &&
	    (eta  <=  1.+eps) &&
	    (zeta >= -1.-eps) &&
	    (zeta <=  1.+eps))
	  {
	    return true;
	  }
	return false;
      }

    case INFPRISM6:
      {      
	// inside the reference triange with zeta in [-1,1]
	if ((xi   >=  0.-eps) &&
	    (eta  >=  0.-eps) &&
	    (zeta >= -1.-eps) &&
	    (zeta <=  1.+eps) &&
	    ((xi + eta) <= 1.+eps))
	  {
	    return true;
	  }

	return false;
      }
#endif

    default:
      std::cerr << "ERROR: Unknown element type " << t << std::endl;
      error();
    }

  // If we get here then the point is _not_ in the
  // reference element.   Better return false.
  
  return false;
}




void FEBase::print_JxW(std::ostream& os) const
{
  for (unsigned int i=0; i<JxW.size(); ++i)
    os << JxW[i] << std::endl;
}




void FEBase::print_phi(std::ostream& os) const
{
  for (unsigned int i=0; i<phi.size(); ++i)
    for (unsigned int j=0; j<phi[i].size(); ++j)
      os << " phi[" << i << "][" << j << "]=" << phi[i][j] << std::endl;
}




void FEBase::print_dphi(std::ostream& os) const
{
  for (unsigned int i=0; i<dphi.size(); ++i)
    for (unsigned int j=0; j<dphi[i].size(); ++j)
      os << " dphi[" << i << "][" << j << "]=" << dphi[i][j];
}



#ifdef ENABLE_SECOND_DERIVATIVES


void FEBase::print_d2phi(std::ostream& os) const
{
  for (unsigned int i=0; i<dphi.size(); ++i)
    for (unsigned int j=0; j<dphi[i].size(); ++j)
      os << " d2phi[" << i << "][" << j << "]=" << d2phi[i][j];
}

#endif




void FEBase::print_xyz(std::ostream& os) const
{
  for (unsigned int i=0; i<xyz.size(); ++i)
    os << xyz[i];
}




void FEBase::print_info(std::ostream& os) const
{
  os << "Shape functions at the Gauss pts." << std::endl;
  this->print_phi(os);
  
  os << "Shape function gradients at the Gauss pts." << std::endl;
  this->print_dphi(os);
  
  os << "XYZ locations of the Gauss pts." << std::endl;
  this->print_xyz(os);
  
  os << "Values of JxW at the Gauss pts." << std::endl;
  this->print_JxW(os);
}




std::ostream& operator << (std::ostream& os, const FEBase& fe)
{
  fe.print_info(os);
  return os;
}



void FEBase::coarsened_dof_values(const NumericVector<Number> &old_vector,
                                  const DofMap &dof_map,
                                  const Elem *elem,
                                  DenseVector<Number> &Ue,
                                  const unsigned int var,
                                  const bool use_old_dof_indices)
{
  // The global DOF indices
  std::vector<unsigned int> old_dof_indices;
  if (use_old_dof_indices)
    dof_map.old_dof_indices (elem, old_dof_indices, var);
  else
    dof_map.dof_indices (elem, old_dof_indices, var);

  // Side/edge DOF indices
  std::vector<unsigned int> new_side_dofs, old_side_dofs;

  // FIXME: what about 2D shells in 3D space?
  unsigned int dim = elem->dim();

  // We use local FE objects for now
  // FIXME: we should use more, external objects instead for efficiency
  const FEType& base_fe_type = dof_map.variable_type(var);
  AutoPtr<FEBase> fe (FEBase::build(dim, base_fe_type));
  AutoPtr<FEBase> fe_coarse (FEBase::build(dim, base_fe_type));

  AutoPtr<QBase> qrule     (base_fe_type.default_quadrature_rule(dim));
  AutoPtr<QBase> qedgerule (base_fe_type.default_quadrature_rule(1));
  AutoPtr<QBase> qsiderule (base_fe_type.default_quadrature_rule(dim-1));
  std::vector<Point> coarse_qpoints;

  // The values of the shape functions at the quadrature
  // points
  const std::vector<std::vector<Real> >& phi_values =
    fe->get_phi();
  const std::vector<std::vector<Real> >& phi_coarse =
    fe_coarse->get_phi();

  // The gradients of the shape functions at the quadrature
  // points on the child element.
  const std::vector<std::vector<RealGradient> > *dphi_values =
    NULL;
  const std::vector<std::vector<RealGradient> > *dphi_coarse =
    NULL;

  const FEContinuity cont = fe->get_continuity();

  if (cont == C_ONE)
    {
      const std::vector<std::vector<RealGradient> >&
        ref_dphi_values = fe->get_dphi();
      dphi_values = &ref_dphi_values;
      const std::vector<std::vector<RealGradient> >&
        ref_dphi_coarse = fe_coarse->get_dphi();
      dphi_coarse = &ref_dphi_coarse;
    }

      // The Jacobian * quadrature weight at the quadrature points
      const std::vector<Real>& JxW =
        fe->get_JxW();

      // The XYZ locations of the quadrature points on the
      // child element
      const std::vector<Point>& xyz_values =
        fe->get_xyz();



  FEType fe_type = base_fe_type, temp_fe_type;
  const ElemType elem_type = elem->type();
  fe_type.order = static_cast<Order>(fe_type.order +
                                     elem->max_descendant_p_level());

  // Number of nodes on parent element
  const unsigned int n_nodes = elem->n_nodes();

  // Number of dofs on parent element
  const unsigned int new_n_dofs =
    FEInterface::n_dofs(dim, fe_type, elem_type);

  // Fixed vs. free DoFs on edge/face projections
  std::vector<char> dof_is_fixed(new_n_dofs, false); // bools
  std::vector<int> free_dof(new_n_dofs, 0);

  DenseMatrix<Real> Ke;
  DenseVector<Number> Fe;
  Ue.resize(new_n_dofs); Ue.zero();


  // When coarsening, in general, we need a series of
  // projections to ensure a unique and continuous
  // solution.  We start by interpolating nodes, then
  // hold those fixed and project edges, then
  // hold those fixed and project faces, then
  // hold those fixed and project interiors

  // Copy node values first
  unsigned int current_dof = 0;
  for (unsigned int n=0; n!= n_nodes; ++n)
    {
      // FIXME: this should go through the DofMap,
      // not duplicate dof_indices code badly!
      // We're also assuming here that child n shares
      // node n
      const unsigned int my_nc =
        FEInterface::n_dofs_at_node (dim, fe_type,
                                     elem_type, n);
      if (!elem->is_vertex(n))
        {
          current_dof += my_nc;
          continue;
        }

      temp_fe_type = base_fe_type;
      if (elem->child(n)->p_level() < elem->p_level())
        {
          temp_fe_type.order = 
            static_cast<Order>(temp_fe_type.order +
                               elem->child(n)->p_level());
        }
      const unsigned int nc =
        FEInterface::n_dofs_at_node (dim, temp_fe_type,
                                     elem_type, n);
      for (unsigned int i=0; i!= nc; ++i)
        {
          Ue(current_dof) =
            old_vector(old_dof_indices[current_dof]);
          dof_is_fixed[current_dof] = true;
          current_dof++;
        }
    }

  // In 3D, project any edge values next
  if (dim > 2 && cont != DISCONTINUOUS)
    for (unsigned int e=0; e != elem->n_edges(); ++e)
      {
        FEInterface::dofs_on_edge(elem, dim, fe_type,
                                  e, new_side_dofs);

        // Some edge dofs are on nodes and already
        // fixed, others are free to calculate
        unsigned int free_dofs = 0;
        for (unsigned int i=0; i !=
             new_side_dofs.size(); ++i)
          if (!dof_is_fixed[new_side_dofs[i]])
            free_dof[free_dofs++] = i;
        Ke.resize (free_dofs, free_dofs); Ke.zero();
        Fe.resize (free_dofs); Fe.zero();
        // The new edge coefficients
        DenseVector<Number> Uedge(free_dofs);

        // Add projection terms from each child sharing
        // this edge
        for (unsigned int c=0; c != elem->n_children();
             ++c)
          {
            if (!elem->is_child_on_edge(c,e))
              continue;
            Elem *child = elem->child(c);
            dof_map.old_dof_indices (child,
              old_dof_indices, var);
            const unsigned int old_n_dofs = old_dof_indices.size();

            temp_fe_type = base_fe_type;
            temp_fe_type.order = 
              static_cast<Order>(temp_fe_type.order +
                                 child->p_level());

            FEInterface::dofs_on_edge(child, dim,
              temp_fe_type, e, old_side_dofs);

            // Initialize both child and parent FE data
            // on the child's edge
            fe->attach_quadrature_rule (qedgerule.get());
            fe->edge_reinit (child, e);
            const unsigned int n_qp = qedgerule->n_points();

            FEInterface::inverse_map (dim, fe_type, elem,
                            xyz_values, coarse_qpoints);

            fe_coarse->reinit(elem, &coarse_qpoints);

            // Loop over the quadrature points
            for (unsigned int qp=0; qp<n_qp; qp++)
              {
                // solution value at the quadrature point
                Number fineval = libMesh::zero;
                // solution grad at the quadrature point
                Gradient finegrad;

                // Sum the solution values * the DOF
                // values at the quadrature point to
                // get the solution value and gradient.
                for (unsigned int i=0; i<old_n_dofs;
                     i++)
                  {
                    fineval +=
                      (old_vector(old_dof_indices[i])*
                      phi_values[i][qp]);
                    if (cont == C_ONE)
                      finegrad.add_scaled((*dphi_values)[i][qp],
                                          old_vector(old_dof_indices[i]));
                  }

                // Form edge projection matrix
                for (unsigned int sidei=0, freei=0; 
                     sidei != new_side_dofs.size();
                     ++sidei)
                  {
                    unsigned int i = new_side_dofs[sidei];
                    // fixed DoFs aren't test functions
                    if (dof_is_fixed[i])
                      continue;
                    for (unsigned int sidej=0, freej=0;
                         sidej != new_side_dofs.size();
                         ++sidej)
                      {
                        unsigned int j =
                          new_side_dofs[sidej];
                        if (dof_is_fixed[j])
                          Fe(freei) -=
                            phi_coarse[i][qp] *
                            phi_coarse[j][qp] * JxW[qp] *
                            Ue(j);
                        else
                          Ke(freei,freej) +=
                            phi_coarse[i][qp] *
                            phi_coarse[j][qp] * JxW[qp];
                        if (cont == C_ONE)
                          {
                            if (dof_is_fixed[j])
                              Fe(freei) -=
                                ((*dphi_coarse)[i][qp] *
                                 (*dphi_coarse)[j][qp]) *
                                JxW[qp] *
                                Ue(j);
                            else
                              Ke(freei,freej) +=
                                ((*dphi_coarse)[i][qp] *
                                 (*dphi_coarse)[j][qp])
                                * JxW[qp];
                          }
                        if (!dof_is_fixed[j])
                          freej++;
                      }
                    Fe(freei) += phi_coarse[i][qp] *
                                 fineval * JxW[qp];
                    if (cont == C_ONE)
                      Fe(freei) +=
                        (finegrad * (*dphi_coarse)[i][qp]) * JxW[qp];
                    freei++;
                  }
              }
          }
        Ke.cholesky_solve(Fe, Uedge);

        // Transfer new edge solutions to element
        for (unsigned int i=0; i != free_dofs; ++i)
          {
            Number &ui = Ue(new_side_dofs[free_dof[i]]);
            assert(std::abs(ui) < TOLERANCE ||
                   std::abs(ui - Uedge(i)) < TOLERANCE);
            ui = Uedge(i);
            dof_is_fixed[new_side_dofs[free_dof[i]]] =
              true;
          }
      }
   
  // Project any side values (edges in 2D, faces in 3D)
  if (dim > 1 && cont != DISCONTINUOUS)
    for (unsigned int s=0; s != elem->n_sides(); ++s)
      {
        FEInterface::dofs_on_side(elem, dim, fe_type,
                                  s, new_side_dofs);

        // Some side dofs are on nodes/edges and already
        // fixed, others are free to calculate
        unsigned int free_dofs = 0;
        for (unsigned int i=0; i !=
             new_side_dofs.size(); ++i)
          if (!dof_is_fixed[new_side_dofs[i]])
            free_dof[free_dofs++] = i;
        Ke.resize (free_dofs, free_dofs); Ke.zero();
        Fe.resize (free_dofs); Fe.zero();
        // The new side coefficients
        DenseVector<Number> Uside(free_dofs);

        // Add projection terms from each child sharing
        // this side
        for (unsigned int c=0; c != elem->n_children();
             ++c)
          {
            if (!elem->is_child_on_side(c,s))
              continue;
            Elem *child = elem->child(c);
            dof_map.old_dof_indices (child,
              old_dof_indices, var);
            const unsigned int old_n_dofs = old_dof_indices.size();

            temp_fe_type = base_fe_type;
            temp_fe_type.order = 
              static_cast<Order>(temp_fe_type.order +
                                 child->p_level());

            FEInterface::dofs_on_side(child, dim,
              temp_fe_type, s, old_side_dofs);

            // Initialize both child and parent FE data
            // on the child's side
            fe->attach_quadrature_rule (qsiderule.get());
            fe->reinit (child, s);
            const unsigned int n_qp = qsiderule->n_points();

            FEInterface::inverse_map (dim, fe_type, elem,
                            xyz_values, coarse_qpoints);

            fe_coarse->reinit(elem, &coarse_qpoints);

            // Loop over the quadrature points
            for (unsigned int qp=0; qp<n_qp; qp++)
              {
                // solution value at the quadrature point
                Number fineval = libMesh::zero;
                // solution grad at the quadrature point
                Gradient finegrad;

                // Sum the solution values * the DOF
                // values at the quadrature point to
                // get the solution value and gradient.
                for (unsigned int i=0; i<old_n_dofs;
                     i++)
                  {
                    fineval +=
                      (old_vector(old_dof_indices[i])*
                      phi_values[i][qp]);
                    if (cont == C_ONE)
                      finegrad.add_scaled((*dphi_values)[i][qp],
                                          old_vector(old_dof_indices[i]));
                  }

                // Form side projection matrix
                for (unsigned int sidei=0, freei=0;
                     sidei != new_side_dofs.size();
                     ++sidei)
                  {
                    unsigned int i = new_side_dofs[sidei];
                    // fixed DoFs aren't test functions
                    if (dof_is_fixed[i])
                      continue;
                    for (unsigned int sidej=0, freej=0;
                         sidej != new_side_dofs.size();
                         ++sidej)
                      {
                        unsigned int j =
                          new_side_dofs[sidej];
                        if (dof_is_fixed[j])
                          Fe(freei) -=
                            phi_coarse[i][qp] *
                            phi_coarse[j][qp] * JxW[qp] *
                            Ue(j);
                        else
                          Ke(freei,freej) +=
                            phi_coarse[i][qp] *
                            phi_coarse[j][qp] * JxW[qp];
                        if (cont == C_ONE)
                          {
                            if (dof_is_fixed[j])
                              Fe(freei) -=
                                ((*dphi_coarse)[i][qp] *
                                 (*dphi_coarse)[j][qp]) *
                                JxW[qp] *
                                Ue(j);
                            else
                              Ke(freei,freej) +=
                                ((*dphi_coarse)[i][qp] *
                                 (*dphi_coarse)[j][qp])
                                * JxW[qp];
                          }
                        if (!dof_is_fixed[j])
                          freej++;
                      }
                    Fe(freei) += (fineval * phi_coarse[i][qp]) * JxW[qp];
                    if (cont == C_ONE)
                      Fe(freei) +=
                        (finegrad * (*dphi_coarse)[i][qp]) * JxW[qp];
                    freei++;
                  }
              }
          }
        Ke.cholesky_solve(Fe, Uside);

        // Transfer new side solutions to element
        for (unsigned int i=0; i != free_dofs; ++i)
          {
            Number &ui = Ue(new_side_dofs[free_dof[i]]);
            assert(std::abs(ui) < TOLERANCE ||
                   std::abs(ui - Uside(i)) < TOLERANCE);
            ui = Uside(i);
            dof_is_fixed[new_side_dofs[free_dof[i]]] =
              true;
          }
      }

  // Project the interior values, finally

  // Some interior dofs are on nodes/edges/sides and
  // already fixed, others are free to calculate
  unsigned int free_dofs = 0;
  for (unsigned int i=0; i != new_n_dofs; ++i)
    if (!dof_is_fixed[i])
      free_dof[free_dofs++] = i;
  Ke.resize (free_dofs, free_dofs); Ke.zero();
  Fe.resize (free_dofs); Fe.zero();
  // The new interior coefficients
  DenseVector<Number> Uint(free_dofs);

  // Add projection terms from each child
  for (unsigned int c=0; c != elem->n_children(); ++c)
    {
      Elem *child = elem->child(c);
      dof_map.old_dof_indices (child, old_dof_indices,
                               var);
      const unsigned int old_n_dofs = old_dof_indices.size();

      // Initialize both child and parent FE data
      // on the child's quadrature points
      fe->attach_quadrature_rule (qrule.get());
      fe->reinit (child);
      const unsigned int n_qp = qrule->n_points();

      FEInterface::inverse_map (dim, fe_type, elem,
        xyz_values, coarse_qpoints);

      fe_coarse->reinit(elem, &coarse_qpoints);

      // Loop over the quadrature points
      for (unsigned int qp=0; qp<n_qp; qp++)
        {
          // solution value at the quadrature point              
          Number fineval = libMesh::zero;
          // solution grad at the quadrature point              
          Gradient finegrad;

          // Sum the solution values * the DOF
          // values at the quadrature point to
          // get the solution value and gradient.
          for (unsigned int i=0; i<old_n_dofs; i++)
            {
              fineval +=
                (old_vector(old_dof_indices[i])*
                 phi_values[i][qp]);
              if (cont == C_ONE)
                finegrad.add_scaled((*dphi_values)[i][qp],
                                    old_vector(old_dof_indices[i]));
            }

          // Form interior projection matrix
          for (unsigned int i=0, freei=0;
               i != new_n_dofs; ++i)
            {
              // fixed DoFs aren't test functions
              if (dof_is_fixed[i])
                continue;
              for (unsigned int j=0, freej=0; j !=
                   new_n_dofs; ++j)
                {
                  if (dof_is_fixed[j])
                    Fe(freei) -=
                      phi_coarse[i][qp] *
                      phi_coarse[j][qp] * JxW[qp] *
                      Ue(j);
                  else
                    Ke(freei,freej) +=
                      phi_coarse[i][qp] *
                      phi_coarse[j][qp] * JxW[qp];
                  if (cont == C_ONE)
                    {
                      if (dof_is_fixed[j])
                        Fe(freei) -=
                          ((*dphi_coarse)[i][qp] *
                           (*dphi_coarse)[j][qp]) *
                          JxW[qp] * Ue(j);
                      else
                        Ke(freei,freej) +=
                          ((*dphi_coarse)[i][qp] *
                           (*dphi_coarse)[j][qp]) * JxW[qp];
                    }
                  if (!dof_is_fixed[j])
                    freej++;
                }
              Fe(freei) += phi_coarse[i][qp] * fineval *
                           JxW[qp];
              if (cont == C_ONE)
                Fe(freei) += (finegrad * (*dphi_coarse)[i][qp]) * JxW[qp];
              freei++;
            }
        }
    }
  Ke.cholesky_solve(Fe, Uint);

  // Transfer new interior solutions to element
  for (unsigned int i=0; i != free_dofs; ++i)
    {
      Number &ui = Ue(free_dof[i]);
      assert(std::abs(ui) < TOLERANCE ||
             std::abs(ui - Uint(i)) < TOLERANCE);
      ui = Uint(i);
      dof_is_fixed[free_dof[i]] = true;
    }

  // Make sure every DoF got reached!
  for (unsigned int i=0; i != new_n_dofs; ++i)
    assert(dof_is_fixed[i]);
}
