/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.editor.settings.storage;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.KeyStroke;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.MultiKeyBinding;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.Repository;
import org.openide.xml.XMLUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;


/**
 * This class contains support static methods for loading / saving and 
 * translating coloring (fontsColors.xml) files. It calls XMLStorage utilities.
 *
 * @author Jan Jancura
 */
public final class KeyMapsStorage {

    // -J-Dorg.netbeans.modules.editor.settings.storage.KeyMapsStorage.level=FINE
    private static final Logger LOG = Logger.getLogger(KeyMapsStorage.class.getName());

    private static final String ROOT = "bindings"; //NOI18N
    private static final String E_BIND = "bind"; //NOI18N
    private static final String A_ACTION_NAME = "actionName"; //NOI18N
    private static final String A_KEY = "key"; //NOI18N
    private static final String A_REMOVE = "remove"; //NOI18N
    private static final String V_TRUE = "true"; //NOI18N
        
    private static final String PUBLIC_ID = "-//NetBeans//DTD Editor KeyBindings settings 1.1//EN"; //NOI18N
    private static final String SYSTEM_ID = "http://www.netbeans.org/dtds/EditorKeyBindings-1_1.dtd"; //NOI18N

    private KeyMapsStorage() {
    }
    
    // load ....................................................................
    
    public static Map<Collection<KeyStroke>, MultiKeyBinding> loadKeyMaps (
        MimePath mimePath,
        String profile,
        boolean defaults
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        assert profile != null : "The parameter profile must not be null"; //NOI18N
        
        FileObject baseFolder = Repository.getDefault().getDefaultFileSystem().findResource("Editors"); //NOI18N
        Map<String, List<Object []>> files = new HashMap<String, List<Object []>>();
        SettingsType.KEYBINDINGS.getLocator().scan(baseFolder, mimePath.getPath(), profile, true, true, !defaults, files);
        
        assert files.size() <= 1 : "Too many results in the scan"; //NOI18N

        List<Object []> profileInfos = files.get(profile);
        if (profileInfos == null) {
            return Collections.<Collection<KeyStroke>, MultiKeyBinding>emptyMap();
        }
        
        Map<Collection<KeyStroke>, MultiKeyBinding> keybindingsMap = new HashMap<Collection<KeyStroke>, MultiKeyBinding>();
        for(Object [] info : profileInfos) {
            FileObject profileHome = (FileObject) info[0];
            FileObject settingFile = (FileObject) info[1];
            boolean modulesFile = ((Boolean) info[2]).booleanValue();

            // Load keybindings from the settingFile
            @SuppressWarnings("unchecked")
            Object [] loadedData = (Object []) XMLStorage.load(settingFile, new KeyMapsReader());
            @SuppressWarnings("unchecked")
            Map<Collection<KeyStroke>, MultiKeyBinding> addedBindings = (Map<Collection<KeyStroke>, MultiKeyBinding>) loadedData[0];
            @SuppressWarnings("unchecked")
            Collection<Collection<KeyStroke>> removedBindings = (Collection<Collection<KeyStroke>>) loadedData[1];

            if (LOG.isLoggable(Level.FINE)) {
                LOG.fine("Loading keybindings from: '" + settingFile.getPath() + "'");
            }

            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("--- Removing keybindings: " + removedBindings);
            }
            
            // First remove all keybindings marked as removed
            for(Collection<KeyStroke> binding : removedBindings) {
                keybindingsMap.remove(binding);
            }

            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("--- Adding keybindings: " + addedBindings);
            }
            
            // Then add all new bindings
            keybindingsMap.putAll(addedBindings);

            if (LOG.isLoggable(Level.FINEST)) {
                LOG.finest("-------------------------------------");
            }
        }
        
        return Collections.unmodifiableMap(keybindingsMap);
    }    
    
    private static class KeyMapsReader extends XMLStorage.Handler {
        private Map<Collection<KeyStroke>, MultiKeyBinding> keyMap = new HashMap<Collection<KeyStroke>, MultiKeyBinding>();
        private Collection<Collection<KeyStroke>> removedShortcuts = new HashSet<Collection<KeyStroke>>();
        
        @Override
        public Object getResult() {
            return new Object[] {keyMap, removedShortcuts};
        }
        
        @Override
        public void startElement(
            String uri,
            String localName,
            String name,
            Attributes attributes
        ) throws SAXException {
            try {
                if (name.equals(ROOT)) {
                    // We don't read anything from the root element
                    
                } else if (name.equals(E_BIND)) {
                    String key = attributes.getValue(A_KEY);
                    
                    if (isModuleFile() && isDefaultProfile() && key != null && key.length() > 0) {
                        // check the key, it should never start with 'A' or 'C', because
                        // these characters do not work on MAC, Alt should be coded as 'O'
                        // and Ctrl as 'D'
                        int idx = key.indexOf('-'); //NOI18N
                        if (idx != -1 && (key.charAt(0) == 'A' || key.charAt(0) == 'C')) { //NOI18N
                            LOG.warning("The keybinding '" + key + //NOI18N
                                "' in " + getProcessedFile().getPath() + " may not work correctly on Mac. " + //NOI18N
                                "Keybindings starting with Alt or Ctrl should " + //NOI18N
                                "be coded with latin capital letters 'O' " + //NOI18N
                                "or 'D' respectively. For details see org.openide.util.Utilities.stringToKey()."); //NOI18N
                        }
                    }
                    
                    KeyStroke[] shortcut = Utils.stringToKeyStrokes(key);
                    String remove = attributes.getValue(A_REMOVE);
                    
                    if (Boolean.valueOf(remove)) {
                        removedShortcuts.add(Arrays.asList(shortcut));
                    } else {
                        String actionName = attributes.getValue(A_ACTION_NAME);
                        if (actionName != null) {
                            MultiKeyBinding mkb = new MultiKeyBinding(shortcut, actionName);
                            keyMap.put(Arrays.asList(shortcut), mkb);
//                            System.out.println("!!! adding: '" + key + "' -> '" + actionName + "'");
                        } else {
                            LOG.warning("Ignoring keybinding '" + key + "' with no action name."); //NOI18N
                        }
                    }
                }
            } catch (Exception ex) {
                LOG.log(Level.WARNING, "Can't parse keybindings file " + getProcessedFile().getPath(), ex); //NOI18N
            }
        }
    } // End of KeyMapsReader class

    
    // delete ..........................................................
    
    public static void deleteProfile(
        MimePath mimePath,
        String profile,
        boolean defaults
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        assert profile != null : "The parameter profile must not be null"; //NOI18N
        
        FileSystem sfs = Repository.getDefault().getDefaultFileSystem();
        FileObject baseFolder = sfs.findResource("Editors"); //NOI18N
        Map<String, List<Object []>> files = new HashMap<String, List<Object []>>();
        SettingsType.KEYBINDINGS.getLocator().scan(baseFolder, mimePath.getPath(), profile, true, defaults, !defaults, files);
        
        assert files.size() <= 1 : "Too many results in the scan"; //NOI18N

        final List<Object []> profileInfos = files.get(profile);
        if (profileInfos != null) {
            try {
                sfs.runAtomicAction(new FileSystem.AtomicAction() {
                    public void run() {
                        for(Object [] info : profileInfos) {
                            FileObject settingFile = (FileObject) info[1];
                            try {
                                settingFile.delete();
                            } catch (IOException ioe) {
                                LOG.log(Level.WARNING, "Can't delete editor settings file " + settingFile.getPath(), ioe); //NOI18N
                            }
                        }
                    }
                });
            } catch (IOException ioe) {
                LOG.log(Level.WARNING, "Can't delete editor keybindings for " + mimePath.getPath() + ", " + profile, ioe); //NOI18N
            }
        }
    }
    
    // save ..........................................................
    
    public static void saveKeyMaps (
        MimePath mimePath, 
        String profile,
        boolean defaults,
        final Collection<MultiKeyBinding> keybindings,  // modified shortcuts
        final Set<Collection<KeyStroke>> removedKeybindings  // shortcuts
    ) {
        assert mimePath != null : "The parameter mimePath must not be null"; //NOI18N
        assert profile != null : "The parameter profile must not be null"; //NOI18N
        
        final FileSystem sfs = Repository.getDefault().getDefaultFileSystem();
        final String settingFileName = SettingsType.KEYBINDINGS.getLocator().getWritableFileName(
                mimePath.getPath(), profile, null, defaults);

        try {
            sfs.runAtomicAction(new FileSystem.AtomicAction() {
                public void run() throws IOException {
                    FileObject baseFolder = sfs.findResource("Editors"); //NOI18N
                    FileObject f = FileUtil.createData(baseFolder, settingFileName);
                    saveKeybindings(f, keybindings, removedKeybindings);
                }
            });
        } catch (IOException ioe) {
            LOG.log(Level.WARNING, "Can't save editor keybindings for " + mimePath.getPath() + ", " + profile, ioe); //NOI18N
        }
    }
    
    private static void saveKeybindings(
        FileObject fo,
        Collection<MultiKeyBinding> keyMap,
        Set<Collection<KeyStroke>> removed
        ) {
        Document doc = XMLUtil.createDocument(ROOT, null, PUBLIC_ID, SYSTEM_ID);
        Node root = doc.getElementsByTagName(ROOT).item(0);
        
        for(MultiKeyBinding mkb : keyMap) {
            Element bind = doc.createElement(E_BIND);
            root.appendChild(bind);
            
            bind.setAttribute(A_ACTION_NAME, mkb.getActionName());
            bind.setAttribute(A_KEY, Utils.keyStrokesToString(mkb.getKeyStrokeList()));
        }
        
        for(Collection<KeyStroke> keyStrokes : removed) {
            String shortcut = Utils.keyStrokesToString(keyStrokes);
            Element bind = doc.createElement(E_BIND);
            root.appendChild(bind);
            
            bind.setAttribute(A_KEY, shortcut);
            bind.setAttribute(A_REMOVE, V_TRUE);
        }
        
        XMLStorage.save(fo, doc);
    }
}
