/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.masterfs.filebasedfs.fileobjects;

import java.io.FileNotFoundException;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.Iterator;
import org.netbeans.modules.masterfs.filebasedfs.utils.FSException;
import org.openide.filesystems.FileLock;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.ref.Reference;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.WeakHashMap;
import org.openide.filesystems.FileAlreadyLockedException;
import org.openide.util.Exceptions;
import org.openide.util.NbBundle;


/**
 * This implementation doesn' allow to get two or more FileLocks for
 * one identical file. This means that in editor there is possible to open two
 * editor tabs but there isn't possible to modify them at the same time.
 *
 * This implemenation creates new special locking file. This file exists on disk as long
 * as the FileLock isValid.  This locking files are
 * hidden by FileBasedFileSystem. Moreover this file naming
 * convention corresponds with defaul regular expression for hiding this files by VisibilityQuery
 * (just in case that these files wont be hidden FileBasedFileSystem).<P>
 * <p/>
 * There might happen that these locking files won't be deleted because of
 * crash of JVM, not released lock and so one. But these corrupted locks must
 * be recognized and can't prevent from providing next locks.<P>
 * <p/>
 * This implementation isn't responsible:
 * - for preventing from opening two or more tabs in editor for one file on disk.
 * - for silent refusing of editing already locked file<P>
 *
 * @author Radek Matous
 */


public class WriteLock extends FileLock {
    static final Map allLocks = Collections.synchronizedMap(new WeakHashMap());
    private File lockFile;
    
    static {        
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                Set s = allLocks.keySet();
                synchronized(allLocks) {
                    for (Iterator it = s.iterator(); it.hasNext();) {
                        File f = (File) it.next();
                        if (f.exists() && WriteLockUtils.hasActiveLockFileSigns(f.getAbsolutePath())) {
                            f.delete();
                        }                    
                    }
                }
            }            
        });
    }
    
    static WriteLock tryLock(final File file) throws IOException  {
        return WriteLock.getInstance(file, false);
    }
    
    static WriteLock tryNioLock(final File file) throws IOException  {
        return WriteLock.getInstance(file, true);
    }

    static WriteLock tryLightWeightLock(final File file) throws IOException  {
        return WriteLock.getLightWeightInstance(file);
    }

    public static void relock(final File theOld, File theNew) {
        List l = new ArrayList();
        Set s = allLocks.keySet();
        synchronized(allLocks) {
            for (Iterator it = s.iterator(); it.hasNext();) {
                File f = (File) it.next();
                File nLockFile = null;
                String relPath = getRelativePath(theOld, f);
                if (relPath != null) {
                    nLockFile = new File(theNew, relPath);
                } else if (f.equals(WriteLockUtils.getAssociatedLockFile(theOld))) {                
                    nLockFile = WriteLockUtils.getAssociatedLockFile(theNew);
                    if (f.exists()) {
                        f.renameTo(nLockFile);
                    }
                }
                
                if (nLockFile != null) {
                    Reference r = (Reference)allLocks.get(f);
                    WriteLock wl = (WriteLock)r.get();
                    if (wl != null) {
                        it.remove();
                        wl.lockFile = nLockFile;
                        l.add(wl);
                    }
                    
                }
            }
        }
        for (Iterator it2 = l.iterator(); it2.hasNext();) {
            WriteLock wl = (WriteLock) it2.next();
            allLocks.put(wl.lockFile, new WeakReference(wl));
        }
    }
    
    private static String getRelativePath(final File dir, final File file) {
        Stack stack = new Stack ();                
        File tempFile = file;
        while(tempFile != null && !tempFile.equals(dir)) {
            stack.push (tempFile.getName());
            tempFile = tempFile.getParentFile();
        }
        if (tempFile == null) return null;
        StringBuilder retval = new StringBuilder();
        while (!stack.isEmpty()) {
            retval.append((String)stack.pop());
            if (!stack.isEmpty()) {
                retval.append("/");//NOI18N
            }
        }                        
        return retval.toString();
    }
    
    
    private static WriteLock getLightWeightInstance(final File file)  throws IOException {
        File lckFile = WriteLockUtils.getAssociatedLockFile(file);
        boolean isAlreadyLocked = WriteLock.allLocks.keySet().contains(lckFile);
        if (isAlreadyLocked) {
            throw new FileAlreadyLockedException(NbBundle.getMessage(
                WriteLock.class, "EXC_FileAlreadyLocked", file.getAbsolutePath()));    
        }
        
        return new WriteLock(lckFile);
    }
    
    private static WriteLock getInstance(final File file, boolean nioLock)  throws IOException {
        boolean isCreated = false;
        IOException toFire = null;
        
        File lckFile = WriteLockUtils.getAssociatedLockFile(file);
        if (!lckFile.exists()) {
            isCreated = true;
        }
        RandomAccessFile raf = getAccessToFile(lckFile);//NOI18N
        assert lckFile.exists();
        
        FileChannel channel = raf.getChannel();
        
        
        if (channel != null && channel.isOpen()) {
            try {
                String content = WriteLockUtils.getContentOfLckFile(lckFile, channel);
                if (content == null && (isCreated || lckFile.length() == 0)) {
                    assert lckFile.length() == 0;
                    content = WriteLockUtils.writeContentOfLckFile(lckFile, channel);
                }
                
                if (content != null && !WriteLock.allLocks.keySet().contains(new File(content))) {
                    if (channel != null && channel.isOpen() && !nioLock) {
                        channel.close();
                    }
                    return (nioLock) ? new NioLock(lckFile, channel, channel.tryLock()) : new WriteLock(lckFile);
                }
            } catch (IOException iex) {
                toFire =  iex;
            }
        }
        
        if (channel != null && channel.isOpen()) {
            channel.close();
        }
        
        if (isCreated && lckFile.exists()) {
            lckFile.delete();
        }
        
        if (toFire == null) {
            if (lckFile.exists()) {
                toFire = new FileAlreadyLockedException(NbBundle.getMessage(
                    WriteLock.class, "EXC_FileLockAlreadyExists", file.getAbsolutePath(), lckFile.getAbsolutePath()));                    
            } else {
                toFire = new FileAlreadyLockedException(NbBundle.getMessage(
                    WriteLock.class, "EXC_FileAlreadyLocked", file.getAbsolutePath()));    
            }
        }
        
        FSException.annotateException(toFire);
        throw toFire;
    }

    private static RandomAccessFile getAccessToFile(final File f) throws FileNotFoundException {
        RandomAccessFile raf = null;
        try {
            raf = new RandomAccessFile(f, "rwd");//NOI18N
        } catch (FileNotFoundException e) {
            FileNotFoundException fex = e;
            if (!f.exists()) {
                fex = (FileNotFoundException)new FileNotFoundException(e.getLocalizedMessage()).initCause(e);
            } else if (!f.canRead()) {
                fex = (FileNotFoundException)new FileNotFoundException(e.getLocalizedMessage()).initCause(e);
            } else if (!f.canWrite()) {
                fex = (FileNotFoundException)new FileNotFoundException(e.getLocalizedMessage()).initCause(e);
            } else if (f.getParentFile() == null) {
                fex = (FileNotFoundException)new FileNotFoundException(e.getLocalizedMessage()).initCause(e);
            } else if (!f.getParentFile().exists()) {
                fex = (FileNotFoundException)new FileNotFoundException(e.getLocalizedMessage()).initCause(e);
            } 
            throw fex;
        }
        return raf;
    }
    
    
    private WriteLock(File lockFile) {
        this.lockFile = lockFile;
        register();
    }
    
    public boolean isValid(final File f) {
        boolean retVal = isValid();
        
        if (retVal) {
            final File associatedLockFile = WriteLockUtils.getAssociatedLockFile(f);
            retVal = lockFile.equals(associatedLockFile);
        } else {
        }
        return retVal;
    }
    
    public void releaseLock() {
        synchronized (WriteLockFactory.class) {
            if (isValid()) {
                deregister();
                super.releaseLock();
            }
            if (getLockFile().exists()) {
                final boolean isDeleted = getLockFile().delete();
                //assert isDeleted : getLockFile().getAbsolutePath();
            }            
        }
    }
    
    private final void deregister() {
        allLocks.remove(getLockFile());
    }
    
    private final void register() {
        allLocks.put(this.getLockFile(), new WeakReference(this));
    }
    
    final File getLockFile() {
        return lockFile;
    }
    
    public final String toString() {
        final String name = getLockFile().getName();
        final String newName = name.substring(WriteLockUtils.PREFIX.length(), (name.length() - WriteLockUtils.SUFFIX.length()));
        return new File(getLockFile().getParentFile(), newName).getAbsolutePath();
    }
    
    private static class NioLock extends WriteLock {
        private FileChannel channel;
        private java.nio.channels.FileLock nioLock;
        
        NioLock(File lockFile, FileChannel channel, java.nio.channels.FileLock nioLock) {
            super(lockFile);
            
            assert nioLock != null;
            assert nioLock.isValid();
            assert channel != null;
            
            this.channel = channel;
            this.nioLock = nioLock;
        }
        
        public void releaseLock() {
            synchronized (WriteLockFactory.class) {
                try {
                    if (nioLock.isValid()) {
                        nioLock.release();
                    }
                    
                    if (channel.isOpen()) {
                        channel.close();
                    }
                } catch (java.io.IOException e) {
                    Exceptions.printStackTrace(e);
                }
                
                super.releaseLock();
            }
        }
    }
}
