/*
 * 123 Session Module
 * (c) Kai Voigt, k@123.org
 *
 * BETA release, this module only has been used and slighlty
 * tested on _one_ machine by _one_ person.  But it looks
 * quite stable :-)
 *
 * This module gives each session a seperate set of variables
 * which will be accessible in later connections from the same
 * browser.  Additionally, it stores user variables if a user
 * has authenticated.
 *
 * Install the module, set the Secret Word, adjust Expiration Time
 * and Garbage Collection Frequency.  Specify an optional database.
 *
 * In each document, the following variables are then available:
 * id->variables->SessionID         - the current Session ID
 * id->variables->UserID            - the User ID, if present
 * id->variables->session_variables - all session variables
 * id->variables->user_variables    - all user variables, if present
 *
 * Read the full documentation at
 * http://123.org/technik/webserver/session/
 *
 * Send questions, donations, feature requests, bug reports, etc. to
 * me.  Please let me know if and where you are using this module.
 */

/*
 * TODO:
 *
 * - putting the session id optionally into the URL to let a
 *   user access the application more than once at the same time.
 * - error handling in SQL stuff.
 * - checking for race conditions.
 * - cleanup of retrieve() and store() functions, they look ugly.
 * - make handling of user and session variables more intelligent,
 *   i.e. not using seperate code.
 */

string cvs_version = "$Id: 123session.pike,v 0.1 1999/12/04 12:36:20 k Exp k $";

#if efun(thread_create)
object global_lock = Thread.Mutex();
#endif

#include <module.h>
inherit "module";
inherit "roxenlib";
import Sql;

/* our current virtual server, will be initialized
   in the first_try() routine */
object current_configuration;

void create() {
  defvar("secret", "ChAnGeThIs", "Secret Word", TYPE_STRING, 
	 "a secret word that is needed to create secure IDs." );
  defvar("garbage", 10, "Garbage Collection Frequency", TYPE_INT, 
	 "after how many connects expiration of old session should happen" );
  defvar("expire", 60, "Expiration Time", TYPE_INT, 
	 "after how many seconds an unactive session is removed" );
  defvar("database", "", "Storage Database", TYPE_STRING, 
	 "which database to use for the session and user variables, syntax is "
   "<pre>[<i>sqlserver</i>://][[<i>user</i>][:<i>password</i>]@]"
   "[<i>host</i>[:<i>port</i>]]/<i>database</i></pre>"
   "The string fields <i>region</i>, <i>name</i>, <i>id</i> and "
   "the mediumtext field <i>content</i> must be present in the "
   "table <i>variables</i>. "
   "If no database is specified, Roxen's internal storage will "
   "be used." );
}

mixed register_module() {
	return ({ MODULE_PARSER | MODULE_FIRST | MODULE_FILTER, 
		"123 Sessions", 
		"We'll do real session variables."
		"<p>"  
		"Warning: This module has not been tested a lot."  
		"<br>"  
		"Read the module code for instructions or refer to the "
    "full documentation at "
		"<a href=\"http://123.org/technik/webserver/session/\">"
		"http://123.org/technik/webserver/session/</a>",
		({}), 1, });
}

// function prototypes
string tag_fetch_user_vars(string tag_name, mapping args,
	object request_id, mapping defines);
string tag_user_var(string tag_name, mapping args,
	object request_id, mapping defines);
string tag_session_var(string tag_name, mapping args,
	object request_id, mapping defines);
void fetch_user_vars(object id, string UserID);

// _encode_value() and _decode_value() make use of the roxen
// [en|de]coding functions, but escapes " characters.  This
// is needed for SQL queries.

string _encode_value(mixed value) {
	return (replace(encode_value(value), "'", "\\'"));
}
mixed _decode_value(string coded_value) {
	return (decode_value(replace(coded_value, "\\'", "'")));
}

// retrieve() and store() are abstracted here and can either be
// the internal roxen procedures or SQL calls.

mapping retrieve_all(string region) {
	return roxen->retrieve(region, current_configuration);
}
void store_all(string region, mapping content) {
	roxen->store(region, content, 1, current_configuration);
}

mapping retrieve(string region, string ID) {
	if (query("database") != "") {
		object(sql) con;
		function sql_connect = current_configuration->sql_connect;
		con = sql_connect(query("database"));
		array(mapping(string:mixed)) result = con->query("select content from variables where region='"+region+"' and name='"+current_configuration->name+"' and id='"+ID+"'");
		if (sizeof(result) != 0) {
			string content = result[0]->content;
			return(_decode_value(content));
		} else {
			return ([]);
		}
	} else {
  	mapping v = retrieve_all(region);
  	if (!mappingp(v[ID])) {
    	return ([]);
  	}
  	return (v[ID]);
	}
}

void store(string region, string ID, mapping content) {
	if (query("database") != "") {
		object(sql) con;
		function sql_connect = current_configuration->sql_connect;
		con = sql_connect(query("database"));
		con->query("delete from variables where region='"+region+"' and name='"+current_configuration->name+"' and id='"+ID+"'");
		con->query("insert into variables (region, name, id, content) values ('"+region+"', '"+current_configuration->name+"', '"+ID+"', '"+_encode_value(content)+"')");
	} else {
  	mapping v = retrieve_all(region);     // This should
  	v[ID] = content;                      // be an
  	store_all(region, v);                 // atomic action.
	}
}

// Some status output for the Configuration Interface.

string status() {
  string result = "";
  mapping v;

  v = retrieve_all("Session_Vars");
  result += sprintf("%d sessions active.<br>\n", sizeof(v));
  v = retrieve_all("User_Vars");
  result += sprintf("%d users active.<br>\n", sizeof(v));
  return (result);
}

// RXML tags to ease access to the user and session variables.

mapping query_tag_callers() {
	return ([ "fetch_user_vars" : tag_fetch_user_vars,
	          "user_var" : tag_user_var,
	          "session_var" : tag_session_var ]);
}

void fetch_user_vars(object id, string UserID) {
	id->variables->session_variables->UserID = UserID;
	id->variables->user_variables = retrieve("User_Vars", UserID);
}

string tag_fetch_user_vars(string tag_name, mapping args, object id, mapping defines) {
	if (!args->userid) {
		return ("No UserID<br>\n");
	}
	fetch_user_vars(id, args->userid);
	return "";
}

// <tag_user_var> can read or set user variables.

string tag_user_var(string tag_name, mapping args, object id, mapping defines) {
	if (!args->name) {
		return "";
	}
	if (args->value) {
		id->variables->user_variables[args->name] = args->value;
		return "";
	} else {
		if (id->variables->user_variables[args->name]) {
			return (string)(id->variables->user_variables[args->name]);
		} else {
			return "";
		}
	}
}

// <tag_session_var> can read or set session variables.

string tag_session_var(string tag_name, mapping args, object id, mapping defines) {
	if (!args->name) {
		return "";
	}
	if (args->value) {
		id->variables->session_variables[args->name] = args->value;
		return "";
	} else {
		if (id->variables->session_variables[args->name]) { 
			return (string)(id->variables->session_variables[args->name]);
		} else {
			return "";
		}
	}
}

// From time to time, we delete expired sessions.  Every session that
// is older than the configured expiration time is removed from the
// storage.  To avoid race conditions, this code is within a lock.

void do_garbage_collection() {

#if efun(thread_create)   
	object lock = global_lock->lock();
#endif
  
	if (query("database") != "") {
		object(sql) con;
		function sql_connect = current_configuration->sql_connect;
		con = sql_connect(query("database"));
		array(mapping(string:mixed)) result = con->query("select * from variables;");
		for (int i=0; i<sizeof(result); i++) {
			mapping content = _decode_value(result[i]->content);
			if (time() > content->lastusage+query("expire")) {
				con->query("delete from variables where region='"+result[i]->region+"' and name='"+current_configuration->name+"' and id='"+result[i]->id+"'");
			}
		}
	} else {
		mapping v = retrieve_all("Session_Vars");
		foreach (indices(v), string SessionID) {
			if (!v[SessionID]->lastusage) {
				m_delete(v, SessionID);
			} else {
				if (time() > (v[SessionID]->lastusage+query("expire"))) {
					m_delete(v, SessionID);
				}
			}
		}
		store_all("Session_Vars", v);
	
#if efun(thread_create)
		destruct(lock);
#endif 

	}
}

// Each browser is identified by a secure Session ID that is stored
// in the browser's cookie area.

mixed first_try(object id) {
	current_configuration = id->conf;
  string SessionID;

  // We want to do a garbage collection after some hits.  Since we don't
  // carry a counter with us, we do some trick with a random number.
  if (random(query("garbage")) == 0) {
    do_garbage_collection();
  }

  if (id->cookies->SessionID) {
    // This browser had been here before, so we take the Session ID
    // from the cookie it sent us.
    SessionID = id->cookies->SessionID;
  }
  if (!SessionID) {
    // Other browsers need a Session ID.  We compute a secure ID
    // from some unique data and set a cookie in the browser.
    object md5 = Crypto.md5();
    md5->update(query("secret"));
    md5->update(sprintf("%d", roxen->increase_id()));
    md5->update(sprintf("%d", time(1)));
    SessionID = Crypto.string_to_hex(md5->digest());
    string Cookie = "SessionID="+SessionID+"; path=/";
    id->cookies->SessionID = SessionID;
    id->misc->moreheads = ([ "Set-Cookie": Cookie,
                             "Expires": "Mon, 26 Jul 1997 05:00:00 GMT",
                             "Pragma": "no-cache",
                             "Last-Modified": http_date(time(1)),
                             "Cache-Control": "no-cache, must-revalidate" ]);
  }

  // Now we fetch the stored variables for this session.  Those might
  // be an empty set.
  id->variables->session_variables = retrieve("Session_Vars", SessionID);
  id->variables->session_variables->lastusage = time();

  // Now let's make the session ID and the variables accessible for the
  // applications.
  id->variables->SessionID = SessionID;

  // If a UserID is known for this session, we fetch the
  // User Variables and make them available.
  if (id->variables->session_variables->UserID) {
    fetch_user_vars(id, id->variables->session_variables->UserID);
  }
}

// At the end of the document, we have to save changed variables
// back into the storage.  Actually, we're storing all variables,
// this could be made more intelligent.

void filter(mapping m, object id) {

  // We again fetch the variables and put them back with the
  // possibly changed variables of the current session.
  id->variables->session_variables->lastusage = time();
  store("Session_Vars", id->variables->SessionID,
                     id->variables->session_variables);

  // same for the user variables, if they exist.
  if (id->variables->session_variables->UserID) {
    store("User_Vars", id->variables->session_variables->UserID,
                       id->variables->user_variables);
  }
} 
