#include <module.h>
inherit "module";
inherit "roxenlib";

#include <process.h>

//#define WATCHDOG_DEBUG

/*
 * Watchdog module for Roxen/1.2alpha and up.
 * This module is meant to "shut off" (in a similar way to the switch module)
 * parts of the server's virtual filesystem if the server is experiencing
 * overload conditions.
 * Currently to determine the overload status the server uses the
 * system's load average via the "uptime" command. If it exceeds a threshold,
 * the request is denied with a "server overloaded" HTTP error message.
 * Future versions could assume the role of a rough traffic shaper too.
 * The module is also fairly simple, and can be easily customized to fit
 * particular purposes.
 *
 * (C) 1997 Francesco Chemolli <kinkie@comedia.it>
 * This code can be used, modified and redistributed freely under the terms
 * of the GNU General Public License version 2.
 * This code comes on a AS-IS basis, with NO WARRANTY OF ANY KIND, either
 * implicit or explicit. Use at your own risk.
 *
 */

#ifdef WATCHDOG_DEBUG
#define LOG(X) perror("Watchdog: "+X+"\n");
#else
#define LOG(X) /**/
#endif

mixed loadavg_call_out_id;
float loadavg;

void update_load_avg () {
	if (loadavg_call_out_id) {
		remove_call_out(loadavg_call_out_id);
	}
	loadavg_call_out_id=0;
	if (!QUERY(watchdog_enable))
		return;
	loadavg_call_out_id=call_out(update_load_avg,QUERY(loadavg_update_timer));
	sscanf(popen("uptime"),"%*sload average: %f%*s",loadavg);
	LOG("updated loadaverage to "+loadavg);
}

mapping first_try (object id) {
	if (!QUERY(watchdog_enable))
		return 0;
	if (loadavg < QUERY(loadavg_threshold)) {
		LOG("okay");
		return 0;
	}
	mixed tmp;
	foreach(QUERY(hosts_to_pass),tmp)
		if (glob(tmp,id->remoteaddr)) {
			LOG (sprintf ("Allowing: host %s matches glob %s",id->remoteaddr, tmp));
			return 0;
		}
	foreach(QUERY(docs_to_pass),tmp)
		if (glob(tmp,id->not_query)) {
			LOG (sprintf("Allowing: document %s matches glob %s",id->not_query,tmp));
			return 0;
		}
	LOG("Watchdog denied access!");
	return http_low_answer(502,parse_rxml(QUERY(overload_message),id));
}

int watchdog_enabled() {
	return !QUERY(watchdog_enable);
}

void create() {
	defvar ("watchdog_enable",0,"Watchdog Enabled",TYPE_FLAG,
			"If this is defined, Roxen will check the server's load average "
			"and issue a \"server overloaded\" response if a configurable "
			"value is exceeded."
			);
	defvar ("loadavg_update_timer",30,"Load average check interval", TYPE_INT,
			"How often the load average is checked. If it's too often, you'll waste "
			"resources (each check implies a fork() and some milliseconds of server "
			"block). If it's too seldom, you won't get an appropriate response time "
			"to server overload.\n",
			0,watchdog_enabled
			); //controllare i valori cambiati!
	defvar ("loadavg_threshold",3.0,"Load average threshold",TYPE_FLOAT,
			"If this load average is exceeded, the server will decide it's "
			"overloaded. ", 0, watchdog_enabled
			);
	defvar ("overload_message",
			"<title>Sorry, service is temporarily unavailible</title>"
			"\n<h2 align=center><configimage src=roxen.gif "
			"alt=\"Service unavailible\">\n"
			"<p><hr noshade>"
			"\n<i>Sorry</i></h2>\n"
			"<br clear>\n<font size=+2>This server is temporarily "
			"disabled due to overload."
			"Please try again later.<p>\n</font>\n"
			"<hr noshade>"
			"<version>\n",
			"Server overload message",
			TYPE_TEXT_FIELD,
			"This is the message which will be used to notify an user the server's "
			"overload.",
			0, watchdog_enabled
			);
//	defvar ("preparse_messages",0,"Pre-RXML-parse the sent messages", TYPE_FLAG,
//			"If set, the response messages will be RXML-parsed when set and NOT "
//			"when sent. This will make request handling faster.",
//			0, watchdog_enabled
//			);
	defvar ("hosts_to_pass",({""}),"IP addresses to allow "
			"anyways (globs)",
			TYPE_STRING_LIST,
			"The hosts which match any glob of these will be allowed access even if "
			"the the server is overloaded",
			0, watchdog_enabled
			);
	defvar ("docs_to_pass", ({"/internal*"}),
			"virtual files to allow anyways (globs)",
			TYPE_STRING_LIST,
			"The (virtual) documents which match any of these will be allowed access "
			"even if the server is overloaded.",
			0, watchdog_enabled
			);
	update_load_avg();
}

void start() {
	update_load_avg();
}

void stop() {
	if (loadavg_call_out_id)
		remove_call_out(loadavg_call_out_id);
}

string|void check_variables(string variable,mixed will_be_set_to) {
	update_load_avg();
}

array register_module() {
	return ({
			MODULE_FIRST,
			"Watchdog",
			"This module acts as a watchdog, and returns a "
			"\"service temporarily unavailible\" HTTP answer if the server is "
			"overloaded. The overload conditions can be determined in different "
			"ways. Currently only a check on the computer's load average is "
			"implemented.",
			0,
			0
			});
}
