/*
 * libSpiff - XSPF playlist handling library
 *
 * Copyright (C) 2007, Sebastian Pipping / Xiph.Org Foundation
 * All rights reserved.
 *
 * Redistribution  and use in source and binary forms, with or without
 * modification,  are permitted provided that the following conditions
 * are met:
 *
 *     * Redistributions   of  source  code  must  retain  the   above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer.
 *
 *     * Redistributions  in  binary  form must  reproduce  the  above
 *       copyright  notice, this list of conditions and the  following
 *       disclaimer   in  the  documentation  and/or  other  materials
 *       provided with the distribution.
 *
 *     * Neither  the name of the Xiph.Org Foundation nor the names of
 *       its  contributors may be used to endorse or promote  products
 *       derived  from  this software without specific  prior  written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT  NOT
 * LIMITED  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS
 * FOR  A  PARTICULAR  PURPOSE ARE DISCLAIMED. IN NO EVENT  SHALL  THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL,    SPECIAL,   EXEMPLARY,   OR   CONSEQUENTIAL   DAMAGES
 * (INCLUDING,  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES;  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT  LIABILITY,  OR  TORT (INCLUDING  NEGLIGENCE  OR  OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Sebastian Pipping, sping@xiph.org
 */

/**
 * @file SpiffWriter.h
 * Interface of SpiffWriter.
 */

#ifndef SPIFF_WRITER_H
#define SPIFF_WRITER_H

#include "SpiffDefines.h"
#include <sstream>

namespace Spiff {



class SpiffXmlFormatter;
class SpiffTrackWriter;
class SpiffPropsWriter;
class SpiffWriterPrivate;



/**
 * Specifies the result of a write operation.
 */
enum SpiffWriterReturnCode {
	SPIFF_WRITER_SUCCESS, ///< Everything fine

	SPIFF_WRITER_ERROR_OPENING ///< File could not be opened
};



/**
 * Writes a playlist as XSPF to a file.
 * XSPF version 0 and 1 are supported.
 */
class SpiffWriter {

private:
	/// @cond DOXYGEN_NON_API
	SpiffWriterPrivate * const d; ///< D pointer
	/// @endcond

public:
	/**
	 * Creates a new playlist writer.
	 *
	 * @param version		XSPF version, must be 0 or 1
	 * @param formatter		XML formatter to use
	 * @param propsWriter	Playlist properties wrapped in a writer
	 */
	SpiffWriter(int version, SpiffXmlFormatter & formatter,
			SpiffPropsWriter & propsWriter);

	/**
	 * Copy constructor.
	 *
	 * @param source  Source to copy from
	 */
	SpiffWriter(const SpiffWriter & source);

	/**
	 * Assignment operator.
	 *
	 * @param source  Source to copy from
	 */
	SpiffWriter & operator=(const SpiffWriter & source);

	/**
	 * Frees all own memory.
	 */
	~SpiffWriter();

	/**
	 * Pre-registers a namespace so it can still
	 * appear in the root element.
	 *
	 * @param uri				Namespace URI
	 * @param prefixSuggestion	Suggested prefix
	 */
	void registerNamespace(const XML_Char * uri,
			const XML_Char * prefixSuggestion);

	/**
	 * Appends an entry to the playlist.
	 *
	 * @param trackWriter	Track to append wrapped in a writer
	 */
	void addTrack(SpiffTrackWriter & trackWriter);

	/**
	 * Finalizes the playlist and writes it to a file.
	 * You can call this method several times to write
	 * the same playlist to several files but you cannot
	 * add new tracks anymore. Call reset() to start over.
	 *
	 * @deprecated		Will be removed with 0.8.x, use writeFile instead
	 * @param filename	Filename of the file to write to
	 * @return			Error code
	 */
	int write(const XML_Char * filename);

	/**
	 * Finalizes the playlist and writes it to a file.
	 * You can call this method several times to write
	 * the same playlist to several files but you cannot
	 * add new tracks anymore. Call reset() to start over.
	 *
	 * @param filename	Filename of the file to write to
	 * @return			Error code
	 */
	int writeFile(const XML_Char * filename);

	/**
	 * Finalizes the playlist and writes it to a file.
	 * You can call this method several times to write
	 * the same playlist to several files but you cannot
	 * add new tracks anymore. Call reset() to start over.
	 *
	 * @param memory	Reference to output memory block, delete on your own
	 * @param numBytes	Size of the memory block in bytes
	 * @return			Error code
	 */
	int writeMemory(char * & memory, int & numBytes);

	/**
	 * Clears all previously added tracks and makes the writer
	 * reusable by another playlist.
	 *
	 * @param version		XSPF version, must be 0 or 1
	 * @param formatter		XML formatter to use
	 * @param propsWriter	Playlist properties wrapped in a writer
	 */
	void reset(int version, SpiffXmlFormatter & formatter, SpiffPropsWriter & propsWriter);

private:
	/**
	 * Does work common to all writing modes.
	 */
	void onBeforeWrite();

};



}


#endif // SPIFF_WRITER_H
