/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "syncml.h"

#include "syncml_internals.h"
#include "sml_devinf_internals.h"
#include "sml_command_internals.h"
#include "sml_elements_internals.h"
#include "sml_parse_internals.h"
#include "parser/sml_xml_assm.h"
#include "parser/sml_xml_parse.h"

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

SmlDevInfDevTyp smlDevInfDevTypeFromString(const char *name, SmlError **error)
{
	if (!strcmp(name, SML_ELEMENT_DEVTYP_PAGER)) {
		return SML_DEVINF_DEVTYPE_PAGER;
	} else if (!strcmp(name, SML_ELEMENT_DEVTYP_HANDHELD)) {
		return SML_DEVINF_DEVTYPE_HANDHELD;
	} else if (!strcmp(name, SML_ELEMENT_DEVTYP_PDA)) {
		return SML_DEVINF_DEVTYPE_PDA;
	} else if (!strcmp(name, SML_ELEMENT_DEVTYP_PHONE)) {
		return SML_DEVINF_DEVTYPE_PHONE;
	} else if (!strcmp(name, SML_ELEMENT_DEVTYP_SMARTPHONE)) {
		return SML_DEVINF_DEVTYPE_SMARTPHONE;
	} else if (!strcmp(name, SML_ELEMENT_DEVTYP_SERVER)) {
		return SML_DEVINF_DEVTYPE_SERVER;
	} else if (!strcmp(name, SML_ELEMENT_DEVTYP_WORKSTATION)) {
		return SML_DEVINF_DEVTYPE_WORKSTATION;
	}
	
	smlErrorSet(error, SML_ERROR_GENERIC, "Unknown devinf type name \"%s\"", name);
	return SML_DEVINF_DEVTYPE_UNKNOWN;
}

const char *smlDevInfDevTypeToString(SmlDevInfDevTyp type, SmlError **error)
{
	switch (type) {
		case SML_DEVINF_DEVTYPE_PAGER:
			return SML_ELEMENT_DEVTYP_PAGER;
		case SML_DEVINF_DEVTYPE_HANDHELD:
			return SML_ELEMENT_DEVTYP_HANDHELD;
		case SML_DEVINF_DEVTYPE_PDA:
			return SML_ELEMENT_DEVTYP_PDA;
		case SML_DEVINF_DEVTYPE_PHONE:
			return SML_ELEMENT_DEVTYP_PHONE;
		case SML_DEVINF_DEVTYPE_SMARTPHONE:
			return SML_ELEMENT_DEVTYP_SMARTPHONE;
		case SML_DEVINF_DEVTYPE_SERVER:
			return SML_ELEMENT_DEVTYP_SERVER;
		case SML_DEVINF_DEVTYPE_WORKSTATION:
			return SML_ELEMENT_DEVTYP_WORKSTATION;
		default:
			;
	}
		
	smlErrorSet(error, SML_ERROR_GENERIC, "Unknown devinf type \"%i\"", type);
	return NULL;
}

SmlDevInf *smlDevInfNew(const char *devid, SmlDevInfDevTyp devtyp, SmlError **error)
{
	smlAssert(devid);
	smlTrace(TRACE_ENTRY, "%s(%s, %i, %p)", __func__, devid, devtyp, error);
	
	SmlDevInf *devinf = smlTryMalloc0(sizeof(SmlDevInf), error);
	if (!devinf)
		goto error;
	
	devinf->devid = g_strdup(devid);
	devinf->devtyp = devtyp;
	devinf->refCount = 1;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, devinf);
	return devinf;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}

SmlDevInf *smlDevInfRef(SmlDevInf *devinf)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, devinf);
	smlAssert(devinf);
	
	g_atomic_int_inc(&(devinf->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, devinf->refCount);
	return devinf;
}

void smlDevInfUnref(SmlDevInf *devinf)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, devinf);
	smlAssert(devinf);
	
	if (g_atomic_int_dec_and_test(&(devinf->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		g_free(devinf->manufacturer);
		g_free(devinf->model);
		g_free(devinf->oem);
		g_free(devinf->softwareVersion);
		g_free(devinf->hardwareVersion);
		g_free(devinf->firmwareVersion);
		g_free(devinf->devid);
		
		GList *d = NULL;
		for (d = devinf->datastores; d; d = d->next) {
			SmlDevInfDataStore *store = d->data;
			smlDevInfDataStoreUnref(store);
		}
		g_list_free(devinf->datastores);
		
		for (d = devinf->ctcaps; d; d = d->next) {
			SmlDevInfCTCap *ctcap = d->data;
			g_free(ctcap->value);
			g_free(ctcap);
		}
		g_list_free(devinf->ctcaps);
		
		g_free(devinf);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

const char *smlDevInfGetManufacturer(SmlDevInf *devinf)
{
	return devinf->manufacturer;
}

void smlDevInfSetManufacturer(SmlDevInf *devinf, const char *man)
{
	if (devinf->manufacturer)
		g_free(devinf->manufacturer);
	devinf->manufacturer = g_strdup(man);
}

const char *smlDevInfGetModel(SmlDevInf *devinf)
{
	return devinf->model;
}

void smlDevInfSetModel(SmlDevInf *devinf, const char *model)
{
	if (devinf->model)
		g_free(devinf->model);
	devinf->model = g_strdup(model);
}

const char *smlDevInfGetOEM(SmlDevInf *devinf)
{
	return devinf->oem;
}

void smlDevInfSetOEM(SmlDevInf *devinf, const char *oem)
{
	if (devinf->oem)
		g_free(devinf->oem);
	devinf->oem = g_strdup(oem);
}

const char *smlDevInfGetFirmwareVersion(SmlDevInf *devinf)
{
	return devinf->firmwareVersion;
}

void smlDevInfSetFirmwareVersion(SmlDevInf *devinf, const char *firmwareVersion)
{
	if (devinf->firmwareVersion)
		g_free(devinf->firmwareVersion);
	devinf->firmwareVersion = g_strdup(firmwareVersion);
}

const char *smlDevInfGetSoftwareVersion(SmlDevInf *devinf)
{
	return devinf->softwareVersion;
}

void smlDevInfSetSoftwareVersion(SmlDevInf *devinf, const char *softwareVersion)
{
	if (devinf->softwareVersion)
		g_free(devinf->softwareVersion);
	devinf->softwareVersion = g_strdup(softwareVersion);
}

const char *smlDevInfGetHardwareVersion(SmlDevInf *devinf)
{
	return devinf->hardwareVersion;
}

void smlDevInfSetHardwareVersion(SmlDevInf *devinf, const char *hardwareVersion)
{
	if (devinf->hardwareVersion)
		g_free(devinf->hardwareVersion);
	devinf->hardwareVersion = g_strdup(hardwareVersion);
}

const char *smlDevInfGetDeviceID(SmlDevInf *devinf)
{
	return devinf->devid;
}

void smlDevInfSetDeviceID(SmlDevInf *devinf, const char *devid)
{
	if (devinf->devid)
		g_free(devinf->devid);
	devinf->devid = g_strdup(devid);
}

SmlDevInfDevTyp smlDevInfGetDeviceType(SmlDevInf *devinf)
{
	return devinf->devtyp;
}

void smlDevInfSetDeviceType(SmlDevInf *devinf, SmlDevInfDevTyp devtyp)
{
	devinf->devtyp = devtyp;
}

SmlBool smlDevInfSupportsUTC(SmlDevInf *devinf)
{
	return devinf->supportsUTC;
}

void smlDevInfSetSupportsUTC(SmlDevInf *devinf, SmlBool supports)
{
	devinf->supportsUTC = supports;
}

SmlBool smlDevInfSupportsLargeObjs(SmlDevInf *devinf)
{
	return devinf->supportsLargeObjs;
}

void smlDevInfSetSupportsLargeObjs(SmlDevInf *devinf, SmlBool supports)
{
	devinf->supportsLargeObjs = supports;
}

SmlBool smlDevInfSupportsNumberOfChanges(SmlDevInf *devinf)
{
	return devinf->supportsNumberOfChanges;
}

void smlDevInfSetSupportsNumberOfChanges(SmlDevInf *devinf, SmlBool supports)
{
	devinf->supportsNumberOfChanges = supports;
}

void smlDevInfAddDataStore(SmlDevInf *devinf, SmlDevInfDataStore *datastore)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, devinf, datastore);
	smlAssert(devinf);
	smlAssert(datastore);
	
	devinf->datastores = g_list_append(devinf->datastores, datastore);
	
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

unsigned int smlDevInfNumDataStores(SmlDevInf *devinf)
{
	smlAssert(devinf);
	
	return g_list_length(devinf->datastores);
}

SmlDevInfDataStore *smlDevInfGetNthDataStore(SmlDevInf *devinf, unsigned int nth)
{
	smlAssert(devinf);
	
	return g_list_nth_data(devinf->datastores, nth);
}

SmlDevInfDataStore *smlDevInfDataStoreNew(const char *sourceRef, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, sourceRef, error);
	
	SmlDevInfDataStore *datastore = smlTryMalloc0(sizeof(SmlDevInfDataStore), error);
	if (!datastore)
		goto error;

	datastore->sourceref = g_strdup(sourceRef);
	datastore->refCount = 1;
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, datastore);
	return datastore;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;	
}


SmlDevInfDataStore *smlDevInfDataStoreRef(SmlDevInfDataStore *datastore)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, datastore);
	smlAssert(datastore);
	
	g_atomic_int_inc(&(datastore->refCount));
	
	smlTrace(TRACE_EXIT, "%s: New refcount: %i", __func__, datastore->refCount);
	return datastore;
}

void smlDevInfDataStoreUnref(SmlDevInfDataStore *datastore)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, datastore);
	smlAssert(datastore);
	
	if (g_atomic_int_dec_and_test(&(datastore->refCount))) {
		smlTrace(TRACE_INTERNAL, "Refcount == 0!");
		
		g_free(datastore->sourceref);
		g_free(datastore->displayname);
		g_free(datastore->rxPrefContentType);
		g_free(datastore->rxPrefVersion);
		g_free(datastore->rxContentType);
		g_free(datastore->rxVersion);
		g_free(datastore->txPrefContentType);
		g_free(datastore->txPrefVersion);
		g_free(datastore->txContentType);
		g_free(datastore->txVersion);

		g_free(datastore);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

const char *smlDevInfDataStoreGetSourceRef(SmlDevInfDataStore *datastore)
{
	return datastore->sourceref;
}

void smlDevInfDataStoreSetSourceRef(SmlDevInfDataStore *datastore, const char *sourceref)
{
	if (datastore->sourceref)
		g_free(datastore->sourceref);
	datastore->sourceref = g_strdup(sourceref);
}

const char *smlDevInfDataStoreGetDisplayName(SmlDevInfDataStore *datastore)
{
	return datastore->displayname;
}

void smlDevInfDataStoreSetDisplayName(SmlDevInfDataStore *datastore, const char *displayName)
{
	if (datastore->displayname)
		g_free(datastore->displayname);
	datastore->displayname = g_strdup(displayName);
}

unsigned int smlDevInfGetMaxGUIDSize(SmlDevInfDataStore *datastore)
{
	return datastore->maxGUIDSize;
}

void smlDevInfSetMaxGUIDSize(SmlDevInfDataStore *datastore, unsigned int max)
{
	datastore->maxGUIDSize = max;
}

void smlDevInfDataStoreSetRxPref(SmlDevInfDataStore *datastore, const char *contenttype, const char *version)
{
	if (datastore->rxPrefContentType)
		g_free(datastore->rxPrefContentType);
	datastore->rxPrefContentType = g_strdup(contenttype);
	
	if (datastore->rxPrefVersion)
		g_free(datastore->rxPrefVersion);
	datastore->rxPrefVersion = g_strdup(version);
}

SmlBool smlDevInfDataStoreGetRxPref(SmlDevInfDataStore *datastore, char **contenttype, char **version)
{
	if (!datastore->rxPrefContentType)
		return FALSE;
	
	*contenttype = datastore->rxPrefContentType;
	*version = datastore->rxPrefVersion;
	
	return TRUE;
}

void smlDevInfDataStoreSetRx(SmlDevInfDataStore *datastore, const char *contenttype, const char *version)
{
	if (datastore->rxContentType)
		g_free(datastore->rxContentType);
	datastore->rxContentType = g_strdup(contenttype);
	
	if (datastore->rxVersion)
		g_free(datastore->rxVersion);
	datastore->rxVersion = g_strdup(version);
}

SmlBool smlDevInfDataStoreGetRx(SmlDevInfDataStore *datastore, char **contenttype, char **version)
{
	if (!datastore->rxContentType)
		return FALSE;
	
	*contenttype = datastore->rxContentType;
	*version = datastore->rxVersion;
	
	return TRUE;
}

void smlDevInfDataStoreSetTxPref(SmlDevInfDataStore *datastore, const char *contenttype, const char *version)
{
	if (datastore->txPrefContentType)
		g_free(datastore->txPrefContentType);
	datastore->txPrefContentType = g_strdup(contenttype);
	
	if (datastore->txPrefVersion)
		g_free(datastore->txPrefVersion);
	datastore->txPrefVersion = g_strdup(version);
}

SmlBool smlDevInfDataStoreGetTxPref(SmlDevInfDataStore *datastore, char **contenttype, char **version)
{
	if (!datastore->txPrefContentType)
		return FALSE;
	
	*contenttype = datastore->txPrefContentType;
	*version = datastore->txPrefVersion;
	
	return TRUE;
}

void smlDevInfDataStoreSetTx(SmlDevInfDataStore *datastore, const char *contenttype, const char *version)
{
	if (datastore->txContentType)
		g_free(datastore->txContentType);
	datastore->txContentType = g_strdup(contenttype);
	
	if (datastore->txVersion)
		g_free(datastore->txVersion);
	datastore->txVersion = g_strdup(version);
}

SmlBool smlDevInfDataStoreGetTx(SmlDevInfDataStore *datastore, char **contenttype, char **version)
{
	if (!datastore->txContentType)
		return FALSE;
	
	*contenttype = datastore->txContentType;
	*version = datastore->txVersion;
	
	return TRUE;
}

void smlDevInfDataStoreSetMemory(SmlDevInfDataStore *datastore, SmlBool shared, unsigned int maxid, unsigned int maxmem)
{
	datastore->sharedMem = shared;
	datastore->maxid = maxid;
	datastore->maxmem = maxmem;
}

void smlDevInfDataStoreGetMemory(SmlDevInfDataStore *datastore, SmlBool *shared, unsigned int *maxid, unsigned int *maxmem)
{
	if (shared)
		*shared = datastore->sharedMem;
		
	if (maxid)
		*maxid = datastore->maxid;
		
	if (maxmem)
		*maxmem = datastore->maxmem;
}

void smlDevInfDataStoreSetSyncCap(SmlDevInfDataStore *datastore, SmlDevInfSyncCap cap, SmlBool supported)
{
	if (supported)
		datastore->synccap = datastore->synccap | cap;
	else
		datastore->synccap = datastore->synccap & ~cap;
}

SmlBool smlDevInfDataStoreGetSyncCap(SmlDevInfDataStore *datastore, SmlDevInfSyncCap cap)
{
	return datastore->synccap & cap ? TRUE : FALSE;
}

void smlDevInfConfigureSession(SmlDevInf *devinf, SmlSession *session)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, devinf, session);
	smlAssert(devinf);
	smlAssert(session);
	
	if (!devinf->supportsNumberOfChanges)
		smlSessionUseNumberOfChanges(session, FALSE);
		
	if (!devinf->supportsLargeObjs)
		smlSessionUseLargeObjects(session, FALSE);
		
	smlTrace(TRACE_EXIT, "%s", __func__);
}

SmlBool smlDevInfAssemble(SmlDevInf *devinf, char **data, unsigned int *size, SmlError **error)
{
	return smlXmlDevInfAssemble(devinf, devinf->version, data, size, error);
}

SmlCommand *smlDevInfNewResult(SmlCommand *cmd, SmlDevInf *devinf, SmlDevInfVersion version, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p, %p)", __func__, cmd, devinf, version, error);
	smlAssert(cmd);
	SmlLocation *source = NULL;

	char *data = NULL;
	unsigned int size = 0;
	if (!smlXmlDevInfAssemble(devinf, version, &data, &size, error))
		goto error;
	
	if (version == SML_DEVINF_VERSION_10)
		source = smlLocationNew("./devinf10", NULL, error);
	else
		source = smlLocationNew("./devinf11", NULL, error);
		
	if (!source)
		goto error_free_data;
	
	SmlCommand *result = smlCommandNewResult(cmd, source, data, size, SML_ELEMENT_DEVINF_XML, error);
	if (!result) {
		smlLocationUnref(source);
		goto error_free_data;
	}
	/* Since the devinf is xml, we want to send it "raw" (without cdata) */
	result->private.results.status->item->raw = TRUE;
	
	smlLocationUnref(source);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, result);
	return result;
	
error_free_data:
	g_free(data);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlDevInfNewPut(SmlDevInf *devinf, SmlDevInfVersion version, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, devinf, version, error);
	smlAssert(devinf);
	SmlLocation *source = NULL;
	
	if (version == SML_DEVINF_VERSION_10)
		source = smlLocationNew("./devinf10", NULL, error);
	else
		source = smlLocationNew("./devinf11", NULL, error);
		
	if (!source)
		goto error;
		
	SmlCommand *cmd = smlCommandNewPut(NULL, source, NULL, 0, SML_ELEMENT_DEVINF_XML, error);
	if (!cmd)
		goto error_free_source;
	
	smlLocationUnref(source);
	
	char *data = NULL;
	unsigned int size = 0;
	if (!smlXmlDevInfAssemble(devinf, version, &data, &size, error))
		goto error_free_cmd;
	
	if (!smlItemAddData(cmd->private.access.item, data, size, error)) {
		g_free(data);
		goto error_free_cmd;
	}
	smlItemSetRaw(cmd->private.access.item, TRUE);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;
	
error_free_cmd:
	smlCommandUnref(cmd);
error_free_source:
	smlLocationUnref(source);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlCommand *smlDevInfNewGet(SmlDevInfVersion version, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%i, %p)", __func__, version, error);
	SmlLocation *target = NULL;
	
	if (version == SML_DEVINF_VERSION_10)
		target = smlLocationNew("./devinf10", NULL, error);
	else
		target = smlLocationNew("./devinf11", NULL, error);
		
	if (!target)
		goto error;
		
	SmlCommand *cmd = smlCommandNewGet(target, SML_ELEMENT_DEVINF_XML, error);
	if (!cmd)
		goto error_free_target;
	
	smlLocationUnref(target);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, cmd);
	return cmd;
	
error_free_target:
	smlLocationUnref(target);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlDevInf *smlDevInfParse(const char *data, unsigned int length, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %p)", __func__, data, length, error);
	smlAssert(data);
	smlAssert(length);
	
	SmlDevInf *devinf = smlXmlDevInfParse(data, length, error);
	if (!devinf)
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return devinf;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

SmlDevInf *smlDevInfFromResult(SmlCommand *result, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, result, error);
	smlAssert(result);
	
	if (result->type != SML_COMMAND_TYPE_RESULTS) {
		smlErrorSet(error, SML_ERROR_GENERIC, "devinf command was not a result");
		goto error;
	}
	
	SmlItem *item = result->private.results.status->item;
	if (!item) {
		smlErrorSet(error, SML_ERROR_GENERIC, "devinf result did not have a item");
		goto error;
	}
	
	char *data = NULL;
	unsigned int size = 0;
	if (!smlItemGetData(item, &data, &size, error))
		goto error;
	
	SmlDevInf *devinf = smlDevInfParse(data, size, error);
	if (!devinf)
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return devinf;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

void smlDevInfAddCTCap(SmlDevInf *devinf, SmlDevInfCTCapType type, const char *value)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %i, %s)", __func__, devinf, type, value);
	smlAssert(devinf);
	smlAssert(value);

	SmlDevInfCTCap *ctcap = smlTryMalloc0(sizeof(SmlDevInfCTCap), NULL);
	if (!ctcap) {
		return;
	}

	ctcap->type = type;
	ctcap->value = g_strdup (value);

	devinf->ctcaps = g_list_append (devinf->ctcaps, ctcap);

	smlTrace(TRACE_EXIT, "%s", __func__);
}

unsigned int smlDevInfNumCTCaps(SmlDevInf *devinf)
{
	smlAssert(devinf);

	return g_list_length(devinf->ctcaps);
}

SmlDevInfCTCapType smlDevInfGetNthCTCapType(SmlDevInf *devinf, unsigned int nth)
{
	smlAssert(devinf);

	SmlDevInfCTCap *ctcap = g_list_nth_data(devinf->ctcaps, nth);
	return ctcap->type;
}

const char *smlDevInfGetNthCTCapValue(SmlDevInf *devinf, unsigned int nth)
{
	smlAssert(devinf);

	SmlDevInfCTCap *ctcap = g_list_nth_data(devinf->ctcaps, nth);
	return ctcap->value;
}

SmlDevInfCTCapType smlDevInfCTCapTypeFromString(const char *name, SmlError **error)
{
	if (!strcmp(name, SML_ELEMENT_CTTYPE)) {
		return SML_DEVINF_CTCAP_CTTYPE;
	} else if (!strcmp(name, SML_ELEMENT_PROPNAME)) {
		return SML_DEVINF_CTCAP_PROPNAME;
	} else if (!strcmp(name, SML_ELEMENT_VALENUM)) {
		return SML_DEVINF_CTCAP_VALENUM;
	} else if (!strcmp(name, SML_ELEMENT_DATATYPE)) {
		return SML_DEVINF_CTCAP_DATATYPE;
	} else if (!strcmp(name, SML_ELEMENT_SIZE)) {
		return SML_DEVINF_CTCAP_SIZE;
	} else if (!strcmp(name, SML_ELEMENT_DISPLAYNAME)) {
		return SML_DEVINF_CTCAP_DISPLAYNAME;
	} else if (!strcmp(name, SML_ELEMENT_PARAMNAME)) {
		return SML_DEVINF_CTCAP_PARAMNAME;
	}
	
	smlErrorSet(error, SML_ERROR_GENERIC, "Unknown ctcap type name \"%s\"", name);
	return SML_DEVINF_CTCAP_UNKNOWN;
}


const char *smlDevInfCTCapTypeToString(SmlDevInfCTCapType type, SmlError **error)
{
	switch (type) {
		case SML_DEVINF_CTCAP_CTTYPE:
			return SML_ELEMENT_CTTYPE;
		case SML_DEVINF_CTCAP_PROPNAME:
			return SML_ELEMENT_PROPNAME;
		case SML_DEVINF_CTCAP_VALENUM:
			return SML_ELEMENT_VALENUM;
		case SML_DEVINF_CTCAP_DATATYPE:
			return SML_ELEMENT_DATATYPE;
		case SML_DEVINF_CTCAP_SIZE:
			return SML_ELEMENT_SIZE;
		case SML_DEVINF_CTCAP_DISPLAYNAME:
			return SML_ELEMENT_DISPLAYNAME;
		case SML_DEVINF_CTCAP_PARAMNAME:
			return SML_ELEMENT_PARAMNAME;
		default:
			;
	}
	
	smlErrorSet(error, SML_ERROR_GENERIC, "Unknown ctcap type \"%i\"", type);
	return NULL;
}

/*@}*/
