/***************************************************************************
 *  ContactList.cs
 *
 *  Copyright (C) 2006 INdT
 *  Written by
 *      Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *      Kenneth Christiansen <kenneth.christiansen@gmail.com>
 *      Renato Araujo Oliveira Filho <renato.filho@indt.org>
 ****************************************************************************/

/*  THIS FILE IS LICENSED UNDER THE MIT LICENSE AS OUTLINED IMMEDIATELY BELOW:
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the Software),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED AS IS, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

using System;
using System.Collections;
using System.Collections.Generic;
using NDesk.DBus;
using ObjectPath = NDesk.DBus.ObjectPath;
using org.freedesktop.DBus;
using org.freedesktop.Telepathy;

namespace Tapioca
{
	public delegate void ContactListAuthorizationRequestedHandler (ContactList sender, Contact contact);
	public delegate void ContactListContactRetrievedHandler (ContactList sender, Contact contact);
	
	public class ContactList : DBusProxyObject
	{
		public event ContactListAuthorizationRequestedHandler AuthorizationRequested;
		public event ContactListContactRetrievedHandler ContactRetrieved;
		
		Tapioca.Connection connection;
		System.Collections.Hashtable contacts;
		
		private enum CLNames 
		{
			subscribe = 0,
			publish = 1,
			known = 2,
			hide = 3,
			deny = 4,
			allow = 5,
			last = 6
		};
		
		
		bool avatar_connected, presence_connected, alias_connected;
		PrivContactList[] cl;

//public methods:
		public Contact[] Contacts
		{
			get {
				Contact[] ret = new Contact[contacts.Values.Count];
				int i = 0;								
				foreach (Contact contact in contacts.Values)
				{
						ret[i] = contact;
						i++;
				}
				return ret;
			}
		}

		public bool AddContact (string contact_uri)
		{
			Handle contact_h = new Handle (connection.TlpConnection, HandleType.Contact, contact_uri);
			contact_h.Request ();
			contact_h.Hold ();
			
			if (contacts.ContainsKey (contact_h.Id)) {
				contact_h.Dispose ();
				return true;
			}			
			Contact contact = new Contact (connection, contact_h, SubscriptionStatus.RemotePending, ContactPresence.Offline, "");
			AddContact (contact);
			cl[(int) CLNames.subscribe].AddMember (contact_h);
			return true;
		}

		public void RemoveContact (Contact contact)
		{
			if (!contacts.ContainsKey (contact.Handle.Id))
				return;

			for (int i= (int) CLNames.subscribe; i < (int) CLNames.last; i++)
			{
				if ((cl[i] != null) && (cl[i].Contains (contact.Handle)))
					cl[i].RemoveMember (contact.Handle);
			}
			
			//contacts.Remove (contact.Handle.Id);
		}
		
		public void AcceptContact (Contact contact)
		{
			if (contacts.ContainsKey (contact.Handle.Id)) {
				if (contact.SubscriptionStatus == SubscriptionStatus.LocalPending)
					cl[(int) CLNames.subscribe].AddMember (contact.Handle);
			}
		}
		
		public void RejectContact (Contact contact)
		{
			if (contacts.ContainsKey (contact.Handle.Id)) {
				if (contact.SubscriptionStatus == SubscriptionStatus.LocalPending)
					cl[(int) CLNames.publish].RemoveMember (contact.Handle);
			}
		}

		public void BlockContact (Contact contact)
		{
			if (cl[(int) CLNames.deny] == null) {
				Console.WriteLine ("CM: Not support block");
				return;
			}
			
			if ((cl[(int) CLNames.deny] != null) && (contacts.ContainsKey (contact.Handle.Id))) 
				cl[(int) CLNames.deny].AddMember (contact.Handle);
			return;
		}
		
		public void UnblockContact (Contact contact)
		{
			if ((cl[(int) CLNames.deny] != null) && (contacts.ContainsKey (contact.Handle.Id))) 
				cl[(int) CLNames.deny].RemoveMember (contact.Handle);
			return;
		}
		
		public Contact GetContact (string uri)
		{
			foreach (Contact c in contacts.Values) {
				if (c.Uri == uri)
					return c;
			}
			return null;
		}

//internal methods:
		
		internal ContactList (Tapioca.Connection connection)
			: base (connection.ServiceName, new ObjectPath (null))
		{
			contacts = new System.Collections.Hashtable ();
			this.connection = connection;
		}

		internal Contact ContactLookup (uint handle)
		{
			if (contacts.ContainsKey (handle))
				return  (Contact) contacts[(uint) handle];
				
			return null;
		}
		
		internal void Dispose ()
		{
			UnloadContacts ();
		}
		
		internal void Clear ()
		{
			UnloadContacts ();
		}
		
		internal void LoadContacts ()
		{	
			bool blocked;
			cl = new PrivContactList[(int)CLNames.last];
			ConnectConnectionSignals ();
			// Loading contacts
			for(int i= (int) CLNames.subscribe; i < (int)CLNames.last; i++)
			{
				try {					
					cl[i] = new PrivContactList (connection.TlpConnection, System.Enum.GetName (typeof (CLNames), i), ServiceName);					
					if (i == (int)CLNames.deny)
						blocked = true;
					else
						blocked = false;
						
					foreach (Handle handle in cl[i].Members)
					{	
						if (!contacts.Contains (handle.Id)) {
							Contact contact = new Contact (connection, handle, SubscriptionStatus.Subscribed, ContactPresence.Offline, "");
							contact.Blocked = blocked;
							AddContact (contact);
						}
					}
					
					foreach (Handle handle in cl[i].LocalPending)
					{	
						if (!contacts.Contains (handle.Id)) {
							Contact contact = new Contact (connection, handle, SubscriptionStatus.LocalPending, ContactPresence.Offline, "");
							contact.Blocked = blocked;
							AddContact (contact);
						}
					}
					
					foreach (Handle handle in cl[i].RemotePending)
					{	
						if (!contacts.Contains (handle.Id)) {
							Contact contact = new Contact (connection, handle, SubscriptionStatus.RemotePending, ContactPresence.Offline, "");
							contact.Blocked = blocked;
							AddContact (contact);
						}
					}
					
					cl[i].MemberAdded += OnMemberAdded;
					cl[i].MemberRemoved += OnMemberRemoved;
					cl[i].MemberLocalPending += OnMemberLocalPending;
					cl[i].MemberRemotePending += OnMemberRemotePending;
					
				}
				catch {
					//Console.WriteLine ("Invalid list");
					cl[i] = null;
					continue;
				}
			}
		}		
		

//private methods:
		private void AddContact (Contact contact)
		{
			if (!contacts.ContainsKey (contact.Handle.Id)) {
				contacts.Add (contact.Handle.Id, contact);
				if ((contact.SubscriptionStatus == SubscriptionStatus.LocalPending) && (AuthorizationRequested != null)) 
					AuthorizationRequested (this, contact);
				else if (ContactRetrieved != null) {
					ContactRetrieved (this, contact);
					RequestPresence (contact.Handle.Id);
				}
			}
		}
		
		private void UnloadContacts ()
		{	
			DiscconectConnectionSignals ();
			
			foreach (Contact contact in contacts.Values) {
				contact.Release ();
			}
			contacts.Clear ();

			if (cl != null) {
				for(int i= (int) CLNames.subscribe; i < (int)CLNames.last; i++)
				{
					if (cl[i] != null)
						cl[i].Dispose ();
				}
				cl = null;
			}
		}						

		private void RequestPresence (uint id)
		{
			if (!connection.SupportPresence)
				return;
				
			uint[] ids = {id};
			connection.TlpConnection.RequestPresence (ids);
		}

		private void OnPresenceUpdate (IDictionary<uint, PresenceUpdateInfo> statuses)
		{
			foreach (KeyValuePair<uint,PresenceUpdateInfo>  entry in statuses)
			{
				Contact contact = ContactLookup (entry.Key);

				if (contact == null)
					return;

				foreach (KeyValuePair<string, IDictionary<string,object>> info in entry.Value.info)
				{
					string message = "";
					foreach (KeyValuePair<string,object> val in info.Value) {						
						if (val.Key == "message")
							message = (string) val.Value;
					}
					contact.UpdatePresence (info.Key, message);
				}
			}
		}
		
		private Contact GetContact (Handle handle)
		{
			if (contacts.ContainsKey (handle.Id))
				return (Contact) contacts[handle.Id];
			return null;
		}
		
		private void OnMemberAdded (PrivContactList cl, Handle handle)
		{					
			Contact contact = GetContact (handle);			
			if (contact == null)
				contact = new Contact (connection, handle, SubscriptionStatus.None, ContactPresence.Offline, "");
						
			switch (cl.Name) {
				case "deny":
					contact.Blocked = true;
					break;
				case "subscribe":
					
					if (contact.SubscriptionStatus == SubscriptionStatus.LocalPending) {
						contact.UpdateStatus (SubscriptionStatus.Subscribed);
						ContactRetrieved (this, contact);
						RequestPresence (contact.Handle.Id);
					} else {
						contact.UpdateStatus (SubscriptionStatus.Subscribed);	
					}					
					break;
				default:
					break;
			}
			
			AddContact (contact);
		}
		
		private void OnMemberRemoved (PrivContactList cl, Handle handle)
		{
			Contact contact = GetContact (handle);
			if (contact != null) {
				switch (cl.Name) {
					case "deny":
						contact.Blocked = false;
						break;
					case "subscribe":
						contacts.Remove (contact.Handle.Id);
						contact.Release ();
						contact.UpdatePresence ("offline", "");
						contact.UpdateStatus (SubscriptionStatus.None);
						break;
					default:
						break;
				}
			}
		}

		private void OnMemberLocalPending (PrivContactList cl, Handle handle)
		{			
			if (cl.Name != "publish") return;
			
			Contact contact = GetContact (handle);
			if (contact != null) {
				contact.UpdateStatus (SubscriptionStatus.LocalPending);
			} 
			else {
				contact = new Contact (connection, handle, SubscriptionStatus.LocalPending, ContactPresence.Offline, "");
				AddContact (contact);
			}
		}

		private void OnMemberRemotePending (PrivContactList cl, Handle handle)
		{			
			if (cl.Name != "publish") return;
			
			Contact contact = GetContact (handle);
			if (contact != null)  
				contact.UpdateStatus (SubscriptionStatus.RemotePending);
			else {
				contact = new Contact (connection, handle, SubscriptionStatus.RemotePending, ContactPresence.Offline, "");
				AddContact (contact);
			}
		}
		
		private void OnAliasesChanged (AliasInfo[] aliases)
		{
			foreach (AliasInfo info in aliases) {
				if (contacts.ContainsKey (info.ContactHandle)) {
					Contact contact = (Contact) contacts[info.ContactHandle];
					contact.UpdateAlias (info.NewAlias);
				}
			}
		}		
		
		private void ConnectConnectionSignals ()
		{
			if (this.connection.SupportAliasing) {
				this.connection.TlpConnection.AliasesChanged += OnAliasesChanged;
				alias_connected = true;
			}
				
			if (this.connection.SupportPresence) {
				this.connection.TlpConnection.PresenceUpdate += OnPresenceUpdate;
				presence_connected = true;
			}
			
			if (this.connection.SupportAvatars) {
				this.connection.TlpConnection.AvatarUpdated += OnAvatarUpdated;
				avatar_connected = true;
			}
		}
		
		private void DiscconectConnectionSignals ()
		{ 
			if (avatar_connected) {
				this.connection.TlpConnection.AvatarUpdated -= OnAvatarUpdated;
				avatar_connected = false;
			}
			
			if (presence_connected) {
				this.connection.TlpConnection.PresenceUpdate -= OnPresenceUpdate;
				presence_connected = false;
			}
			
			if (alias_connected) {
				this.connection.TlpConnection.AliasesChanged -= OnAliasesChanged;
				alias_connected = false;
			}	
		}
		
		private void OnAvatarUpdated (uint contact_id, string token)
		{
			Contact c = ContactLookup (contact_id);
			if (c != null)
				c.UpdateAvatarToken (token);
		}
	}
}
